(*****************************************************)
(* Toolbar object with "ToolTips".                   *)
(* Descends from TToolbar Object, which is supplied  *)
(* with Borland Pascal 7.0                           *)
(* Written by Mike Bower (70142,1723)                *)
(*                    ----                           *)
(* Copyright (c) 1994 Michael J. Bower               *)
(*               All Rights Reserved                 *)
(*                    ----                           *)
(* Build Date: 27-June 1994 version 1.0a             *)
(* Requires Borland Pascal 7.0 and Windows 3.1       *)
(* For an example of using this unit, see            *)
(* MFILEAPP.PAS, which is supplied in the ZIP file   *)
(*****************************************************)

unit ToolTBar;

interface
uses
  Objects,
  OWindows,
  WinProcs,
  WinTypes,
  Strings,
  Win31,
  Toolbar;

(* Constants *)
const IDTIMER = 1;                {Timer ID}
const DELAY = 1000;               {Timer Delay, in millisec}
const MAX_TOOLTIP_LENGTH = 40;    {Max chars in ToolTip ASCIIZ string}
const X_WHITESPACE = 2;           {Whitespace to Left/Right of ToolTip, pixels}
const Y_WHITESPACE = 2;           {WhiteSpace above/below Tooltip, pixels}
const X_OFFSET = 5;               {Pixels from Mouse Hotspot X pos}
const Y_OFFSET = 35;              {Pixels from Mouse HotSpot Y pos}


(*****************************************)
(*         TToolTip class                *)
(*****************************************)
type
  PToolTip = ^TToolTip;
  TToolTip = object(TWindow)
    BackGroundBrush: HBrush;
    ToolTipString : array[0..MAX_TOOLTIP_LENGTH -1] of char;
    constructor Init(aParent: PwindowsObject; aTipID: word);
    destructor Done; virtual;
    procedure SetupWindow; virtual;
    procedure GetWindowClass(var WndClass:TWndClass); virtual;
    procedure Show; virtual;
    procedure Hide; virtual;
    procedure Paint(PaintDC:HDC; var PaintInfo:TPaintStruct);virtual;
    procedure MakeToolTipFont(var AFont: HFont);
    procedure WMMouseMove(var msg:Tmessage); virtual WM_FIRST + WM_MOUSEMOVE;
    function GetClassName: pchar; virtual;
  end;

(*****************************************)
(*      TToolTipToolBar class            *)
(*****************************************)
type
  PToolTipToolBar = ^TToolTipToolBar;
  TToolTipToolBar = object(TToolBar)
    JustMadeToolTip: Boolean;
    ToolPointedTo  : PTool;
    MousePosition  : TPoint;
    constructor Init(AParent: PWindowsObject; AName: PChar; Orient: Word);
    function CreateTool(Num: Integer; Origin: TPoint; Command: Word;
                        BitmapName: PChar): PTool; virtual;
    procedure ReadResource; virtual;
    procedure WMLButtonUp(var Msg: TMessage); virtual WM_FIRST + WM_LBUTTONUP;
    procedure WMMouseMove(var msg:Tmessage);  virtual WM_FIRST + WM_MOUSEMOVE;
    procedure WMTimer(var msg:Tmessage);      virtual WM_FIRST + WM_TIMER;
  end;

(*****************************************)
(*       TToolTipButton class            *)
(*****************************************)
type
  PToolTipButton = ^TToolTipButton;
  TToolTipButton = object(TToolButton)
    ToolTip : PToolTip;
    constructor Init( AParent: PWindowsObject; X, Y: Integer;
                      ACommand: Word; BitmapName: PChar);
    procedure ShowToolTip; virtual;
    procedure HideToolTip; virtual;
  end;

const
  RToolTipToolbar: TStreamRec = (
    ObjType: 12302;
    VmtLink: Ofs(TypeOf(TToolTipToolbar)^);
    Load:    @TToolTipToolbar.Load;
    Store:   @TToolTipToolbar.Store
  );

const
  RToolTip: TStreamRec = (
    ObjType: 12303;
    VmtLink: Ofs(TypeOf(TToolTip)^);
    Load:    @TToolTip.Load;
    Store:   @TToolTip.Store
  );

procedure RegisterToolTipToolBar;

(* End of INTERFACE *)

implementation


procedure RegisterToolTipToolBar;
begin
  RegisterType(RToolTipToolBar);
  RegisterType(RToolTip);
end;


(********************)
(* TToolTip methods *)
(********************)
constructor TToolTip.Init(aParent: PwindowsObject; aTipID: word);
begin
  inherited Init(aParent, 'ToolTip');
  attr.exstyle := attr.exstyle or WS_EX_TOPMOST;
  attr.style := WS_POPUP or WS_BORDER;
  LoadString(hInstance, aTipId, ToolTipString, SizeOf(ToolTipString));
end;


destructor TToolTip.Done;
begin
  inherited Done;
end;


procedure TToolTip.SetupWindow;
begin
  inherited SetupWindow;
end;


procedure TToolTip.GetWindowClass(var WndClass:TWndClass);
begin
  inherited GetWindowClass(WndClass);
  {...Make a light yellow background}
  WndClass.HBrBackground := CreateSolidBrush(RGB(255,255,192));
end;


function TToolTip.GetClassName: pchar;
begin
  GetClassName := 'ToolTip';
end;


procedure TToolTip.Show;
var
  DC      : HDC;
  TextSize: longint;
  CurPos  : Tpoint;
  TheFont : HFont;
begin
  {...Get the size of the tooltip, and move the window into position}
  DC := GetDC(hWindow);
  MakeToolTipFont(TheFont);
  SelectObject(DC, TheFont);
  TextSize := GetTextExtent(DC, ToolTipString, StrLen(ToolTipString));
  DeleteObject(TheFont);
  ReleaseDC(hWindow,DC);
  attr.w := LoWord(TextSize) + (X_WHITESPACE * 2);
  attr.h := HiWord(TextSize) + (Y_WHITESPACE * 2);
  GetCursorPos(CurPos);
  attr.x := CurPos.x + X_OFFSET;
  attr.y := CurPos.y + Y_OFFSET - attr.h;
  MoveWindow(hWindow, attr.x , attr.y, attr.w, attr.h,false);
  {...Display the Window - but don't activate}
  ShowWindow(hWindow, SW_SHOWNA);
end;


procedure TToolTip.Hide;
begin
  ShowWindow(hWindow, SW_HIDE);
end;


procedure TToolTip.Paint(PaintDC:HDC; var PaintInfo:TPaintStruct);
var
  TheFont: HFont;
begin
  MakeToolTipFont(TheFont);
  SelectObject(PaintDC, TheFont);
  SetBkMode(PaintDC, TRANSPARENT);
  TextOut(PaintDC, X_WHITESPACE - 1, Y_WHITESPACE - 1 ,ToolTipString, StrLen(ToolTipString));
  DeleteObject(TheFont);
end;


procedure TToolTip.MakeToolTipFont(var AFont: HFont);
var
  ALogFont: TLogFont;
begin
  FillChar(ALogFont, SizeOf(ALogFont), #0);
  with ALogFont do
  begin
    lfHeight        := 4;         {...Make font                         }
    lfWeight        := FW_NORMAL; {...Indicate a normal attribute       }
    lfItalic        := 0;         {...Non-zero value indicates italic   }
    lfUnderline     := 0;         {...Non-zero value indicates underline}
    lfOutPrecision  := Out_Stroke_Precis;
    lfClipPrecision := Clip_Stroke_Precis;
    lfQuality       := Default_Quality;
    lfPitchAndFamily:= Variable_Pitch;
    StrCopy(lfFaceName, 'MS SANS SERIF');
  end;
  AFont := CreateFontIndirect(ALogFont);
end;

procedure TToolTip.WMMouseMove(var msg:TMessage);
begin
  Hide;
end;

(***************************)
(* TToolTipToolBar methods *)
(***************************)
constructor TToolTipToolBar.Init(AParent: PWindowsObject; AName: PChar; Orient: Word);
begin
  inherited Init(AParent, AName, Orient);
  ToolPointedTo := nil;
end;

{ A mouse movement was detected, so clear the Tool tip }
procedure TToolTipToolBar.WMMousemove(var msg:Tmessage);
var
  LastToolPointedTo : PTool;
  function IsHit(Item: PTool): Boolean; far;
  begin
    IsHit := Item^.HitTest(TPoint(Msg.LParam));
  end;
begin
  inherited WMMouseMove(msg);
  KillTimer(HWindow, IDTIMER);
  {...Mouse moved, so hide the tooltip}
  if (ToolPointedTo <> nil) and (JustMadeToolTip = false) then begin
    PToolTipButton(ToolPointedTo)^.HideToolTip;
  end;
  LastToolPointedTo := ToolPointedTo;
  ToolPointedTo := Tools.FirstThat(@IsHit);
  {...If the mouse skipped across to another tool tip, we must remove te old one}
  if (ToolPointedTo <> LastToolPointedTo) and (LastToolPointedTo <> nil) then begin
    PToolTipButton(LastToolPointedTo)^.HideToolTip;
  end;
  {...Now, start a new timer sequence}
  if ToolPointedTo <> nil then begin
    GetCursorPos(MousePosition);
    SetTimer(hWindow, IDTIMER, DELAY, nil);
  end;
  JustMadeToolTip := false;
end;

{This method is called when the mouse has been steady for the }
{timer count.  This method checks that the mouse position has }
{ not changed and if not, the Tool tip is shown               }
procedure TToolTipToolBar.WMTimer(var msg:Tmessage);
var
  CursorPos: TPoint;
begin
  if MSG.Wparam = IDTIMER then begin
    KillTimer(HWindow, IDTIMER);
    GetCursorPos(CursorPos);
    if (MousePosition.x = CursorPos.x) and
       (MousePosition.y = cursorPos.y) then begin
      PToolTipButton(ToolPointedTo)^.ShowToolTip;
      JustMadeToolTip := true;
    end;
  end;
end;

{If the mouse button is released, hide the tooltip}
procedure TToolTipToolBar.WMLButtonUp(var Msg: TMessage);
begin
  if ToolPointedTo <> nil then begin
    PToolTipButton(ToolPointedTo)^.HideToolTip;
    JustMadeToolTip := false;
    MousePosition.x := -1; {...this assures the tooltip won't repaint}
    MousePosition.y := -1;
  end;
  inherited WMLButtonUp(Msg);
end;


{ Read the resource file }
procedure TToolTipToolBar.ReadResource;
type
  ResRec = record
    Bitmap          : word;
    Command         : word;
    StringResourceID: word;
  end;
  PResArray = ^TResArray;
  TResArray = array [1..$FFF0 div sizeof(ResRec)] of ResRec;

var
  ResIdHandle  : THandle;
  ResDataHandle: THandle;
  ResDataPtr   : PResArray;
  Count        : Word;
  X            : Word;
  Origin       : TPoint;
  BitInfo      : TBitmap;
  P            : PTool;

begin
  ResIDHandle   := FindResource(HInstance, ResName, 'ToolBarData');
  ResDataHandle := LoadResource(HInstance, ResIDHandle);
  ResDataPtr    := LockResource(ResDataHandle);
  if (ResIDHandle = 0) or (ResDataHandle = 0) or (ResDataPtr = nil) then begin
    Status := em_InvalidChild;
    Exit;
  end;

  X := 0;
  Origin.X := 2;
  Origin.Y := 2;

  Count := PWord(ResDataPtr)^;
  Inc(LongInt(ResDataPtr), SizeOf(Count)); { Skip Count }
  for X := 1 to Count do begin
    with ResDataPtr^[X] do begin
      P := CreateTool(X, Origin, Command, PChar(Bitmap));
      if P <> nil then begin
        if TypeOf(p^) = TypeOf(TToolTipButton) then begin
          PToolTipButton(P)^.ToolTip := new(PtoolTip, init(@self, StringResourceID));
        end;
        NextToolOrigin(X, Origin, P);
        Tools.Insert(P);
      end; {if}
    end; {with}
  end; {for}
  UnlockResource(ResDataHandle);
  FreeResource(ResDataHandle);
end;


function TToolTipToolBar.CreateTool(Num: Integer; Origin: TPoint;
                                    Command: Word; BitmapName: PChar): PTool;
begin
  if Word(BitmapName) = 0 then begin
    CreateTool := New(PToolSpacer, Init(@Self, Command))
  end
  else begin
    CreateTool := New(PToolTipButton,
                      Init( @Self, Origin.X, Origin.Y, Command,BitmapName));
  end;
end;


(**************************)
(* TToolTipButton methods *)
(**************************)
constructor TToolTipButton.Init( AParent: PWindowsObject; X, Y: Integer;
                                 ACommand: Word; BitmapName: PChar);
begin
  inherited init(aParent, X, Y, ACommand, BitmapName);
end;


procedure TToolTipButton.ShowToolTip;
begin
  ToolTip^.Show;
end;


procedure TToolTipButton.HideToolTip;
begin
  ToolTip^.Hide;
end;


end. {of unit}
