// Copyright 1993 by Peter Bentley <pete@tecc.co.uk>
//
// Permission to use, copy, modify, distribute, and sell this software and its
// documentation for any purpose is hereby granted without fee, provided that
// the above copyright notice appear in all copies and that both that
// copyright notice and this permission notice appear in supporting
// documentation, and that the name of Peter Bentley not be used in
// advertising or publicity pertaining to distribution of the software without
// specific, written prior permission.  Peter Bentley makes no representations
// about the suitability of this software for any purpose.  It is provided
// "as is" without express or implied warranty.
//
// PETER BENTLEY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
// INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
// EVENT SHALL PETER BENTLEY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
// CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
// DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
// TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
// PERFORMANCE OF THIS SOFTWARE.
//
// The base socket classes are here.  These were written by
// Pete and he'll probably tell you to piss off if you
// ask him about it.  I would.

// Anyway, the big thing to be added here is a mechanism to
// allow asynchronous AddressResolution etc.

// These classes are probably the most complicated in the whole
// of windis.  Trace them.  There's lots and lots of complicated virtual
// functions getting called and they're really really object oriented.
// Think **hard** before you change them...

#include "stdafx.h"
#include <ctype.h>
#include <errno.h>
#include "windis.h"
#include "tcsock.h"


/////////////////////////////////////////////////////////////////////////////
// Global C stuff used by the C part of the library (fielding messages,'
// creating windows etc
//
extern "C" {
	long FAR PASCAL _export tcSockWindProc(HWND, UINT, UINT, LONG);
}
extern char tcSockClassName[];
extern HINSTANCE tcSockInst;
extern HWND tcSockHwndParent;


/////////////////////////////////////////////////////////////////////////////
// tcWinsockStarter.  Not for public consumption.  We have a single global
// instance of this class.  The constructor does nothing, the start()
// function calls WSAStartup() if necessary and then the destructor (called
// once at program exit) calls WSACleanup().  This means that if
// we keep opening/closing a single socket we don't keep calling WSAStartup()
// & WSACleanup() which seems to kill Novell's LWP
//
class tcWinsockStarter {
  private:
	int		references;
	WSADATA	wsData;
  public:
	tcWinsockStarter() {
		references = 0;
	}
	~tcWinsockStarter() {
		stop();
	}
	int		start(tcLogger *l =NULL) {
		// Start Winsock.  TODO: handle version numbers properly.
		if( references++ != 0 ) // Need to start winsock since first call
			return 0;
#if TCSOCK_TRACE			
		if (l != NULL) l->logf(TC_INFO,
							"Initialising WinSock library: WSAStartup()");
#endif							
		int rc = WSAStartup(0x0101, &wsData);
		if (rc != 0) {
			if (l != NULL) l->logf(TC_ERROR, "Socket startup failure: %d",
						 			WSAGetLastError());
			return -1;
		}
		return 0;
	}
	int		stop() {
		if( --references == 0)
			WSACleanup();					// Close down Winsock
		return 0;
	}
	// Information functions
	int			isStarted()		{ return references;				};
    const char	*description()	{ return wsData.szDescription;	};
};

static tcWinsockStarter	theGlobalStarter;

/////////////////////////////////////////////////////////////////////////////
// tcSocket class variables and member functions
//
tcSocket::errDesc tcSocket::errString[] = {
/*
 * Windows Sockets definitions of regular Microsoft C error constants
 */
	WSAEINTR,			"WSAEINTR: Interrupted 'system call'",
	WSAEBADF,			"WSAEBADF: Bad file descriptor",
	WSAEACCES,			"WSAEACCES: Permission denied",
	WSAEFAULT,			"WSAEFAULT: Bad address",
	WSAEINVAL,			"WSAEINVAL: Invalid argument",
	WSAEMFILE,			"WSAEMFILE: Too many open files",
/*
 * Windows Sockets definitions of regular Berkeley error constants
 */
	WSAEWOULDBLOCK,		"WSAEWOULDBLOCK: Would block",
	WSAEINPROGRESS,		"WSAEINPROGRESS: Blocking call already in progress",
	WSAEALREADY,		"WSAEALREADY: Operation already in progress",
	WSAENOTSOCK,		"WSAENOTSOCK: Not a socket",
	WSAEDESTADDRREQ,	"WSAEDESTADDRREQ: Destination address required",
	WSAEMSGSIZE,		"WSAEMSGSIZE: Message too long",
	WSAEPROTOTYPE,		"WSAEPROTOTYPE: Wrong protocol for socket",
	WSAENOPROTOOPT,		"WSAENOPROTOOPT: Protocol option not available",
	WSAEPROTONOSUPPORT,	"WSAEPROTONOSUPPORT: Protocol not suported",
	WSAESOCKTNOSUPPORT,	"WSAESOCKTNOSUPPORT: Socket type not supported",
	WSAEOPNOTSUPP,		"WSAEOPNOTSUPP: Operation not supported on socket",
	WSAEPFNOSUPPORT,	"WSAEPFNOSUPPORT: Protocol family not supported",
	WSAEAFNOSUPPORT,	"WSAEAFNOSUPPORT: Address familt not supported",
	WSAEADDRINUSE,		"WSAEADDRINUSE: Address already in use",
	WSAEADDRNOTAVAIL,	"WSAEADDRNOTAVAIL: Can't assign requested address",
	WSAENETDOWN,		"WSAENETDOWN: Network is down",
	WSAENETUNREACH,		"WSAENETUNREACH: Network is unreachable",
	WSAENETRESET,		"WSAENETRESET: Network dropped connection on reset",
	WSAECONNABORTED,	"WSAECONNABORTED: Software caused connection abort",
	WSAECONNRESET,		"WSAECONNRESET: Connection reset by peer",
	WSAENOBUFS,			"WSAENOBUFS: No buffer space available",
	WSAEISCONN,			"WSAEISCONN: Socket is already connected",
	WSAENOTCONN,		"WSAENOTCONN: Socket is not connected",
	WSAESHUTDOWN,		"WSAESHUTDOWN: Can't send after socket shutdown",
	WSAETOOMANYREFS,	"WSAETOOMANYREFS: Too many references: can't splice",
	WSAETIMEDOUT,		"WSAETIMEDOUT: Connection timed out",
	WSAECONNREFUSED,	"WSAECONNREFUSED: Connection refused",
	WSAELOOP,			"WSAELOOP: Too many levels of symbolic link (!)",
	WSAENAMETOOLONG,	"WSAENAMETOOLONG: File name too long (!)",
	WSAEHOSTDOWN,		"WSAEHOSTDOWN: Host is down",
	WSAEHOSTUNREACH,	"WSAEHOSTUNREACH: No route to host",
	WSAENOTEMPTY,		"WSAENOTEMPTY: Directory not empty (!)",
	WSAEPROCLIM,		"WSAEPROCLIM: Too many processes (!)",
	WSAEUSERS,			"WSAEUSERS: Too many users (!)",
	WSAEDQUOT,			"WSAEDQUOT: Disk quota exceeded (!)",
	WSAESTALE,			"WSAESTALE: Stale NFS handle (!)",
	WSAEREMOTE,			"WSAEREMOTE: Too many levels of remote in path (!)",
/*
 * Extended Windows Sockets error constant definitions
 */
	WSASYSNOTREADY,		"WSASYSNOTREADY: System not ready",
	WSAVERNOTSUPPORTED,	"WSAVERNOTSUPPORTED: Version not supported",
	WSANOTINITIALISED,	"WSANOTINITIALISED: WinSock not initialised",
/*
 * Resolver errors
 */
	WSAHOST_NOT_FOUND,	"WSAHOST_NOT_FOUND: Host not found (authoritative)",
	WSATRY_AGAIN,		"WSATRY_AGAIN: Host not found (non-authoritative)",
	WSANO_RECOVERY,		"WSANO_RECOVERY: Nameserver error",
	WSANO_DATA,			"WSANO_DATA: No host data record of requested type",
	0,					NULL,
};

/////////////////////////////////////////////////////////////////////////////
// Constructor & destructor
//
tcSocket::tcSocket(tcLogger *l, int type) {

	hosttask = 0;
	hostbuf = NULL;
	m_Port = -1;
	logger = l;
	
#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Creating tcSocket, type=%d, this=%p", type, this);
#endif
	
	theGlobalStarter.start(l);			// Make sure winsock is running
	
	sock = socket(AF_INET, type, 0);	// Get a SOCKET from Winsock

#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcSocket: SOCKET=%d", sock);
#endif
	Init(sock);
}

/////////////////////////////////////////////////////////////////////////////
// Create a tcSocket from an existing SOCKET (eg after an accept())
//
tcSocket::tcSocket(tcLogger *l, SOCKET s) {

	hosttask = 0;
	hostbuf = NULL;
	m_Port = -1;
	logger = l;

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Creating tcSocket, SOCKET=%d, this=%p", s, this);
#endif

	theGlobalStarter.start(l);			// Make sure winsock is running

	sock = s;
	Init(sock);
}
                            
/////////////////////////////////////////////////////////////////////////////
// Initialise the tcSocket.
// i)   Check the SOCKET is valid
// ii)  Create a fake Window to receive socket messages
// iii) Store the tcSocket this pointer as a long word in the Window
//      data (so the WindProc can recover the pointer easily)
// iv)  Make sure the socket is not expecting any events
void tcSocket::Init(SOCKET sock) {

	// Stick myself in the timeout queue
	m_hTimeout = ::TimeoutAddSocket( this );
	
	lastErr = 0;
	if (sock == INVALID_SOCKET) {
		lastErr = WSAGetLastError();
    	logger->logf(TC_ERROR, "Socket creation failure: %s",
					 lastErrorMessage());
	}
	else {
		hWnd = CreateWindow(tcSockClassName, "", WS_CHILD, 0, 0, 0, 0,
							tcSockHwndParent, NULL, tcSockInst, NULL);
		if (hWnd == NULL) {
			logger->logf(TC_ERROR, "Couldn't create socket's window!");
			Close();
			return;
		}
		SetWindowLong(hWnd, 0, (long)this);

		evMask = 0;  					// Clear event mask
        updateMask();
	}
}

/////////////////////////////////////////////////////////////////////////////
// tcSocket destructor

tcSocket::~tcSocket() {

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Destroying tcSocket: %p", this);
#endif
    
    ::TimeoutRemSocket( m_hTimeout );	// remove myself from the timeout list
    // Cancel any asynch host resolution requests
    if( hosttask != 0 )
    	WSACancelAsyncRequest( hosttask );
    	// Ignore error return from this
    
	Close();							// Make sure the connection is closed

	// Delete any Window we have.  Note that DestroyWindow will send
	// messages to the Window.  To be on the safe side we clear the
	// this pointer so no pending WM_SOCKET messages get a chance to be
	// delivered
	if (hWnd != NULL) {	
		SetWindowLong(hWnd, 0, 0);
		DestroyWindow(hWnd);
    }
    delete [] hostbuf;  
    hostbuf = NULL;                                
	theGlobalStarter.stop();	// decrement reference count
}

/////////////////////////////////////////////////////////////////////////////
// Update the event mask currently being used by WSAAsyncSelect()
//
void tcSocket::updateMask() {			// Update the event mask
#if TCSOCK_DEBUG
	logger->logf(TC_TRACE, "tcSocket(%p): WSAAsyncSelect(%d, 0x%x, %d, 0x%lx)",
				 this, sock, (int)hWnd, WM_SOCKET, evMask);
#endif
	int rc = WSAAsyncSelect(sock, hWnd, WM_SOCKET, evMask);
	if (rc != 0) {
		lastErr = WSAGetLastError();
		logger->logf(TC_ERROR, "Select error: %d", lastErr);
	}
}

/////////////////////////////////////////////////////////////////////////////
// Close().  Break the connection
//
void tcSocket::Close() {
	if (sock != INVALID_SOCKET) {
		evMask = 0;
		updateMask();
#if TCSOCK_DEBUG
		logger->logf(TC_DEBUG, "tcSocket(%p): closesocket(%d)", this, sock);
#endif
		// Convoluted, because of a sample log which seemed to show
		// closesocket being called twice on the same socket.  I'm
		// *sure* the DLL wouldn't have called SendMessage() from inside
		// closesocket(), but just to be sure, we mark our own socket
		// as invalid so our WindProc won't call us back.  Ugh.
		SOCKET tmp = sock;
		sock = INVALID_SOCKET;
		closesocket(tmp);
	}
}
///////////////////////////////////////////////////////////
// This is really the main connect routine
// Must be passed a hent structure so there are
// **no** async calls going on
int tcSocket::_Connect( sockaddr_in addr , int port )
{
	addr.sin_family = AF_INET;
	addr.sin_port = htons((u_short)port);
	addMask(FD_CONNECT);				// Get connection notifications
	logger->logf(TC_INFO, "tcSocket(%p): connect(%d, %s, %d)",
				 this, sock, inet_ntoa(addr.sin_addr), port);

	int rc = connect(sock, (sockaddr *) &addr, sizeof(addr));
	logger->logf(TC_INFO, "Connect returned %d", rc);
	if (rc == SOCKET_ERROR) {			// Connection not yet complete
		lastErr = WSAGetLastError();
		if (lastErr != WSAEWOULDBLOCK) {
#if TCSOCK_DEBUG		
			logger->logf(TC_ERROR, "tcSocket(%p): Connect error: %s",
						 this, lastErrorMessage());
#endif
			Close();
			return SOCKET_ERROR;
		}
#if TCSOCK_DEBUG		
		logger->logf(TC_DEBUG, "Current event mask: %04x", evMask);
		logger->logf(TC_INFO, "tcSocket(%p): Connection in progress...",
					this);
#endif					
	}
	else {		// Connection completed OK (eg braindead LWP)
		logger->logf(TC_INFO, "Connect() completed synchronously(!)");
        Connected(0);					// Invoke callback
	}
	return( 0 );
}

/////////////////////////////////////////////////////////////////////////////
// Connect: Connect to a remote server specifying host name and
// port number
//
int tcSocket::Connect(const char *host, int port) {
	sockaddr_in	addr;

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "tcSocket(%p)::Connect(%s, %u)", this, host, port);
#endif

	// Naive approach to dotted decimal IP addresses.  If the first
	// character is a digit, treat the whole lot as dotted decimal
	if (isdigit(host[0])) {
		addr.sin_addr.s_addr = inet_addr(host);	// Convert to binary
		if (addr.sin_addr.s_addr == NULL) {		// See if it worked
			lastErr = WSAEINVAL;
			logger->logf(TC_ERROR, "Invalid numeric IP address: \"%s\"", host);
			return SOCKET_ERROR;
        }
		return _Connect( addr , port );
	}
    else {					// Alpha name --- try and look up in DNS
    	return GetHost( host , port );
  	}
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
// Connect().  Connect to a host using host name, service name and
// protocol type (eg "tcp").  Works out port number and calls the other
// Connect()

int tcSocket::Connect(const char *host, const char *serv, const char *prot) {

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "tcSocket(%p)::Connect(%s, %s, %s)",
				 this, host, serv, prot);
#endif

#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcSocket(%p): getservbyname(%s, %s)",
				 this, serv, prot);
#endif
	servent *ent = getservbyname(serv, prot);
	if (ent == NULL) {
		lastErr = WSAGetLastError();
		logger->logf(TC_ERROR, "Unable to find service name \"%s\": %s",
					 serv, lastErrorMessage());
		return SOCKET_ERROR;
	}
	return Connect(host, ntohs(ent->s_port));
}

/////////////////////////////////////////////////////////////////////////////
// Listen().  Listen for incoming connections
//
int tcSocket::Listen(int port) {
	sockaddr_in	addr;

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "tcSocket(%p)::Listen(%d)", this, port);
#endif
	// Ensure no local address binding
	addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = htonl(INADDR_ANY);
    addr.sin_port  = htons((u_short)port);
	
	if (bind(sock, (sockaddr *) &addr, sizeof(addr)) == SOCKET_ERROR) {
		lastErr = WSAGetLastError();
		logger->logf(TC_ERROR, "tcSocket(%p): Bind failed: %s", this,
					 lastErrorMessage());
		Close();
        return lastErr;
	}
	// Set up incoming connection queue
	if (listen(sock, 5) == SOCKET_ERROR) {
		lastErr = WSAGetLastError();
		logger->logf(TC_ERROR, "tcSocket(%p): Listen failed: %s", this ,
					 lastErrorMessage());
		Close();
        return lastErr;
    }
	
	addMask(FD_ACCEPT);			// Get notification when connections arrive

	logger->logf(TC_INFO, "tcSocket(%p): Listening on port %u", this, port);
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
// Accept().  Accept a new connection *** UNTESTED since message handling
// changes***
//
tcSocket *tcSocket::Accept() {

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "tcSocket(%p)::Accept()", this);
#endif
	SOCKET ns = accept(sock, NULL, NULL);
	if (ns == INVALID_SOCKET) {
		lastErr = WSAGetLastError();
		logger->logf(TC_ERROR, "tcSocket(%p): Accept failed: %s",
					 this, lastErrorMessage());
		return NULL;
	}
	return new tcSocket(logger, ns);			// Create new socket
}

/////////////////////////////////////////////////////////////////////////////
// EMK - name resolution stuff
//                            
int tcSocket::GetHost( const char * host , int port )
{                                                   
	// Set up the port storage for when this returns
	m_Port = port;
	
//	addr.sin_addr.s_addr = *((long *) hent->h_addr);
	logger->logf( TC_INFO , "tcSocket(%p) Resolving %s" , this , host );
	// Memory leak coming your way.
	hostbuf = new char[ MAXGETHOSTSTRUCT ];

	// We know that the name is a character string at this point
	hosttask = WSAAsyncGetHostByName( hWnd , WM_SOCKET_HOST , host ,
									hostbuf , MAXGETHOSTSTRUCT );
	if( hosttask == 0 ) {
		lastErr = WSAGetLastError();
		logger->logf(TC_ERROR, "tcSocket(%p): WSAsyncGetHostByName failed: %s", this  ,
					 lastErrorMessage());                         
		// clean up buffer
		delete [] hostbuf;
		hostbuf = NULL;
		return SOCKET_ERROR;
	}
	else  {
		return 0;
	}
}
		
		
/////////////////////////////////////////////////////////////////////////////
// Default action functions for socket events
//
void tcSocket::Connected(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): Connected: error %d", this, error);
}

void tcSocket::Accepted(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): Accepted: error %d", this, error);
}

void tcSocket::Closed(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): Closed: error %d", this, error);
}

void tcSocket::Readable(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): Readable: error %d", this, error);
}

void tcSocket::Writable(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): Writable: error %d", this, error);
}

void tcSocket::OOBData(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): OOBData: error %d", this, error);
}

// This function is called back when we get a WM_SOCKET_HOST message
void tcSocket::HostResolve( HANDLE task, int error ) 
{ 
#if TCSOCK_TRACE
	logger->logf( TC_ERROR , "tcSocket(%p): Got WM_SOCKET_HOST(%d,%d)" , task, error);
#endif	
	// Check for error
	if( error == 0 )
	{
		// The buffer contains a hent structure
		hostent * hent = (hostent *)hostbuf;
		sockaddr_in addr;    
		if( hent->h_addr != NULL )
		{
			addr.sin_addr.s_addr = *((long *) hent->h_addr);
			ASSERT( m_Port != -1 );
			int Ret = _Connect( addr , m_Port);
		}
		else
		{
			lastErr = WSAGetLastError();
			logger->logf( TC_ERROR , "tcSocket(%p): Null host entry (couldn't resolve?) %s" , lastErrorMessage() );
		}
	}
	else
	{
		// There's an error
		Connected( error );
	};
	delete [] hostbuf;
	hostbuf = NULL;   
	hosttask = 0;
	m_Port = -1;
}                                                           


void tcSocket::Timeout() {
	 logger->logf( TC_ERROR , "tcSocket(%p): Timeout" , this );
}

/////////////////////////////////////////////////////////////////////////////
// sockErrMessage() -- convert a numeric winsock error to something
// approximately human readable
//
const char *tcSocket::sockErrMessage(int err) {
	static char fakebuf[80];
	
	if (err == 0) return "No WinSock error";

	for (errDesc *cur = errString; cur->num != 0; cur++) {
		if (cur->num == err) return cur->string;
	}
	sprintf(fakebuf, "Unrecognised error number %d", err);
	return fakebuf;
}

/////////////////////////////////////////////////////////////////////////////
// tcBufSocket member functions
//

tcBufSocket::tcBufSocket(tcLogger *l, int rbs, int wbs)
: tcSocket(l, SOCK_STREAM) {			// Only makes sense with TCP sockets

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Creating tcBufSocket, this=%p", this);
#endif
	Init(rbs, wbs);
}

tcBufSocket::tcBufSocket(tcLogger *l, SOCKET s, int rbs, int wbs)
: tcSocket(l, s) {

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Creating tcBufSocket, this=%p", this);
#endif
	Init(rbs, wbs);
}

tcBufSocket::~tcBufSocket() {

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Destroying tcBufSocket, this=%p", this);
#endif
	if (rBuf != NULL) delete [] rBuf;
	if (wBuf != NULL) delete [] wBuf;
}

void tcBufSocket::Init(int rbs, int wbs) {

	rbSize = wbSize = nrBuf = nwBuf = 0;	// Initialise
	if ((rBuf = new char[rbs]) != NULL) rbSize = rbs;
	if ((wBuf = new char[wbs]) != NULL) wbSize = wbs;
	setMask(FD_READ | FD_WRITE | FD_OOB | FD_CLOSE);
	wrpending = 0;
}


/////////////////////////////////////////////////////////////////////////////
// send()  --- Buffer data for later sending.  It will be sent the
// next time the app. returns to Windows, causing the objects
// dataWritten callback to be invoked.  If the buffer is empty, we may
// need to send ourselves a FD_WRITE message to kickstart everything
//
int tcBufSocket::send(const char *data, int amt) {
    int empty = (nwBuf == 0);				// Is buffer empty?

	if (amt == -1) amt = strlen(data);		// Default amount
	
#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcBufSocket(%p): send(%p, %u)", this, data, amt);
#endif
	if (amt > wrSpace()) {
		lastErr = WSAENOBUFS;
		logger->logf(TC_ERROR, "tcBufSocket(%p): send(): Out of buffer space",
					 this);
		return SOCKET_ERROR;
	}
	if (amt > 0)  {
		memcpy(wBuf + nwBuf, data, amt);	// Copy the data in
		nwBuf += amt;
    }

	// If buffer was empty and no FD_WRITE messages outstanding, post one
	// But only if we know there isn't one of our own messages still
	// pending
	if (empty && (wrpending == 0)) {
		wrpending++;
#if TCSOCK_DEBUG
		logger->logf(TC_DEBUG, "Posting FD_WRITE message to self");
#endif
		PostMessage(hWnd, WM_SOCKET, sock, WSAMAKESELECTREPLY(FD_WRITE, 0));
	}
	return wrSpace();
}

/////////////////////////////////////////////////////////////////////////////
// sendi() --- Try and send data immediately.  If the socket will accept
// the data, then fine, otherwise we buffer it (so it will be sent soon)
// and return 0 (yuck).
//
int tcBufSocket::sendi(const char *data, int amt) {
	int rc;

	if (amt == -1) amt = strlen(data);		// Default amount
#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcBufSocket(%p): sendi(%p, %u)", this, data, amt);
#endif
	if (nwBuf) {
		if (sendBuffer() == SOCKET_ERROR) return SOCKET_ERROR;
		if (nwBuf) {						// Still chars buffered
			send(data, amt);				// Queue data for sending
		}
		return SOCKET_ERROR;
	}

	rc = ::send(sock, data, amt, 0);		// Immediate send
	if (rc < 0) {
		lastErr = WSAGetLastError();
		if (lastErr == WSAEWOULDBLOCK) {	// Socket would have blocked
			wrpending++;					// No need for kick start
			rc = 0;
		}
		else {
			writeError(lastErr);
			return SOCKET_ERROR;
		}
	}
	if (rc == amt) return amt;				// Success

	amt -= rc;
	data += rc;								// Move past sent data

	// Queue for later sending
	if (send(data, amt) == SOCKET_ERROR) return SOCKET_ERROR;	// Error
	return 0;								// Indicate we're blocking
}

/////////////////////////////////////////////////////////////////////////////
// Receive data from the buffer
//
int tcBufSocket::recv(char *buf, int amt) {

#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcBufSocket(%p): recv(%p, %u)", this, buf, amt);
#endif
	if (amt > nrBuf) amt = nrBuf;			// Limit to amount buffered
	if (amt != 0) {
		memcpy(buf, rBuf, amt);				// Give data to user
		rdPullUp(amt);						// Re-align the data 
	}
	return amt;
}


/////////////////////////////////////////////////////////////////////////////
// Attempt to send the current contents of the send buffer
//
int tcBufSocket::sendBuffer() {

	if (nwBuf == 0) return 0;			// Nothing to do

	int error = 0;
	while ((error == 0) && (nwBuf != 0)) {
		int amt = ::send(sock, wBuf, nwBuf, 0);
#ifdef TCSOCK_DEBUG
		logger->logf(TC_DEBUG, "::send(%p, %d) returned %d", wBuf, nwBuf, amt);
#endif
		if (amt < 0) {
			error = WSAGetLastError();		// Save error value
		}
		else if (amt > 0) {					// Actually wrote something
			wrPullUp(amt);
		}
	}
	if (error != 0) {
		lastErr = error;
		if (lastErr != WSAEWOULDBLOCK && lastErr != WSAEINPROGRESS ) {
			writeError(lastErr);
		}
		return SOCKET_ERROR;
	}
	return nwBuf;
}

/////////////////////////////////////////////////////////////////////////////
// Readable().  We override the tcSocket Readable() member so we can
// buffer input.  We read as much as possible and then call the object's
// dataRead callback

void tcBufSocket::Readable(int error) {		// Called by socket manager

#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcBufSocket(%p): Readable(%d)", this, error);
#endif
	if ((error == 0) && (rdSpace() != 0)) {	// Try to fill buffer
		int amt = ::recv(sock, rBuf + nrBuf, rdSpace(), 0);
		if (amt < 0) {
			error = WSAGetLastError();		// Save error number
		}
		else if (amt > 0) {
			nrBuf += amt;					// Update count
#if TCSOCK_DEBUG
			logger->logf(TC_DEBUG,
						 "tcBufSocket(%p):  read %d bytes. %d buffered",
						 this, amt, nrBuf);
#endif
			if (rdSpace() == 0)	 {			// No space for more data!
				logger->logf(TC_INFO, "tcBufSocket(%p): Recv buffer full",
							 this);
			}
			dataRead();						// Call callback
		}
	}
	// EMK
	if( error == WSAEINPROGRESS )
	{
		// Something is already going on in the socket library
		// so we just post to ourself
		PostMessage(hWnd, WM_SOCKET, sock, WSAMAKESELECTREPLY(FD_READ, 0));
#if TCSOCK_DEBUG
		logger->logf(TC_DEBUG, "tcBufSocket(%p): Readable but something in progress", this );
#endif
		
	}                                                                
	else
	{
		if ((error != 0) && (error != WSAEWOULDBLOCK))
			readError(error);					// Pass on to error handler
	}
}

/////////////////////////////////////////////////////////////////////////////
// Writable().  We override tcSocket()'s Writable to send as much
// data from the send buffer as possible.  If we manage to empty the
// send buffer, then we call the object's dataWritten() callback.

void tcBufSocket::Writable(int error) {		// Called by socket manager

#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcBufSocket(%p): Writable(%d)", this, error);
#endif
	if (error != 0)
		logger->logf(TC_ERROR, "tcBufSocket::Writable called with error %d",
					 error);

	if (wrpending > 0) wrpending--;			// Note the message has arrived
	if (nwBuf == 0) {
#if TCSOCK_DEBUG
		logger->logf(TC_DEBUG, "tcBufSocket(%p): Writable() but no data)",
					this);
#endif
		return;
    }

	if (sendBuffer() == SOCKET_ERROR) 
	{
		if (lastErr == WSAEWOULDBLOCK)
			logger->logf(TC_DEBUG,
						 "Write would block.  Waiting for FD_WRITE message");
		// EMK
		if( lastErr == WSAEINPROGRESS )
		{
			// Something is already going on in the socket library
			// so we just post to ourself
			PostMessage(hWnd, WM_SOCKET, sock, WSAMAKESELECTREPLY(FD_WRITE, 0));
#if TCSOCK_DEBUG
			logger->logf(TC_DEBUG, "tcBufSocket(%p): Writable but something in progress", this );
#endif
		
		}                                                                
	}
	if (nwBuf == 0) {
		dataWritten();						// Call callback
	}
}

void tcBufSocket::readError(int error) {

	lastErr = error;
	logger->logf(TC_ERROR, "tcBufSocket(%p): Read error: %s", this,
				 sockErrMessage(error));
}

void tcBufSocket::writeError(int error) {

	lastErr = error;
	logger->logf(TC_ERROR, "tcBufSocket(%p): Write error: %s", this,
				 sockErrMessage(error));
}


/////////////////////////////////////////////////////////////////////////////
// Line buffered sockets...Not terribly well tested.  These are a simple
// derivation of tcBufSockets.  No extra output processing is done as
// a tcBuf Socket can already do that, but we override the dataRead()
// member to split incoming data into lines.  For every line we call
// the object's lineRead() callback.


/////////////////////////////////////////////////////////////////////////////
// readLine().  The application calls this to get the next line of data
// out of the socket's buffer.  It's pretty inefficient.

int tcLineBufSocket::readLine(char *buffer, int len) {

#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcLineBufSocket(%p): readLine(%p, %u)", this,
				 buffer, len);
#endif
	if (nrBuf == 0) {					// No data
		lastErr = EIO;
		logger->logf(TC_ERROR, "No data available for readLine");
		return SOCKET_ERROR;
	}
	char *s = (char *) memchr(rBuf, '\n', nrBuf);	// Find end of line
	if (s == NULL) {
		lastErr = EIO;
		logger->logf(TC_ERROR, "Incomplete line data");
		return SOCKET_ERROR;
	}
	int linelen = (s - rBuf) + 1;		// Space needed to hold line
	if (linelen > len) {				// Too big...
		lastErr = WSAENOBUFS;
		logger->log(TC_ERROR, "readLine: User's buffer is too small");
		return SOCKET_ERROR;			
	}
	if (recv(buffer, linelen) != linelen) {
		lastErr = WSAEFAULT;			// Ok, so that's *really* horrible!
		logger->logf(TC_ERROR, "Internal error in readLine()!");
		return SOCKET_ERROR;
	}
	linelen--;							// Get rid of the \n
	buffer[linelen] = '\0';
	return (linelen);
}

/////////////////////////////////////////////////////////////////////////////
// dataRead().  Called by tcBufSocket::Readable() when it has added more
// data to the buffer.  While there are still lines of data in the
// buffer, call the object's lineRead() callback
//
void tcLineBufSocket::dataRead() {

	if (memchr(rBuf, '\n', nrBuf) != NULL) {	// Got a line?
		while (memchr(rBuf, '\n', nrBuf) != NULL) lineRead();
	}
	else {								// Test for overflow
		if (rdSpace() == 0) {			// TODO: Finish this
			logger->logf(TC_ERROR, "Warning! Read buffer overflow!");
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
// CRLF Line buffered sockets...Not terribly well tested.  These are a simple
// derivation of tcBufSockets.  No extra output processing is done as
// a tcBuf Socket can already do that, but we override the dataRead()
// member to split incoming data into lines.  For every line we call
// the object's lineRead() callback.
int tcCRLFLineBufSocket::readLine(char *buffer, int len) {

	if (nrBuf == 0) {					// No data
		lastErr = EIO;
		logger->logf(TC_ERROR, "No data available for readLine");
		return SOCKET_ERROR;
	}
	char *s = (char *) memchr(rBuf, '\n', nrBuf);	// Find end of line
	if (s == NULL) {
		lastErr = EIO;
		logger->logf(TC_ERROR, "Incomplete line data");
		return SOCKET_ERROR;
	}
	int linelen = (s - rBuf) + 1;		// Space needed to hold line
										// 2 because there's a cr and
										// and lf
	if (linelen > len) {				// Too big...
		lastErr = WSAENOBUFS;
		logger->log(TC_ERROR, "readLine: User's buffer is too small");
		return SOCKET_ERROR;			
	}
	if (recv(buffer, linelen) != linelen) {
		lastErr = WSAEFAULT;			// Ok, so that's *really* horrible!
		logger->logf(TC_ERROR, "Internal error in readLine()!");
		return SOCKET_ERROR;
	}
	linelen--;							// Get rid of the \n
	buffer[linelen] = '\0';
	linelen--;							// Get rid of the \r
	buffer[linelen] = '\0';
#if TCSOCK_TRACE
	logger->logf(TC_DEBUG, "tcCRLFLineBufSocket(%p): readLine(%s)", this, buffer );
#endif
	return (linelen);
}

/////////////////////////////////////////////////////////////////////////////
// dataRead().  Called by tcBufSocket::Readable() when it has added more
// data to the buffer.  While there are still lines of data in the
// buffer, call the object's lineRead() callback
//
void tcCRLFLineBufSocket::dataRead() {

	if (memchr(rBuf, '\n', nrBuf) != NULL) {	// Got a line?
		while (memchr(rBuf, '\n', nrBuf) != NULL) lineRead();
	}
	else {								// Test for overflow
		if (rdSpace() == 0) {			// TODO: Finish this
			logger->logf(TC_ERROR, "Warning! Read buffer overflow!");
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
// The C code that binds it all together.  We rely on the application to
// provide initialisation data, which we store.  At init time, we
// register a new Window class to use for socket messages, whose
// WNDPROC points at tcSockWindProc() which handles event dispatching.
//
HINSTANCE tcSockInst = NULL;
HWND tcSockHwndParent;
char tcSockClassName[66];					// Yeah, I know...

/////////////////////////////////////////////////////////////////////////////
// Free function to initialise the library.  Needs an HWND to parent
// the hidden socket windows, an HINSTANCE for CreateWindow (although
// it can infer this from the HWND if omitted) and a class name to use
// when creating the hidden windows (with a default)

int tcSocketInit(HWND hwnd, const char *className, HINSTANCE hinst, BOOL DelayInit ) {
	WNDCLASS 	wndclass;
    ATOM		atom;

	if( !DelayInit )
	{
		theGlobalStarter.start( NULL );
	}
	// Handle defaulted hinst
	if (hinst == NULL) {
#ifdef WIN32
		hinst = (HINSTANCE)GetWindowLong(hwnd, GWL_HINSTANCE); //CRCS 5/10/94
#else
		hinst = (HINSTANCE)GetWindowWord(hwnd, GWW_HINSTANCE);
#endif
	}

	// Check we aren't being called twice (the MessageBox shouldn't
    // really be here...)
	if (tcSockInst != NULL) {
		::MsgBox( MB_OK | MB_ICONEXCLAMATION , IDS_MSG_SOCK_TWICE );
        return -1;
	}
    tcSockHwndParent = hwnd;

// Register a windows class
	tcSockInst				= hinst;
	strncpy(tcSockClassName, className, 64);
	tcSockClassName[64] = '\0';

	wndclass.style			= 0;
	wndclass.lpfnWndProc	= tcSockWindProc;
	wndclass.cbClsExtra		= 0;
	wndclass.cbWndExtra		= 2 * sizeof(long);
	wndclass.hInstance		= tcSockInst;
	wndclass.hIcon			= NULL;
	wndclass.hCursor		= NULL;
	wndclass.hbrBackground	= NULL;
	wndclass.lpszMenuName	= NULL;
	wndclass.lpszClassName	= tcSockClassName;

	atom = RegisterClass(&wndclass);
	return (atom == 0) ? -1 : 0;
}

/////////////////////////////////////////////////////////////////////////////
// tcSockWindProc().  Event dispatching for tcSockets.
// For WM_SOCKET messages, figure out the pointer to the tcSocket, which
// was stored in the Windows extra instance data bytes as a long, so
// we just pull it out & cast it.  A NULL pointer means the  tcSocket
// is not prepare to receive events.
// Next we docode the message type and the error code.  We switch on
// the message type to call the correct member function of the
// tcSocket, passing it the error code.
//
// All other messages get punted back to Windows for it to deal with
//
long FAR PASCAL _export tcSockWindProc(HWND hwnd, UINT message, UINT wParam,
										LONG lParam) {
    int error;
    int event;         
    int buflen;
    tcSocket * s;
        
    switch( message ) {
    case WM_SOCKET:
		event = WSAGETSELECTEVENT(lParam);			// Get info.
		error = WSAGETSELECTERROR(lParam);
		s = (tcSocket *) GetWindowLong(hwnd, 0);	// Get socket

		if ((s == NULL) || !(s->isValid())) return 0;	// Already closed

		switch(event) {
			case FD_READ:		s->Readable(error); 	break;
			case FD_WRITE:		s->Writable(error); 	break;
			case FD_OOB:		s->OOBData(error); 		break;
			case FD_ACCEPT:		s->Accepted(error); 	break;
			case FD_CONNECT:	s->Connected(error);	break;
			case FD_CLOSE:		s->Closed(error); 		break;
		}
		return 0;
	case WM_SOCKET_HOST:
		// We have the host information 
                error = WSAGETASYNCERROR(lParam);
                buflen = WSAGETASYNCBUFLEN(lParam);
		s = (tcSocket *)GetWindowLong( hwnd , 0 );
		if ((s == NULL) || !(s->isValid())) 
			return 0;	// Already closed
		else
		{
			
			s->HostResolve( HANDLE(wParam) , error );
        }
        return 0;	
    default:
		return DefWindowProc(hwnd, message, wParam, lParam);
    }
}
