/***********************************************************************
**
** CSOCKET.CPP
**
** Copyright 1994 by Ewan Kirk <ewan@kirk.demon.co.uk>
**
** Permission to use, copy, modify, distribute, and sell this software and its
** documentation for any purpose is hereby granted without fee, provided that
** the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation, and that the name of Ewan Kirk not be used in
** advertising or publicity pertaining to distribution of the software without
** specific, written prior permission.  Ewan Kirk makes no representations
** about the suitability of this software for any purpose.  It is provided
** "as is" without express or implied warranty.
**
** EWAN KIRK DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
** INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
** EVENT SHALL EWAN KIRK BE LIABLE FOR ANY SPECIAL, INDIRECT OR
** CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
** DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
** TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
** PERFORMANCE OF THIS SOFTWARE.
**
** This file is part of WINDIS. 
**
** CSockets are a simple wrapper round the tcsocket* classes.
** Included here is the time socket and the finger sockets
** which don't have to do very much.
** More complicated sockets like the nntp socket and the
** smtp sockets are implemented in their separate files.
** 
** The logger is also here.
**
** V1.0 - Initial Revision 01-Sep-94
**
*/

#include "stdafx.h"
#include <stdarg.h>
#include "windis.h"
#include "utility.h"
#include "csocket.h"
#include "finwnd.h"
#include "timedlg.h"

// A line socket created from scratch
CLineSocket::CLineSocket( tcLogger * Log , int Rbs , int Wbs )
	: tcCRLFLineBufSocket( Log , Rbs , Wbs )
{
}                                    

// A line socket created from another socket after accept
CLineSocket::CLineSocket( tcLogger * Log , SOCKET s , int Rbs , int Wbs )
	: tcCRLFLineBufSocket( Log , s , Rbs , Wbs )
{
}

CLineSocket::~CLineSocket()
{
	// Nothing to do on deconstruction
}

// The simple inline constructors for the time Socket
CTimeSocket::CTimeSocket( tcLogger * Log , CTimeDlg * pParent , int Rbs , int Wbs )
	: tcBufSocket( Log , Rbs , Wbs )
{
	m_Parent = pParent;
}


CTimeSocket::~CTimeSocket()
{
}

void CTimeSocket::dataRead()
{
	long l;
	time_t Time;
	// Get the data out of the buffer
	ASSERT( nrBuf == 4 );                             
	// read the stuff out of the buffer
	recv( (char *)&l , 4 );
	Time = ntohl( l ) - 2208988800l;  
	ASSERT_VALID( m_Parent );
	m_Parent->ReadTime( Time );
}

void CTimeSocket::dataWritten()
{
	// We shouldn't have got this actually since we never write to
	// the socket             
	ASSERT( 0 == 1 ); // This will fail if we ever get here
}

void CTimeSocket::Connected(int Error)
{
	ASSERT_VALID( m_Parent );
	m_Parent->Connected( Error , this );
}

void CTimeSocket::Timeout()
{
	ASSERT_VALID( m_Parent );
	CTimeoutType Type = TypTimeout();
	logger->logf( TC_ERROR , "CSmtpSocket(%p): Timeout %d" , this , Type );
	m_Parent->Timeout(this);
}

void CTimeSocket::Closed( int Error )
{
	ASSERT_VALID( m_Parent );
	m_Parent->Closed( Error ,this );
}

// The simple inline constructors for the time Socket
CFingerSocket::CFingerSocket( tcLogger * Log , CFingerWnd * pParent , int Rbs , int Wbs )
	: CLineSocket( Log , Rbs , Wbs )
{
	ASSERT_VALID( pParent );
	m_Parent = pParent;
}


CFingerSocket::~CFingerSocket()
{
}

void CFingerSocket::lineRead()
{
	// Get the data out of the buffer
	char Buffer[ 256 ];
	readLine( Buffer , 255 );
	m_Parent->AddLine( Buffer );
}

void CFingerSocket::Connected(int Error)
{
	ASSERT_VALID( m_Parent );
	m_Parent->Connected( Error );
}                      

void CFingerSocket::Closed( int Error )
{
	ASSERT_VALID( m_Parent );
	m_Parent->Closed( Error ,this );
}

void CFingerSocket::Timeout()
{
	ASSERT_VALID( m_Parent );
	CTimeoutType Type = TypTimeout();
	logger->logf( TC_ERROR , "CSmtpSocket(%p): Timeout %d" , this , Type );
	m_Parent->Timeout(this);
}



///////////////////////////////////////////////////////////////////////////
// The logger functions                                                    
//
// These classes are local to this file
// This is a completely null logger which does bugger all
class CNullLogger : public tcLogger {
  public:
	virtual void log( tcLogLevel , const char * ) {;}
};                                                       

class CFileLogger : public tcLogger {
  public:
  	CFileLogger( const char * Str );
  	~CFileLogger();
  	virtual void log( tcLogLevel Level , const char * Str );
  private:
  	BOOL		m_Valid;
  	CStdioFile 	m_File;
};

// The global function that allocates a logger
tcLogger * AllocateLogger( const char * Name )
{
	// Get the logger configuration options
	// Can we be bothered with logging?
	if( !gConfig->GetSocketLogging() )
		return new CNullLogger;        
	// Ok, we want to log but where do we log to?
	CString Path = gConfig->GetLogPath();
	CString Tmp = Name;         
	ASSERT( Tmp.Find(".") == -1 );
	Path += "\\" + Tmp.Left( 8 ) + ".log";
	return new CFileLogger( Path );
}
	


CFileLogger::CFileLogger( const char * FileName )
{                                                          
	ASSERT( FileName != NULL );
	m_Valid = FALSE;	
	if( !m_File.Open( FileName , CFile::modeCreate ) )
		return;
	m_Valid = TRUE;
}

CFileLogger::~CFileLogger()
{                  
	if( m_Valid )
		m_File.Close();
}

void CFileLogger::log( tcLogLevel , const char * Str)
{       
	if( m_Valid )
	{                               
		m_File.WriteString( Str );
		m_File.WriteString( "\n" );              
		m_File.Flush();
	}
}
