/***********************************************************************
**
** WEXPIRE.CPP
**
** Copyright 1994 by Ewan Kirk <ewan@kirk.demon.co.uk>
**
** Permission to use, copy, modify, distribute, and sell this software and its
** documentation for any purpose is hereby granted without fee, provided that
** the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation, and that the name of Ewan Kirk not be used in
** advertising or publicity pertaining to distribution of the software without
** specific, written prior permission.  Ewan Kirk makes no representations
** about the suitability of this software for any purpose.  It is provided
** "as is" without express or implied warranty.
**
** EWAN KIRK DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
** INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
** EVENT SHALL EWAN KIRK BE LIABLE FOR ANY SPECIAL, INDIRECT OR
** CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
** DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
** TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
** PERFORMANCE OF THIS SOFTWARE.
**
** This file is part of WINDIS. 
**
** The implementation of the Expirer object.  Create this object,
** tell it to get started and it expires your news for you.
** Has a StatusDialog member which keeps you informed of progress.
** But, there might be a few bugs in this.  Take care.
**
** V1.0 - Initial Revision 01-Sep-94
**
*/
// wexpire.cpp : implementation file
//

#include "stdafx.h"
#include "windis.h"
#include "utility.h"
#include "wexpire.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

BOOL CExpirer::DoError( const char * Str )
{               
	::MsgBox( MB_OK | MB_ICONHAND , (LPSTR)Str );
	return FALSE;
}

/////////////////////////////////////////////////////////////////////////////
// CExpirer constructor

CExpirer::CExpirer()
{
	m_ActiveList = 	NULL;
	m_CurrentGroup = NULL;
	m_CurrentPos = NULL;
	m_GroupFile = m_IndexFile = m_NewGroupFile = m_NewIndexFile = NULL;
	
	m_CurrentTime = CTime::GetCurrentTime();
 	m_CurrentExpire = CTimeSpan( 0 , 0 , 0 , 0  );
 	m_OkToWriteActive = FALSE;
}                                                         

CExpirer::~CExpirer()
{                                              
	if ( m_ActiveList )
	{
		if( m_ActiveList->IsOk() && m_OkToWriteActive )
			m_ActiveList->WriteActive();
		delete m_ActiveList;
	}
}

BOOL CExpirer::Init()
{   
	char TmpName[_MAX_PATH];
	
 	wsprintf( TmpName, "%sactive" , m_Setup->NewsDir );
 	              
 	m_ActiveList = new CActiveList( TmpName );
 	if (!m_ActiveList->IsOk() )
 		return DoError( "Cannot Open Active File" );
	
	// Now open the Expire list
	m_ExpireList = new CExpireList( TmpName , m_Setup->ExpireDaysOverRide );
	if (!m_ExpireList->IsOk() )
		return DoError( "Cannot Open Expire File" );
	
	CString TmpArchive;
	if( !m_ExpireList->GetExpireTime("[Default]", m_DefaultExpireTime , TmpArchive ) )
		return DoError( "Cannot find a default expiry time in expire file" );

	// Open the First Group
	m_CurrentPos = m_ActiveList->GetStartPosition();
	return TRUE;
}

 
BOOL CExpirer::ExpireMessage()
{                                                           
	char Buf[ NEWS_BUF_LEN + 1 ];
	CString Subject;
	// this Handler is repeatedly called to expire each
	// message in the database.  Once we reach the
	// end of the index file, we post an OnExpireNext
	// message to move onto the next group.

	wsprintf( Buf , "%ld" , m_CurrentNum );
	if( m_IndexFile->ReadString( Buf , NEWS_BUF_LEN ) )
	{
		// Do this index entry
		if ( m_CurrentNum != atol( Buf + 9 ) )
		{                     
			// Bad but it works
			wsprintf( Buf , "In %s, found %lu when expected %lu" , 
							m_CurrentGroup->GetGroup(), atol( Buf + 9 ) , m_CurrentNum );
			DoError( Buf );
			return FALSE;
		}
		CTime ArtTime( (time_t)atol(Buf + 18) );
		CTimeSpan  Diff = m_CurrentTime - ArtTime;
		if( Diff > m_CurrentExpire )
		{
			// This guy is being expired
			m_CurrentGroup->IncLoNum();
		}
		else
		{
			// This guy isn't so copy him to the new file
			DWORD Offset = atol( Buf );
			DWORD Where = m_NewGroupFile->GetPosition();
			BOOL DoneSubject = FALSE;
			m_GroupFile->Seek( Offset, CFile::begin );
			while( m_GroupFile->ReadString( Buf , NEWS_BUF_LEN ) )
			{
				if ( !DoneSubject && (strnicmp(Buf, "Subject:", 8) == 0) )
				{                    
					DoneSubject = TRUE;
					if (strlen(Buf) > 136) 
					{
						Buf[136] = '\n';
						Buf[137] = '\0';
					}
					Subject = Buf + 8;
				}
				m_NewGroupFile->WriteString( Buf );
				if (strnicmp(Buf, "@@@@END", 7) == 0)
					break;
			}                                      
			
			wsprintf( Buf , "%08ld %08ld %09ld %s", Where, m_CurrentNum ,
								(time_t)ArtTime.GetTime(), (const char *)Subject);
			m_NewIndexFile->WriteString( Buf );
		}                                                    
		m_CurrentNum++;
		return TRUE;
	}
	else
	{
		
		// Get the statistics
		long Deleted = m_GroupFile->GetLength() - m_NewGroupFile->GetLength();
				
		wsprintf( Buf , "%s(%d) %5ldkb %5ldkb %6ld", m_CurrentGroup->GetGroup() ,
						(int)((long)m_CurrentExpire.GetTotalSeconds() / 86400l) ,
						Deleted / 1000 ,
						m_NewGroupFile->GetLength()	/1000,
						m_CurrentGroup->GetHiNum() - m_CurrentGroup->GetLoNum() );
		m_StatDlg.SetLine2( Buf );		
		// Close and delete all the files
		delete m_GroupFile;
		delete m_NewGroupFile;
		delete m_IndexFile;
		delete m_NewIndexFile;

		// Rename and remove files
		CFile::Remove( 	m_CurrentGroup->GetGroupFile( m_Setup->NewsBaseDir ) );
		CFile::Remove( 	m_CurrentGroup->GetIndexFile( m_Setup->NewsBaseDir ) );
		CFile::Rename( 	m_CurrentGroup->GetNewGroupFile( m_Setup->NewsBaseDir ),
		        		m_CurrentGroup->GetGroupFile( m_Setup->NewsBaseDir )    );
		CFile::Rename(  m_CurrentGroup->GetNewIndexFile( m_Setup->NewsBaseDir ),
		        		m_CurrentGroup->GetIndexFile( m_Setup->NewsBaseDir )    );

		return FALSE;
	}
	return FALSE;
} 

BOOL CExpirer::ExpireNextGroup()
{                               
	// When we get this message, we try to open the next group
	// and its files
	// When we get to the end of the files, expire the history file
	CString FName;
	
	if( m_CurrentPos != NULL )
	{
		m_CurrentGroup = m_ActiveList->GetNextGroup( m_CurrentPos );
        if( m_CurrentGroup != NULL )
        {   
        	m_StatDlg.SetLine1( m_CurrentGroup->GetGroup() );
        	TRY
        	{   
        		CFileStatus Stat;
        		UINT Mode;
       			FName = m_CurrentGroup->GetGroupFile( m_Setup->NewsBaseDir );
       			if ( !CFile::GetStatus( FName , Stat ) )
       				// This group doesn't exist at the moment so we'll need
       				// to create the old files.
       				Mode = CFile::modeRead | CFile::modeCreate | CFile::typeBinary;
       			else
    				Mode = CFile::modeRead | CFile::typeBinary;
    				   			
	   			m_GroupFile = new CStdioFile( FName , Mode );
	       		FName = m_CurrentGroup->GetIndexFile( m_Setup->NewsBaseDir );
				m_IndexFile = new CStdioFile( FName , Mode );

				// new files are always created
       			FName = m_CurrentGroup->GetNewGroupFile( m_Setup->NewsBaseDir );
       			m_NewGroupFile = new CStdioFile( FName , CFile::modeCreate | CFile::typeBinary );
       			FName = m_CurrentGroup->GetNewIndexFile( m_Setup->NewsBaseDir );
				m_NewIndexFile = new CStdioFile( FName , CFile::modeCreate | CFile::typeBinary );
				m_IndexFile->SeekToBegin();
				m_NewIndexFile->SeekToBegin();
				
				// Set up the expiry time
				m_ExpireList->GetExpireTime( m_CurrentGroup->GetGroup() , m_CurrentExpire , m_CurrentArchive );
			}
			CATCH( CFileException, e )
			{
				DoError( "Problem Opening Files" );
			}
			END_CATCH
				
		}
		else
		{
			DoError( "Problem Opening Group" );
		}   
		m_CurrentNum = 1;  // Fixed by Dave Hodgson (14Nov94)
		return TRUE;
	}
	else
	{
		// Now expire the History File          
		char * Buf = new char[ NEWS_BUF_LEN + 1];
		CString HistoryDir = m_Setup->NewsDir;
		CString HistoryName = HistoryDir + "history.snw";
		CString NewName = HistoryDir + "history.new";
		CString BackName = HistoryDir + "history.bak";
		TRY
		{
		
			// Set up the expiry time
			m_ExpireList->GetExpireTime( m_CurrentGroup->GetGroup() , m_CurrentExpire , m_CurrentArchive);
			
			CStdioFile HistoryFile( HistoryName , CFile::modeRead | CFile::typeBinary );
			CStdioFile NewFile( NewName , CFile::modeCreate | CFile::typeBinary );
			int i = 0;
			m_StatDlg.SetLine1( "History File" );
			while( HistoryFile.ReadString( Buf , NEWS_BUF_LEN ) )
			{         
				i++;
				if( i % 10 == 0 )
				{
					char WeeBuf[ 16 ];
					itoa( i , WeeBuf , 10 );
				}
				CString Out = Buf;
				char * p = strtok( Buf , " \t\n" );
				
				p = strtok( NULL , " \t\n" );
				CTime ArtTime = (time_t)atol( Buf );
				CTimeSpan  Diff = m_CurrentTime - ArtTime;

				if( Diff <= m_CurrentExpire )
					NewFile.WriteString( Buf );
			};
			HistoryFile.Close();
			NewFile.Close();                                
			
			unlink( BackName );
			rename( HistoryName , BackName );
			rename( NewName , HistoryName );
			// Done it all so set the Ok to write flag
			m_OkToWriteActive = TRUE;
		}
		CATCH( CFileException , e )
		{
			DoError( "Problem Expiring History" );
		}         
		AND_CATCH_ALL( e )
		{            
			DoError( "Unknown Problem" );
		}
		END_CATCH_ALL         
		delete [] Buf;
		return FALSE;
	}
	return FALSE;
} 
   
   
   
/////////////////////////////////////////////////////////////////////
// Expire support classes

CExpire::CExpire( const char * Group , int Days , BOOL Flag, const char * Archive )
 : m_Group( Group ) , m_Expire( Days ) , m_Flag( Flag ) , m_Archive( Archive )
{
	// NOthing to do here
}

CExpire::~CExpire()
{
	// String reclaimed automatically
}

// All the work goes on in this class
CExpireList::CExpireList( const char * FileName , int OverRide )
  : m_FileName( FileName ) , m_Ok( FALSE ) , m_OverRide( OverRide )
{
	char Buffer[ 256 ];                       
	char * p;
	
	// Create the list
	m_ExpireList = new CObArray();	
	// Try to load the expire file
	CStdioFile ExpireFile;
	if( !ExpireFile.Open( m_FileName , CFile::modeRead | CFile::typeText ) )
	{
		// Some sort of error message
	}
	else
	{
		while( ExpireFile.ReadString( Buffer , 255 ) )
		{
			if (Buffer[0] == '\x1a')
				break;
			if ((Buffer[0] == '#') || (Buffer[0] == '\n'))
				continue;
			if ((p = strtok(Buffer, " ,\t\n\r")) == NULL) 
				return;

			CString Tmp = Buffer;
			CString Group;
			BOOL Flag = FALSE;
			if ( Tmp[ Tmp.GetLength() - 1 ] == '*' )
			{
				 Group = Tmp.Left( Tmp.GetLength() - 1 );
				 Flag = TRUE;
			}
			else
				Group = Tmp;

			if ((p = strtok(NULL, " ,\t\r\n")) == NULL) 
				return;

			int Days = atoi( p );
			
			CString Archive;
			if ((p = strtok(NULL, " ,\t\r\n")) != NULL) {
				Archive = p;
			}
			CExpire * Exp = new CExpire( Group , Days , Flag , Archive );
			
			// Add this to the list
			m_ExpireList->Add( Exp );
		} // Endwhile
		// File will be closed automatically?
	} 
	// If we get here, we're swimming in cream so just set the
	// flag to true
	m_Ok = TRUE;
}

CExpireList::~CExpireList()
{                         
	// loop through the list deleting all the elements
	if ( m_ExpireList != NULL )
	{
		for( int i = 0 ; i < m_ExpireList->GetSize() ; i++ )
		{
			delete (CExpire *)m_ExpireList->GetAt( i );
		}
		delete m_ExpireList;
	}
	// All done I think;
}
                                  
BOOL CExpireList::GetExpireTime( const char * Target , CTimeSpan &Span , CString& Archive )
{
	if( m_OverRide != -1 )
	{
		// We are overriding all the expiry stuff so we don't
		// need to do anything else 
		Span = CTimeSpan(m_OverRide , 0 , 0 , 0 );
		Archive = "";
		return TRUE;		
 	}
	// Ok, here we've got to do some sort of reading and writing
	CTimeSpan Best;
	int Match = 0;
	
	for( int i = 0 ; i < m_ExpireList->GetSize() ; i++ )
	{
		CExpire * Exp = (CExpire *)m_ExpireList->GetAt( i );
		int Days = Exp->GetExpire();
		CString Group = Exp->GetGroup();
		BOOL Flag = Exp->GetFlag();
		int TestLength = Group.GetLength();		
		
		if ((TestLength > Match) && (Flag)) 
		{
			if (strnicmp(Target, Group, TestLength) == 0) {
				Best = CTimeSpan( Days , 0 , 0 , 0 );
				Archive = Exp->GetArchive();
				Match = TestLength;
			}
		}
		else {
			if (stricmp(Target, Group) == 0) {
				Best = CTimeSpan( Days , 0 , 0 , 0  );
				Archive = Exp->GetArchive();
				Match = TestLength;
			}
		}
	}

	if (Match != 0) 
	{
	 	Span = Best;
	 	return TRUE;
	}
	else
	{
		Span = CTimeSpan( -1 , 0 , 0 , 0 );
		return FALSE;
	}
	
}
   
