/******************************************
*  Function:    Clock
*  Purpose:     Provides an on-screen clock
*               (Screen Saver)
*  Parameters:  hourmode12  :=  .T. for 12 Hour Am/Pm mode (default)
*                               .F. for 24 Military time
*               dispstring  :=  Character string to display under clock
*                               (default is "Press Any Key to Continue...")
*  Returns:     NIL
*  Author:      Donald T. Olsen (c) copyright 1993
*               96 OSS/OSOAC
*               Dyess AFB, TX 79607
*               Phone: (915)-696-2588 .or. AV: 461-2588
*               Internet:
*                    96wglglmpc@strathost.stratcom.af.mil
*                    dolsen@dylan.af.mil
*                    dolsen@donatello.af.mil
*  Date:        24 March 1993
*  Clipper:     Ver 5.01a
*  Compile:     RTLINK clock /l /m /n
*  Cost:        Free (of course), I only ask that you not sell it for
*               your own profit.  If you change/enhance/rewrite it, do
*               not redistribute your version.  Give me credit for the
*               effort put into it.  (Integrity)
*  Notes:       Additional notes can be found at the end of this program
*               (That is my workspace.)
*
*/

#include "SET.CH"
#include "SETCURS.CH"

//                 Characters           Array Pos:       Used In Numbers:
STATIC    char  :=  { " Ŀ  ", ;     //     1            1
                      "    ", ;     //     2            1
                      "", ;     //     3            1
                      "Ŀ", ;     //     4            2,3,5,6,7,8,9,0
                      "    ", ;     //     5            2,3,4,5,7,9
                      ";", ;     //     6            2
                      "    ", ;     //     7            2,5,6
                      ";", ;     //     8            2,3,5,6,8,9,0
                      "  ͵", ;     //     9            3
                      "   ", ;     //    10            4,6,8,9,0
                      "͵", ;     //    11            4,9
                      "͸", ;     //    12            5
                      "͵", ;     //    13            8
                      "͸", ;     //    14            6
                      "    ", ;     //    15            4,7
                      "   ", ;     //    16            4
                      "     "  }     //    17            Blank (12-Hour mode)

// cross-reference of chars array to construct the #s           Array Pos
STATIC    ref  :=  { { 17, 17, 17, 17, 17 }, ;     //  Blank        1
                     {  4, 10, 10, 10,  8 }, ;     //  Zero         2
                     {  1,  2,  2,  2,  3 }, ;     //  One          3
                     {  4,  5,  6,  7,  8 }, ;     //  Two          4
                     {  4,  5,  9,  5,  8 }, ;     //  Three        5
                     { 16, 10, 11,  5, 15 }, ;     //  Four         6
                     {  4,  7, 12,  5,  8 }, ;     //  Five         7
                     {  4,  7, 14, 10,  8 }, ;     //  Six          8
                     {  4,  5,  5,  5, 15 }, ;     //  Seven        9
                     {  4, 10, 13, 10,  8 }, ;     //  Eight       10
                     {  4, 10, 11,  5,  8 }  }     //  Nine        11
// and the others
STATIC  clktop := 9, clkleft := 1, clkdir := 1

// clktop   := Starting row postion of clock when first called
// clkleft  := Starting column position (1 while clock moves up)
//                                     (32 while clock moves down)
// clkdir   := Starting direction of clock movement 1=Up, -1=Down

FUNCTION Clock( hourmode12, dispstring )
LOCAL inscreen, inrow, incol, incursor, incolor, inscore, nhour, moved, ;
      am, ht, ho, mt, mo, st, so, line, outtahere, ddate
inscreen   := SAVESCREEN( 0, 0, MAXROW(), MAXCOL() )
inrow      := ROW()
incol      := COL()
incursor   := SETCURSOR( SC_NONE )
incolor    := SETCOLOR("BG+/N")
inscore    := SET(_SET_SCOREBOARD, .F.)
moved      := .F.
outtahere  := .F.
hourmode12 := IF(hourmode12 = NIL, .T., hourmode12)
dispstring := IF(dispstring = NIL, "Press Any Key to Continue...", dispstring)
IF VALTYPE(hourmode12) + VALTYPE(dispstring) != "LC"
  _Err_Msg("Programmer Error - Invalid Parameter type passed to Clock")
  RETURN( NIL )
ENDIF

// TIME() is in format "HH:MM:SS"  or  "12:56:23"

// Clear the screen and Display the unchangables (dots and dispstring)
DispUnchangables( dispstring )

// Loop till a key is pressed, I'm outtahere.
WHILE !outtahere
  ddate := DATE()
  DispDate( ddate )

  // Loop till the date changes
  WHILE ddate = DATE() .and. !outtahere
    am  := ( VAL(SUBSTR(TIME(), 1, 2) ) < 12 )
    IF hourmode12
      DispAmPm( am )
	ENDIF

    // Loop till the Am/Pm changes
    WHILE am = ( VAL(SUBSTR(TIME(), 1, 2) ) < 12 ) .and. !outtahere
	  nhour := VAL(SUBSTR(TIME(), 1, 2))
      // nhour will be a string after the IF statement
	  IF hourmode12
        // In 12 hour mode must subtract 12 from any time 1 pm and after
        // Also Midnight to one o'clock is a 12 not a 0
        nhour := PADL( ALLTRIM( STR( IF(nhour>12, nhour-12, IF(nhour = 0, 12, nhour)), 2 )), 2, "0")
        ht  := IF( SUBSTR(nhour, 1, 1) = "0", 1, ;
              VAL( SUBSTR(nhour, 1, 1) ) + 2 )
	  ELSE
		nhour := PADL( ALLTRIM( STR( nhour, 2 ) ), 2, "0")
        ht  := VAL( SUBSTR(nhour, 1, 1) ) + 2
	  ENDIF
	  ho	:= VAL( SUBSTR(nhour, 2, 1) ) + 2
      // 0 in the Update_um function is translated into the screen column
      // at which the hours, minutes and seconds will be displayed
	  Update_um( 0, ht, ho )

      // Loop till the hour changes
      WHILE ( nhour = SUBSTR(TIME(), 1, 2) .or. ;
             IF(VAL(nhour)<12,VAL(nhour)+12,VAL(nhour)-12) = ;
			 VAL(SUBSTR(TIME(), 1, 2)) ) ;
			 .and. !outtahere

        mt    := VAL( SUBSTR(TIME(), 4, 1) ) + 2
		mo	  := VAL( SUBSTR(TIME(), 5, 1) ) + 2
		Update_um( 1, mt, mo )

        // Loop till the minute changes
        WHILE ALLTRIM(STR(mt-2,1))+ALLTRIM(STR(mo-2,1)) = ;
              SUBSTR(TIME(), 4, 2) .and. !outtahere
          st    := VAL( SUBSTR(TIME(), 7, 1) ) + 2
          so    := VAL( SUBSTR(TIME(), 8, 1) ) + 2
          // IF seconds/ones reaches 0 move clock
          IF so - 2 = 0 .and. !moved
            MoveClock()
            // Moved indicates whether the clock has been moved during
            // this 10 second cycle.  Without this flag the clock might
            // move multiple times while the seconds are being updated
            // with 0 in the ones position.  This way it will only move once.
            moved := .T.
            // If clock is at the top or bottom of the screen, flip sides
            IF clktop = -1 .or. clktop = MAXROW() - 6
              Flip_um()
              // Redisplay the date and AmPm (if in that mode)
              Dispdate(ddate)
              IF hourmode12
                DispAmPm(am)
              ENDIF
            ENDIF
          ENDIF
          // If seconds/ones is 5 set moved off
          IF so - 2 = 5
            moved := .F.
          ENDIF
          Update_um( 2, st, so )
		  outtahere := ( INKEY() <> 0 )
		ENDDO
	  ENDDO
	ENDDO
  ENDDO
ENDDO
// Reset the environment, you may need other resets
RESTSCREEN( 0, 0, MAXROW(), MAXCOL(), inscreen )
SETPOS( inrow, incol )
SETCURSOR( incursor )
SETCOLOR( incolor )
SET(_SET_SCOREBOARD, inscore)
RETURN( NIL )


/******************************************
*  Function:    Update_um
*  Purpose:     Update any one of the Hour,
*               Minute or Seconds Display
*  Parameters:  Which ( a 0 for hours )
*                     ( a 1 for minutes )
*                     ( a 2 for seconds )
*               Tens/Ones - array ref of
*                  which number to display
*  Called:      Anytime there is a change to
*               the Hours, Minutes or Seconds
*/
STATIC FUNCTION Update_um( which, tens, ones )
LOCAL line, leftpos
// Starting postions of numbers are 14 columns apart so...
leftpos := clkleft+(which*14)
FOR line = 1 to 5
  @ clktop+line, leftpos SAY char[ ref[ tens, line ] ] + " " + ;
                            char[ ref[ ones, line ] ] COLOR "BG+/N"
NEXT
RETURN( NIL )

/******************************************
*  Function:    MoveClock
*  Purpose:     Moves clock up or down screen
*               and sets new clktop reference
*/
STATIC FUNCTION MoveClock()
SCROLL(0, 0, MAXROW(), MAXCOL(), clkdir)
clktop := clktop - clkdir
RETURN( NIL )

/******************************************
*  Function:    DispUnchangables
*  Purpose:     Displays dots and message these never need to be
*               displayed again, except of course when the clock
*               is called again.
*/
STATIC FUNCTION DispUnchangables( dispstring )
SCROLL()
@ clktop + 2, clkleft + 12 SAY "             " COLOR "W+*/N"
@ clktop + 4, clkleft + 12 SAY "             " COLOR "W+*/N"
@ clktop + 6, 0 SAY PADC(dispstring, 80) COLOR "W+/R"
RETURN( NIL )

/******************************************
*  Function:    DispAmPm
*  Purpose:     Displays AM/PM at the
*               correct position
*  Parameters:  Logical .T. for AM
*                       .F. for PM
*  Called:      On initial start-up and when clock is fliped side-to-side
*               and when Am/Pm changes
*/
FUNCTION DispAmPm( am )
LOCAL incolor := SETCOLOR("GR+/N")
IF am
  @ clktop + 2, clkleft + 40 SAY "Ĵ  "
ELSE
  SETCOLOR("B+/N")
  @ clktop + 2, clkleft + 40 SAY "  "
ENDIF
@ clktop + 1, clkleft + 40 SAY "Ŀ ¿"
SETCOLOR(incolor)
RETURN( NIL )

/******************************************
*  FUNCTION:    DispDate( indate )
*  Purpose:     Converts a date string to a character ( Friday  13 Sep 1991 )
*  Parameters:  Date to be converted
*  Called:      On initial start-up and when clock is fliped side-to-side
*               and when date changes
*/
FUNCTION DispDate( indate )
LOCAL datestring, centerstr, displeft, displine
datestring := CDOW( indate ) + "  " + LTRIM( STR( DAY( indate ), 0 ) ) + " " +;
             CMONTH( indate ) + " " + LTRIM( STR( YEAR( indate ), 0 ) )
centerstr  := IF(clkdir = 1, 61, 16)
displine   := IF(clkdir = 1, 4, 3)
displeft   := INT( centerstr - LEN(datestring)/2 )
@ clktop+displine, displeft-14 SAY SPACE(28)
@ clktop+displine, displeft SAY datestring COLOR "G+/N"
@ clktop+displine+1, displeft-15 SAY SPACE(30)
@ clktop+displine+1, displeft-1 SAY REPLICATE( CHR(196), LEN(datestring)+2 ) ;
                                COLOR "G+/N"
RETURN( NIL )

/******************************************
*  FUNCTION:    Flip_um
*  Purpose:     Flips clock and date on screen
*  Parameters:  None
*  Returns:     Nothing
*  Notes:       After this function is called a call to functions DispDate() and
*               DispAmPm() must be made in order to completely redraw the screen.
*  Called:      When clock reaches either the top of the screen or the bottom
*/
FUNCTION Flip_um()
LOCAL clockscr := SAVESCREEN(clktop, clkleft, clktop+5, clkleft+39)
clkleft := IF(clkleft = 1, 32, 1)
clkdir  := IF(clkdir = -1, 1, -1)
SCROLL(clktop, 0, clktop+5, MAXCOL())
RESTSCREEN(clktop, clkleft, clktop+5, clkleft+39, clockscr)
RETURN( NIL )

/******************************************
*  FUNCTION:    _Err_Msg
*  Purpose:     A generic function to provide on screen notice of an error
*  Parameters:  Message to display, and optional Color logical to indicate
*               if message should be displayed in color (default ISCOLOR() )
*  Returns:     NIL
*
*/
FUNCTION _Err_Msg(l1, color)
LOCAL width,lcol,rcol,icolor,origscrn
color    := IF(color=NIL, ISCOLOR(), color)
IF VALTYPE(l1) + VALTYPE(color) != "CL"
  _Err_Msg("Programmer Error - Invalid parameter passed to _Err_Msg()")
  RETURN( NIL )
ENDIF
l1       := ALLTRIM( l1 )
width    := IF( LEN(l1) < 33, 33, LEN(l1) )
lcol     := ROUND( (80-width)/2, 0 ) - 3
rcol     := lcol + width + 5
icolor   := SETCOLOR()
origscrn := SAVESCREEN( MAXROW()-4, lcol, MAXROW(), rcol+1 )
IF color
  SETCOLOR( "W/R" )
ENDIF
// Shadow( MAXROW()-4, lcol, MAXROW()-1, rcol)
@ MAXROW()-4, lcol, MAXROW()-1, rcol BOX "͸Գ "
IF color
  SETCOLOR( "W+/R" )
ENDIF
@ MAXROW()-4, 35 SAY "<<ERROR>>"
@ MAXROW()-3, ROUND( (80-LEN(l1))/2,0 ) SAY l1
@ MAXROW()-2, 24 SAY "** Press any key to continue **"
TONE(70,4)
TONE(50,4)
INKEY(0)
RESTSCREEN( MAXROW()-4, lcol, MAXROW(), rcol+1, origscrn )
SETCOLOR( icolor )
RETURN( NIL )


/*****************************   Notes   *****************************

HOUR ref from clkleft    :=  0
MINUTES ref from clkleft := 14
SECONDS ref from clkleft := 28

AM/PM ref from clkleft   := 40

clkleft will either be 1   (clock moving up)
                    or 32  (clock moving down)

clkdir will either be 1    (clock moving up)
                   or -1   (clock moving down)

Keyline
             1 1           2 2           4                   6
 1           3 5           7 9           1                   1
  Ŀ   Ŀ   Ŀ Ŀ   Ŀ Ŀ Ŀ ¿
                                 Ĵ  
      ;   ͸ ͸   ;   ͵
                                      Wednesday  31 September 1993
  ;   ; ;   ; ;      


                1               3           4 4             6           7
                6               2           4 6             0           2
                                 Ŀ   Ŀ   Ŀ Ŀ   Ŀ Ŀ Ŀ ¿
                                                                Ĵ  
  Wednesday  31 September 1993       ;   ͸ ͸   ;   ͵
                                
                                 ;   ; ;   ; ;



Here is what all of the numbers look like.

͸
  Ŀ    Ŀ  Ŀ       Ŀ  Ŀ  Ŀ  Ŀ  Ŀ  Ŀ 
                                                        
       ;    ͵  ͵  ͸  ͸        ͵  ͵      
                                                         
   ;  ;        ;  ;        ;  ;  ; 
;

*/

