//------------------------------------------------------------------------------
//
// 	FILE: CB.C
//
//	Freeware, not for resale. Copyright 1994 William Querry. All rights reserved.
//
//		1.01  01-16-94  Removed text "Bank Balance" (that was there in error)
//                    on the "========" line in reconcile area.
//          01-16-94  Added output header: Program name, version, run date
//
//		Program for us that rarely get around to balancing our checkbook and
//		all of a sudden find that our balance and the bank's don't match for
//		some reason. Nothing fancy here, but it works.
//
//		Reads text file (CHK_BOOK.TXT) of checkbook entries formatted as:
//    	1. Check:
//    		[O]C,number,date,amount[,who]
//    	2. Check (voided):
//    		V,number[,date]
//    	3. ATM withdrawal:
//    		[O]A,date,amount[,where]
//    	4. Deposit:
//    		[O]D,date,amount[,what]
//			5. Service Charge:
//    		S,date,amount[,comment]
//			6. Interest:
//    		I,date,amount[,comment]
//			7. Manual deposit entry for balancing checkbook:
//				+,date,amount[,comment]
//			8. Manual withdrawal entry for balancing checkbook:
//				-,date,amount[,comment]
//
//			[O] outstanding entry (not yet cleared bank)
//       [,who] who check was written to (up to 80 chars)
//			[,where] where ATM is located (up to 80 chars)
//			[,what] what the deposit represents (up to 80 chars)
//
//------------------------------------------------------------------------------
#include <dos.h>                       // MSC 5.xx standard lib. function proto
#include <stdio.h>                     // MSC 5.xx standard lib. function proto
#include <stdlib.h>                    // MSC 5.xx standard lib. function proto
#include <string.h>                    // MSC 5.xx standard lib. function proto

#include "\lib\esiutil5\culproto.h"    // ESI Utility lib vers. 5.0 protos

#define DEBUG_ON 0                     // set to enable debugging printfs

#define MAX_TRANS 3000                 // max. number transactions permitted

int main(int argc, char *argv[]);
void amt_to_bfr(long amt,char *amt_bfr);
int display_trans(void);
int read_trans(void);

extern int fgets_stripped(char *bfr,int size,FILE *fs,int *len);
extern void MOV_MEM(void *sourceptr,void *destptr,int movbytes);

struct transactions_str
{
	int cleared;                        // set if check/deposit has cleared
	int voided;                         // set if check was voided

	int type;                           // transaction type
	unsigned int date;                  // transaction date
	long amt;                           // transaction amount

	int no;                             // check number
	char *desc;                         // optional who/what/where
};
struct transactions_str trans[MAX_TRANS];

//
// Transaction types
//
#define CHECK 'C'                      // withdrawal
#define ATM 'A'                        // withdrawal ATM or counter check
#define SERVICE_CHARGE 'S'             // withdrawal
#define DEPOSIT 'D'                    // deposit
#define INTEREST 'I'                   // deposit
#define MANUAL_PLUS '+'                // manual override deposit
#define MANUAL_MINUS '-'               // manual override withdrawal
#define VOIDED 'V'                     // voided check
#define OUTSTANDING 'O'                // outstanding transaction

//
// exit error codes
//
#define SUCCESS 0
#define NO_FILE 1
#define BAD_RECORD 2
#define BAD_TYPE 3

int trecs;                             // total number of records in struct


int main(int argc, char *argv[])
{
   int ecode;
   struct dosdate_t date;
#if DEBUG_ON
	int x;
#endif

#if DEBUG_ON
   x = sizeof(struct transactions_str);
   x = sizeof(trans);
#endif

   argv = argv;                        // currently not used

   _dos_getdate(&date);

   printf("CB 1.01 (quick and dirty checkbook reconciler). Run on: %d/%d/%d\n\n",date.month,date.day,date.year-1900);

   if (argc >= 2)                      // wants some help since no parms used
   {
   	printf("Freeware, not for resale. Copyright 1994 William Querry. All rights reserved.\n\n");
   	printf("Reads a checkbook file (CHK_BOOK.TXT) and displays a checkbook ledger.\n");
   	printf("Use command line re-direction if you want to make and print a file.\n");
   	printf("Look at source code header comments and sample CHK_BOOK.TXT for file format.\n");
   	return(0);
   }
   else
   {
   	if ((ecode = read_trans()) == SUCCESS)
   		ecode = display_trans();

		if (ecode != SUCCESS)
			beep();

		return(ecode);
	}
}                                      // end main()


//------------------------------------------------------------------------------
//
//---void amt_to_bfr(long amt,char *bfr)
//
//------------------------------------------------------------------------------
void amt_to_bfr(long amt,char *amt_bfr)
{
	int x;
   char bfr[256],vbfr[20];

	ltoa(amt,bfr,10);
	x = strlen(bfr);
	switch (x)
	{
		case 0:
			strcpy(vbfr,"0.00");
			break;
		case 1:
			strcpy(vbfr,"0.0");
			strcat(vbfr,bfr);
			break;
		case 2:
			strcpy(vbfr,"0.");
			strcat(vbfr,bfr);
			break;
		default:
			memset(vbfr,0,9);
			strleft(bfr,vbfr,x-2,10);
			strcat(vbfr,".");
			strright(bfr,&vbfr[x-1],x-2,8);
			break;
	}
	amt_bfr[8] = 0;
	strrfld(amt_bfr,vbfr,8);
}                                      // end amt_to_bfr()

//------------------------------------------------------------------------------
//
//---int display_trans(void)
//
// Display format:
//
//          1         2         3         4         5         6         7
//01234567890123456789012345678901234567890123456789012345678901234567890123456789
//  Number    Date       Check Amt   Deposit Amt   Balance   Comments
//  12345     11/11/95   12345.67                  12345.67  fffffff
//O 12345     11/11/95   12345.67                  12345.67
//  12345     VOID VOID VOID VOID VOID VOID VOID   12345.67
//  Deposit   11/11/95               12345.67      12345.67
//O Deposit   11/11/95               12345.67      12345.67
//  ATM       11/11/95   12345.67                  12345.67
//O ATM       11/11/95   12345.67                  12345.67
//  SC        11/11/95   12345.67                  12345.67
//  Interest  11/11/95               12345.67      12345.67
//  Man. Sub  11/11/95   12345.67                  12345.67
//  Man. Add  11/11/95               12345.67      12345.67
//--------------------------------------------------------------------------------
//Total Checks   : 1234  $123456.12
//Total ATMs     : 1234  $123456.12
//Total Deposits : 1234  $123456.12
//Total Interests: 1234  $123456.12
//Total S-Charges: 1234  $123456.12
//Total Man. Adds: 1234  $123456.12
//Total Man. Subs: 1234  $123456.12
//Total Voids    : 1234
//--------------------------------------------------------------------------------
//Bank Balance                 $ ______.__
//1234 Checks/ATM Outstanding -$ 123456.12
//1234 Deposits Outstanding   +$ 123456.12
//                             ===========
//         Reconciled Balance $ _______.__
//01234567890123456789012345678901234567890123456789012345678901234567890123456789
//          1         2         3         4         5         6         7
//------------------------------------------------------------------------------
int display_trans(void)
{
	static char voided_txt[] = "VOID VOID VOID VOID VOID VOID VOID";
	static char col_txt[] = "  Number    Date       Check Amt   Deposit Amt   Balance   Comments\n";
	static char div_txt[] = "--------------------------------------------------------------------------------\n";

	static char no_atm_txt[] = "ATM";
	static char no_dep_txt[] = "Deposit";
	static char no_sc_txt[] = "SC";
	static char no_int_txt[] = "Interest";
	static char no_add_txt[] = "Man. Add";
	static char no_sub_txt[] = "Man. Sub";


   int ecode;
   int yr,mon,day;
   int i;
   int tot_deposits,      tot_checks,     out_deposits,     out_checks;
   long tot_deposits_amt, tot_checks_amt, out_deposits_amt, out_checks_amt;
   int tot_int_no,   tot_sc_no,  tot_atm_no;
   long tot_int_amt, tot_sc_amt, tot_atm_amt;
   int tot_add_no,   tot_sub_no;
   long tot_add_amt, tot_sub_amt;
   long run_balance;
   int tot_void_checks;
   char date_bfr[20],no_bfr[20],amt_bfr[20],bal_bfr[20],dbfr[80];

#if DEBUG_ON
	char bfr[256];

	printf("\nValues in transaction array:\n");
	for (i = 0; i < trecs; i++)
	{
		unpacdat(trans[i].date,&yr,&mon,&day);
		strdate(bfr,yr,mon,day,3);
		printf("%d\t TYPE=%c\t DATE=%s\t ",i,trans[i].type,bfr);

		amt_to_bfr(trans[i].amt,bfr);
		printf("AMT=%s (%ld)\t ",bfr,trans[i].amt);

		if (trans[i].desc)
			printf("DESC=%s",trans[i].desc);
		printf("\n");
	}
#endif

   // initialize totals
   tot_deposits     = out_deposits     = tot_checks     = out_checks = 0;
   tot_deposits_amt = out_deposits_amt = tot_checks_amt = out_checks_amt = 0L;
   tot_int_no  = tot_sc_no  = tot_atm_no  = tot_add_no  = tot_sub_no = 0;
   tot_int_amt = tot_sc_amt = tot_atm_amt = tot_add_amt = tot_sub_amt = 0L;
   tot_void_checks = 0;
   run_balance = 0L;
   ecode = SUCCESS;                    // assume successful run

	printf(col_txt);
	for (i = 0; i < trecs; i++)
	{
		itoa(trans[i].no,no_bfr,10);

		unpacdat(trans[i].date,&yr,&mon,&day);
		strdate(date_bfr,yr,mon,day,3);

		amt_to_bfr(trans[i].amt,amt_bfr);

		memset(dbfr,' ',sizeof(dbfr));
		dbfr[sizeof(dbfr)-1] = 0;

		if (trans[i].cleared)
			dbfr[0] = ' ';
		else
			dbfr[0] = 'O';

		switch (trans[i].type)
		{
			case CHECK:                   // transaction = check
				tot_checks++;
				tot_checks_amt += trans[i].amt;
				run_balance -= trans[i].amt;
				if (!trans[i].cleared)
				{
					out_checks++;
					out_checks_amt += trans[i].amt;
				}
				MOV_MEM(no_bfr,   &dbfr[2],  strlen(no_bfr));
				MOV_MEM(date_bfr, &dbfr[12], strlen(date_bfr));
				MOV_MEM(amt_bfr,  &dbfr[23], strlen(amt_bfr));
				break;

			case VOIDED:                  // transaction = voided check
				tot_void_checks++;
				MOV_MEM(no_bfr,   &dbfr[2],  strlen(no_bfr));
				MOV_MEM(voided_txt,&dbfr[12],strlen(voided_txt));
				break;

			case ATM:                     // transaction = ATM/counter check
				tot_atm_no++;
				tot_atm_amt += trans[i].amt;
				run_balance -= trans[i].amt;
				if (!trans[i].cleared)
				{
					out_checks++;
					out_checks_amt += trans[i].amt;
				}
				MOV_MEM(no_atm_txt, &dbfr[2],  strlen(no_atm_txt));
				MOV_MEM(date_bfr,   &dbfr[12], strlen(date_bfr));
				MOV_MEM(amt_bfr,    &dbfr[23], strlen(amt_bfr));
				break;

			case SERVICE_CHARGE:          // transaction = service charge
				tot_sc_no++;
				tot_sc_amt += trans[i].amt;
				run_balance -= trans[i].amt;
				MOV_MEM(no_sc_txt, &dbfr[2],  strlen(no_sc_txt));
				MOV_MEM(date_bfr,  &dbfr[12], strlen(date_bfr));
				MOV_MEM(amt_bfr,   &dbfr[23], strlen(amt_bfr));
				break;

			case DEPOSIT:                 // transaction = deposit
				tot_deposits++;
				tot_deposits_amt += trans[i].amt;
				run_balance += trans[i].amt;
				if (!trans[i].cleared)
				{
					out_deposits++;
					out_deposits_amt += trans[i].amt;
				}
				MOV_MEM(no_dep_txt, &dbfr[2],strlen(no_dep_txt));
				MOV_MEM(date_bfr, &dbfr[12], strlen(date_bfr));
				MOV_MEM(amt_bfr,  &dbfr[35], strlen(amt_bfr));
				break;

			case INTEREST:                // transaction = interest
				tot_int_no++;
				tot_int_amt += trans[i].amt;
				run_balance += trans[i].amt;
				MOV_MEM(no_int_txt, &dbfr[2],strlen(no_int_txt));
				MOV_MEM(date_bfr, &dbfr[12], strlen(date_bfr));
				MOV_MEM(amt_bfr,  &dbfr[35], strlen(amt_bfr));
				break;

			case MANUAL_PLUS:             // transaction = balance checkbook, add
				tot_add_no++;
				tot_add_amt += trans[i].amt;
				run_balance += trans[i].amt;
				MOV_MEM(no_add_txt, &dbfr[2],strlen(no_add_txt));
				MOV_MEM(date_bfr, &dbfr[12], strlen(date_bfr));
				MOV_MEM(amt_bfr,  &dbfr[35], strlen(amt_bfr));
				break;

			case MANUAL_MINUS:            // transaction = balance checkbook, minus
				tot_sub_no++;
				tot_sub_amt += trans[i].amt;
				run_balance -= trans[i].amt;
				MOV_MEM(no_sub_txt, &dbfr[2],strlen(no_sub_txt));
				MOV_MEM(date_bfr, &dbfr[12], strlen(date_bfr));
				MOV_MEM(amt_bfr,  &dbfr[23], strlen(amt_bfr));
				break;

			default:
				printf("** ERROR ** Invalid type code=%d\n",trans[i].type);
				return(BAD_TYPE);
				break;
		}

		amt_to_bfr(run_balance,bal_bfr);
		MOV_MEM(bal_bfr,  &dbfr[49], strlen(amt_bfr));
		if (trans[i].desc)
			MOV_MEM(trans[i].desc,&dbfr[59],strlen(trans[i].desc));
		printf("%s\n",dbfr);
	}

	printf(div_txt);
	amt_to_bfr(tot_checks_amt,bal_bfr);
	printf("Total Checks   : %d  $%s\n",tot_checks,bal_bfr);

	amt_to_bfr(tot_atm_amt,bal_bfr);
	printf("Total ATMs     : %d  $%s\n",tot_atm_no,bal_bfr);

	amt_to_bfr(tot_deposits_amt,bal_bfr);
	printf("Total Deposits : %d  $%s\n",tot_deposits,bal_bfr);

	amt_to_bfr(tot_int_amt,bal_bfr);
	printf("Total Interests: %d  $%s\n",tot_int_no,bal_bfr);

	amt_to_bfr(tot_sc_amt,bal_bfr);
	printf("Total S-Charges: %d  $%s\n",tot_sc_no,bal_bfr);

	amt_to_bfr(tot_add_amt,bal_bfr);
	printf("Total Man. Adds: %d  $%s\n",tot_add_no,bal_bfr);

	amt_to_bfr(tot_sub_amt,bal_bfr);
	printf("Total Man. Subs: %d  $%s\n",tot_sub_no,bal_bfr);

	printf("Total Voids    : %d\n",tot_void_checks);

	printf(div_txt);
	printf("Bank Balance                 $ ______.__\n");
	amt_to_bfr(out_checks_amt,bal_bfr);
	printf("Checks/ATM Outstanding      -$  %s   (%d)\n",bal_bfr,out_checks);

	amt_to_bfr(out_deposits_amt,bal_bfr);
   printf("Deposits Outstanding        +$  %s   (%d)\n",bal_bfr,out_deposits);

	printf("                             ===========\n");
	printf("         Reconciled Balance $ _______.__\n");
	return(ecode);                      // return error code
}                                      // end display_trans()


//------------------------------------------------------------------------------
//
//---int read_trans(void)
//
//------------------------------------------------------------------------------
int read_trans(void)
{
	static char split_txt[] = "\t,";

	FILE *fstuff;
   struct transactions_str *tptr;
   char bfr[256],*ptr,*tcode,*field,copy_bfr[256];
   int ecode,line,len;
   int yr,mon,day;

   ecode = SUCCESS;                    // assume successful run

	// open transactions file
	if ((fstuff = fopen("CHK_BOOK.TXT","r")) == NULL)
	{
		printf("** ERROR: Check book file (CHK_BOOK.TXT) is missing!\n\n");
		ecode = NO_FILE;
		goto getout;
	}

	// read and parse all transaction records
	for (trecs = 0, line = 0, tptr = &trans[0]; (trecs < MAX_TRANS); line++)
	{
	   if (fgets_stripped(bfr,255,fstuff,&len) == NULL)
	   	break;                        // no more records

#if DEBUG_ON
		printf("Line %d\t: %s \n",line+1,bfr);
#endif

		strcpy(copy_bfr,bfr);            // used for reporting errors

		ptr = &bfr[0];                   // point to transaction type code

		tptr = &trans[trecs];              // point to next record posn

		if (*ptr == OUTSTANDING)
		{
			tptr->cleared = 0;            // set transaction has not cleared
			ptr++;                        // point to transaction type code
		}
		else
			tptr->cleared = 1;            // assume transaction has cleared

		tptr->voided = 0;                // assume no voided check
		tptr->date = 0;                  // set invalid date
		tptr->amt = 0;                   // set no transaction amount
		tptr->desc = 0;                  // set no transaction desc

		tcode = strtok(ptr,split_txt);   // isolate transaction code in record

		switch (tptr->type = *tcode)     // save/determine transaction type
		{
			case 0:                       // blank line - ignore
			case ';':                     // comment line - ignore
				break;

			case CHECK:                   // transaction = check
				// parse check number
				if ((field = strtok(NULL,split_txt)) == 0)
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Check record missing required check number field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->no = atoi(field);              // save check number

				// parse check date
				if ((field = strtok(NULL,split_txt)) == 0)
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Check record missing required date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				if (icondate(field,&yr,&mon,&day))
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Check record has invalid date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->date = packdate(yr,mon,day);   // convert date to DOS format

				// parse check amount (whole dollars portion)
				if ((field = strtok(NULL,".")) == 0) // get whole dollar amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Check record missing or invalid amount field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt = atol(field) * 100;       // save whole dollars
				if (tptr->amt < 0)                   // Check must be positive!
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Check record amount field must be positive:\n\t%s\n",line,copy_bfr);
					goto getout;
				}

				// parse check amount (cents portion)
				if ((field = strtok(NULL,split_txt)) == 0)     // get cents amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Check record missing cents amount:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt += atol(field);            // add in cents

				// parse check description
				if (field = strtok(NULL,split_txt))  // get Check description
				{                                    // (if there is one)
					tptr->desc = malloc(strlen(field) + 1);
					strcpy(tptr->desc,field);
				}
				trecs++;                   // point to next record
				break;

			case VOIDED:                  // transaction = voided check
				// parse check number
				if ((field = strtok(NULL,split_txt)) == 0)
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Check record missing required check number field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->no = atoi(field);              // save check number
				tptr->voided = 1;                    // set this check voided
				trecs++;                   // point to next record
				break;

			case ATM:                     // transaction = ATM/counter check
				// parse ATM date
				if ((field = strtok(NULL,split_txt)) == 0)
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): ATM record missing required date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				if (icondate(field,&yr,&mon,&day))
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): ATM record has invalid date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->date = packdate(yr,mon,day);   // convert date to DOS format

				// parse ATM amount (whole dollars portion)
				if ((field = strtok(NULL,".")) == 0) // get whole dollar amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): ATM record missing or invalid amount field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt = atol(field) * 100;       // save whole dollars
				if (tptr->amt < 0)                   // ATM must be positive!
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): ATM record amount field must be positive:\n\t%s\n",line,copy_bfr);
					goto getout;
				}

				// parse ATM amount (cents portion)
				if ((field = strtok(NULL,split_txt)) == 0)     // get cents amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): ATM record missing cents amount:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt += atol(field);            // add in cents

				// parse ATM description
				if (field = strtok(NULL,split_txt))  // get ATM description
				{                                    // (if there is one)
					tptr->desc = malloc(strlen(field) + 1);
					strcpy(tptr->desc,field);
				}
				trecs++;                   // point to next record
				break;

			case SERVICE_CHARGE:          // transaction = service charge
				// parse SC date
				if ((field = strtok(NULL,split_txt)) == 0)
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): SC record missing required date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				if (icondate(field,&yr,&mon,&day))
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): SC record has invalid date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->date = packdate(yr,mon,day);   // convert date to DOS format

				// parse SC amount (whole dollars portion)
				if ((field = strtok(NULL,".")) == 0) // get whole dollar amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): SC record missing or invalid amount field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt = atol(field) * 100;       // save whole dollars
				if (tptr->amt < 0)                   // SC must be positive!
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): SC record amount field must be positive:\n\t%s\n",line,copy_bfr);
					goto getout;
				}

				// parse SC amount (cents portion)
				if ((field = strtok(NULL,split_txt)) == 0)     // get cents amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): SC record missing cents amount:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt += atol(field);            // add in cents

				// parse SC description
				if (field = strtok(NULL,split_txt))  // get SC description
				{                                    // (if there is one)
					tptr->desc = malloc(strlen(field) + 1);
					strcpy(tptr->desc,field);
				}
				trecs++;                   // point to next record
				break;

			case DEPOSIT:                 // transaction = deposit
				// parse Deposit date
				if ((field = strtok(NULL,split_txt)) == 0)
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Deposit record missing required date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				if (icondate(field,&yr,&mon,&day))
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Deposit record has invalid date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->date = packdate(yr,mon,day);   // convert date to DOS format

				// parse deposit amount (whole dollars portion)
				if ((field = strtok(NULL,".")) == 0) // get whole dollar amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Deposit record missing or invalid amount field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt = atol(field) * 100;       // get whole dollars
				if (tptr->amt < 0)                   // deposit must be positive!
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Deposit record amount field must be positive:\n\t%s\n",line,copy_bfr);
					goto getout;
				}

				// parse deposit amount (cents portion)
				if ((field = strtok(NULL,split_txt)) == 0)     // get cents amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Deposit record missing cents amount:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt += atol(field);            // add in cents

				// parse deposit description
				if (field = strtok(NULL,split_txt))  // get deposit description
				{                                    // (if there is one)
					tptr->desc = malloc(strlen(field) + 1);
					strcpy(tptr->desc,field);
				}
				trecs++;                   // point to next record
				break;

			case INTEREST:                // transaction = interest
				// parse Interest date
				if ((field = strtok(NULL,split_txt)) == 0)
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Interest record missing required date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				if (icondate(field,&yr,&mon,&day))
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Interest record has invalid date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->date = packdate(yr,mon,day);   // convert date to DOS format

				// parse Interest amount (whole dollars portion)
				if ((field = strtok(NULL,".")) == 0) // get whole dollar amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Interest record missing or invalid amount field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt = atol(field) * 100;       // save whole dollars
				if (tptr->amt < 0)                   // Interest must be positive!
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Interest record amount field must be positive:\n\t%s\n",line,copy_bfr);
					goto getout;
				}

				// parse Interest amount (cents portion)
				if ((field = strtok(NULL,split_txt)) == 0)     // get cents amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Interest record missing cents amount:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt += atol(field);            // add in cents

				// parse Interest description
				if (field = strtok(NULL,split_txt))  // get Interest desc
				{                                    // (if there is one)
					tptr->desc = malloc(strlen(field) + 1);
					strcpy(tptr->desc,field);
				}
				trecs++;                   // point to next record
				break;

			case MANUAL_PLUS:             // transaction = balance checkbook, add
				// parse Manual Add date
				if ((field = strtok(NULL,split_txt)) == 0)
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Manual Add record missing required date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				if (icondate(field,&yr,&mon,&day))
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Manual Add record has invalid date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->date = packdate(yr,mon,day);   // convert date to DOS format

				// parse Manual Add amount (whole dollars portion)
				if ((field = strtok(NULL,".")) == 0) // get whole dollar amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Manual Add record missing or invalid amount field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt = atol(field) * 100;       // save whole dollars
				if (tptr->amt < 0)                   // Manual Add must be positive!
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Manual Add record amount field must be positive:\n\t%s\n",line,copy_bfr);
					goto getout;
				}

				// parse Manual Add amount (cents portion)
				if ((field = strtok(NULL,split_txt)) == 0)     // get cents amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Manual Add record missing cents amount:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt += atol(field);            // add in cents

				// parse Manual Add description
				if (field = strtok(NULL,split_txt))  // get Manual Add description
				{                                    // (if there is one)
					tptr->desc = malloc(strlen(field) + 1);
					strcpy(tptr->desc,field);
				}

				trecs++;                   // point to next record
				break;

			case MANUAL_MINUS:            // transaction = balance checkbook, minus
				// parse Manual Sub date
				if ((field = strtok(NULL,split_txt)) == 0)
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Manual Sub record missing required date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				if (icondate(field,&yr,&mon,&day))
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Manual Sub record has invalid date field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->date = packdate(yr,mon,day);   // convert date to DOS format

				// parse Manual Sub amount (whole dollars portion)
				if ((field = strtok(NULL,".")) == 0) // get whole dollar amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Manual Sub record missing or invalid amount field:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt = atol(field) * 100;       // save whole dollars
				if (tptr->amt < 0)                   // Manual Sub must be positive!
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Manual Sub record amount field must be positive:\n\t%s\n",line,copy_bfr);
					goto getout;
				}

				// parse Manual Sub amount (cents portion)
				if ((field = strtok(NULL,split_txt)) == 0)     // get cents amount
				{
					ecode = BAD_RECORD;
					printf("** ERROR (line %d): Manual Sub record missing cents amount:\n\t%s\n",line,copy_bfr);
					goto getout;
				}
				tptr->amt += atol(field);            // add in cents

				// parse Manual Sub description
				if (field = strtok(NULL,split_txt))  // get Manual Sub description
				{                                    // (if there is one)
					tptr->desc = malloc(strlen(field) + 1);
					strcpy(tptr->desc,field);
				}

				trecs++;                   // point to next record
				break;
		}
	}

getout:
	if (fstuff)                         // close transactions file if open
		fclose(fstuff);

	return(ecode);                      // return error code
}                                      // end read_trans()




                           //-----  END CB.C  -----//
