/************************************************************
  GATEVIEW.C -- system info viewer, uses installable VxD RINGO
  by Alex Shmidt, November 1993
************************************************************/

#define WINVER   0x30a    /* for Win 3.1 SDK to get compatible with 3.0 */
#include <windows.h>
#include "386.h"
#include "ringo.h"
#include "gateview.h"
#include "gatehlp.h"

#define WM_RINGO 0      /* RINGO's message WM_USER + WM_RINGO */

long  FAR PASCAL WndProc (HWND,WORD,WORD,LONG);
void  PrintDescriptor (LPSTR, LPSEGDESCRIPTOR, int, BYTE);
BOOL  ParsePageEntry (LPPAGE, LPPGMAP);
BYTE  ParseDescriptor (LPSEGDESCRIPTOR, LPSEGDESCMAP);
LPSTR PrintPageType (LPPGMAP, LPSTR);
BOOL  PageDir (LPSTR, LPPAGE, int);
DWORD PageTable (LPSTR, LPPAGE, int, int);
void  PageDirListBox (HWND, LPSTR, UINT);
void  GDTListBox (HWND, LPSTR, UINT);
void  LDTListBox (HWND, LPSTR, UINT);
void  IDTListBox (HWND, LPSTR, UINT);
void  TSSListBox (HWND, LPSTR);
void  CRDRListBox (HWND, LPSTR);
void  VxDListBox (HWND, LPSTR);
void  VMListBox  (HWND, LPSTR);
void  LoadControlInfo (WORD,LPMSGSTRUC,LPSTR);
DWORD GetPageTableVM (DWORD);
void  PhysListBox (HWND, LPSTR, DWORD);

SYSTEMSTRUCT   sysstruct;
UINT     uSelector1, uSelector2;

int PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpszCmdParam,
                        int nCmdShow)
{
static char szAppName[] = "GateView";
HWND        hwnd;
MSG         msg;
WNDCLASS    wndclass;

  if (!hPrevInstance)
  {
    wndclass.style          = CS_HREDRAW | CS_VREDRAW;
    wndclass.lpfnWndProc    = (WNDPROC) WndProc;
    wndclass.cbClsExtra     = 0;
    wndclass.cbWndExtra     = DLGWINDOWEXTRA;
    wndclass.hInstance      = hInstance;
    wndclass.hIcon          = LoadIcon (NULL, IDI_APPLICATION);
    wndclass.hCursor        = LoadCursor (NULL,IDC_ARROW);
    wndclass.hbrBackground  = COLOR_WINDOW + 1;
    wndclass.lpszMenuName   = NULL;
    wndclass.lpszClassName  = szAppName;

    RegisterClass (&wndclass);
  }

  uSelector1 = uSelector2 = SELECTOROF((LPWORD)&uSelector1);
  uSelector1 = AllocSelector (uSelector1);
  uSelector2 = AllocSelector (uSelector2);

  hwnd = CreateDialog (hInstance, szAppName, 0, 0L);
  ShowWindow (hwnd, nCmdShow);

  while (GetMessage (&msg,NULL,0,0))
  {
    TranslateMessage (&msg);
    DispatchMessage (&msg);
  }

  return   msg.wParam;
}

long FAR PASCAL WndProc (HWND hwnd,WORD msg,WORD wParam,LONG lParam)
{
char              buffer[128];
LPPAGE            ptabptr;
int               tab1[] = {20,60,90,105,115,125,135,145};
int               tab2[] = {75,125,190,255};
char              control[80];
int               index;
static UINT       lbindex;
int               pdir;
static WPARAM     lbowner;
static WORD       stopvm;
DWORD             phys;

  switch (msg)
  {
    case WM_CREATE:
      GateRegisterWindow (hwnd);
      break;

    case WM_SHOWWINDOW:     /* prepare the VMM event list box */
      SendMessage (GetDlgItem (hwnd, IDLB_EVENT), LB_RESETCONTENT, 0, 0L);
      SendMessage (GetDlgItem (hwnd, IDLB_EVENT), LB_SETTABSTOPS, 4, (DWORD)(LPINT)tab2);
      break;

    case WM_COMMAND:
      switch (wParam)
      {
        case IDPB_PDIR:             /* page directory */
          lbowner = wParam;
          sysstruct.VMCount = 10;
          GetSys (&sysstruct);      /* call RINGO */

          wsprintf (buffer,
                   "Page Directory    Physical = %08lX   Linear = %08lX",
                   sysstruct.SysCr.CR3R, sysstruct.SysCr.PDIRL);
          SetWindowText (GetDlgItem (hwnd,IDS_HDR), buffer);

          PageDirListBox (GetDlgItem (hwnd, IDLB_PDIR), buffer, 0);
          break;

        case IDLB_PDIR:            /* page table */
          if (HIWORD(lParam) == LBN_DBLCLK & lbowner == IDPB_PDIR)
          {
            lbowner = wParam;
            lbindex = LOWORD(SendMessage (LOWORD(lParam), LB_GETCURSEL, 0, 0L));
            if (lbindex < 2)
              break;
            pdir = LOWORD(SendMessage (LOWORD(lParam), LB_GETITEMDATA, lbindex, 0L));

            SetSelectorBase (uSelector1, sysstruct.SysCr.PDIRL);
            SetSelectorLimit (uSelector1, P_SIZE);
            ptabptr = (LPPAGE)MAKELP(uSelector1, 0) + pdir;

            SetSelectorBase (uSelector1,
                     MapPhysToLinear (*ptabptr & PAGE_FRAME_ADS_MASK, P_SIZE));
            SetSelectorLimit (uSelector1, P_SIZE);
            ptabptr = (LPPAGE)MAKELP(uSelector1, 0);
            wsprintf (buffer,"Page Table  %04X",pdir);
            SetWindowText (GetDlgItem (hwnd,IDS_HDR), buffer);
            SendMessage (LOWORD(lParam), LB_RESETCONTENT, 0, 0L);
            SendMessage (LOWORD(lParam), LB_SETTABSTOPS, 8, (DWORD)(LPINT)tab1);
            wsprintf (buffer,"Entry\tLin Range\tPhys #\t  Attributes  and  VMM Type");
            SendMessage (LOWORD(lParam), LB_ADDSTRING, 0, (LONG)(LPSTR)buffer);
            wsprintf (buffer,"");
            SendMessage (LOWORD(lParam), LB_ADDSTRING, 0, (LONG)(LPSTR)buffer);
            for (index = 0; index < PTABLE_SIZE; index++,ptabptr++)
              if ((phys = PageTable (buffer, ptabptr, index, pdir)) != 0xFFFFFFFF)
              {
                lbindex = LOWORD(SendMessage (LOWORD(lParam), LB_ADDSTRING, 0, (LONG)(LPSTR)buffer));
                if (lbindex != LB_ERR && lbindex != LB_ERRSPACE)
                  SendMessage (LOWORD(lParam), LB_SETITEMDATA, lbindex, phys);
              }
            SendMessage (LOWORD(lParam), LB_SETCURSEL, 0, 0L);
          }
          else if (HIWORD(lParam) == LBN_DBLCLK & lbowner == IDLB_PDIR)
          {
            lbowner = 0;
            lbindex = LOWORD(SendMessage (LOWORD(lParam), LB_GETCURSEL, 0, 0L));
            if (lbindex < 2)
              break;
            phys = SendMessage (LOWORD(lParam), LB_GETITEMDATA, lbindex, 0L);
            wsprintf (buffer,"Virtual Addresses for the Physical Address %08lX",
                                                  phys);
            SetWindowText (GetDlgItem (hwnd,IDS_HDR), buffer);
            PhysListBox (LOWORD(lParam), buffer, phys);
          }
          break;

        case IDPB_GDT:              /* GDT */
          lbowner = wParam;
          GetSys (&sysstruct);

          wsprintf (buffer, "GDTBase = %08lX   Limit = %04X",
                   sysstruct.SysGdt.GDTRB, sysstruct.SysGdt.GDTRL);
          SetWindowText (GetDlgItem (hwnd,IDS_HDR), buffer);

          GDTListBox (GetDlgItem (hwnd, IDLB_PDIR), buffer, 0);
          break;

        case IDPB_LDT:              /* LDT */
          lbowner = wParam;
          GetSys (&sysstruct);

          wsprintf (buffer, "LDT Sel = %04X  Base = %08lX  Limit = %04X",
                   sysstruct.SysLdt.LDTRS, sysstruct.SysLdt.LDTB,
                   sysstruct.SysLdt.LDTSL);
          SetWindowText (GetDlgItem (hwnd,IDS_HDR), buffer);

          LDTListBox (GetDlgItem (hwnd, IDLB_PDIR), buffer, 0);
          break;

        case IDPB_IDT:              /* IDT */
          lbowner = wParam;
          GetSys (&sysstruct);

          wsprintf (buffer, "IDTBase = %08lX   Limit = %04X",
                   sysstruct.SysIdt.IDTRB, sysstruct.SysIdt.IDTRL);
          SetWindowText (GetDlgItem (hwnd,IDS_HDR), buffer);

          IDTListBox (GetDlgItem (hwnd, IDLB_PDIR), buffer, 0);
          break;

        case IDPB_TASK:             /* TSS */
          lbowner = wParam;
          GetSys (&sysstruct);

          wsprintf (buffer, "TR = %04X   TSSBase = %08lX  TSSLimit = %04X",
                   sysstruct.SysTss.TREG,sysstruct.SysTss.TSSB,
                   sysstruct.SysTss.TSSL);
          SetWindowText (GetDlgItem (hwnd,IDS_HDR), buffer);

          TSSListBox (GetDlgItem (hwnd, IDLB_PDIR), buffer);
          break;

        case IDPB_CRDR:             /* Control and Debug registers */
          lbowner = wParam;
          GetSys (&sysstruct);
          wsprintf (buffer, "Control and Debug registers");
          SetWindowText (GetDlgItem (hwnd,IDS_HDR), buffer);
          CRDRListBox (GetDlgItem (hwnd, IDLB_PDIR), buffer);
          break;

        case IDPB_CLEAR:            /* clear the VMM Event list box */
          SendMessage (GetDlgItem (hwnd, IDLB_EVENT), LB_RESETCONTENT, 0, 0L);
          SendMessage (GetDlgItem (hwnd, IDLB_EVENT), LB_SETTABSTOPS, 4, (DWORD)(LPINT)tab2);
          break;

        case IDPB_STOPVM:           /* 'DOS' button */
          stopvm ^= 1;
          NoDosBox (stopvm ? Create_VM : 0xffff);
          wsprintf (buffer,"%s",stopvm?(LPSTR)"dos":(LPSTR)"DOS");
          SetWindowText (LOWORD(lParam), buffer);
          break;

        case IDPB_VXD:              /* VxD list */
          lbowner = wParam;
          GetSys (&sysstruct);
          wsprintf (buffer,"VxD Root (VMM DDB pointer)  =   %08lX",
                    sysstruct.VxDRoot);
          SetWindowText (GetDlgItem (hwnd,IDS_HDR), buffer);
          VxDListBox (GetDlgItem (hwnd, IDLB_PDIR), buffer);
          break;

        case IDPB_VM:               /* VM list */
          lbowner = wParam;
          sysstruct.VMCount = 10;
          wsprintf (buffer,"VMM version %d.%d,  build revision %08ld,   Number of running VMs = %04d",
                    HIBYTE(sysstruct.VMMVer), LOBYTE(sysstruct.VMMVer),
                    sysstruct.VMMRev, (int)GetSys (&sysstruct));
          SetWindowText (GetDlgItem (hwnd,IDS_HDR), buffer);
          VMListBox (GetDlgItem (hwnd, IDLB_PDIR), buffer);
          break;

        case IDPB_HELP:
          MessageBox (hwnd,helpmsg,"GateView",MB_OK);
          break;

        case IDPB_EXIT:
          DestroyWindow (hwnd);
          return 0;
      }
      break;

    /* RINGO sends us this message every time his Control Procedure gets hit */
    case WM_USER + WM_RINGO:
      LoadControlInfo (wParam,(LPMSGSTRUC)lParam,control);
      wsprintf (buffer,"%s\tVM %08lX%s",(LPSTR)control_msg[wParam],
               ((LPMSGSTRUC)lParam)->msg_VM,(LPSTR)control);
      SendMessage (GetDlgItem (hwnd, IDLB_EVENT), LB_ADDSTRING, 0, (LONG)(LP[VXD_FIRST]
        push    [esi].DDB_Next
        pop     [eax].DDB_Next
        mov     [esi].DDB_Next,eax
        ret
InsertRingoDDB   endp

RemoveRingoDDB  proc
        push    gs
        assume  gs:_DATA
        mov     gs,gs:[gdt_datasel]
        assume  gs:_GATESEG
        mov     esi,gs:[VXD_FIRST]
        push    gs:[Ringo_DDB].DDB_Next
        pop     [esi].DDB_Next
        pop     gs
        ret
RemoveRingoDDB  endp

; this is our Control Procedure -- every VMM message parks here
; we translate the message into the structure GATEVIEW can interpret
; then we schedule Sys VM event. In there we call PostMessage function
; in the nested execution frame and post WM_USER + WM_RINGO message
RingoControlProc        proc
        push    gs
        call    GetRingoGdtDataSel
        cmp     gs:[HWND],0
        jz      short @f
        pushad
; build WM_RINGO message out of System_Control message
        mov     ecx, gs:[msgwrite]
        mov     gs:[ecx][msgqueue].MSG_Msg,eax
r,"\tFlags %08lX",msg->msg_Param1);
      break;

    case Power_Event:
      wsprintf (buffer,"\tEvent Msg %08lX",msg->msg_Param3);
      break;

    default:
      lstrcpy (buffer,"");
      break;
  }
}

/* dumps the content of Control and Debug registers */
void CRDRListBox (HWND lbhwnd, LPSTR buffer)
{
int      tabs[] = {25,75,100};

  SendMessage (lbhwnd, LB_RESETCONTENT, 0, 0L);
  SendMessage (lbhwnd, LB_SETTABSTOPS, 3, (DWORD)(LPINT)tabs);

  wsprintf (buffer,"CR0 = \t%08lX\tDR0 = \t%08lX", sysstruct.SysCr.CR0R,
                                                   sysstruct.SysDr.DR0R);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"CR2 = \t%08lX\tDR1 = \t%08lX", sysstruct.SysCr.CR2R,
                                                   sysstruct.SysDr.DR1R);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"CR3 = \t%08lX\tDR2 = \t%08lX", sysstruct.SysCr.CR3R,
                                                   sysstruct.SysDr.DR2R);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"\t\tDR3 = \t%08lX", sysstruct.SysDr.DR3R);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"\t\tDR6 = \t%08lX", sysstruct.SysDr.DR6R);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"\t\tDR7 = \t%08lX", sysstruct.SysDr.DR7R);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
}

/* print out the list of currently running Virtual Machines */
/* their status flags, and documented parameters */
void VMListBox (HWND lbhwnd, LPSTR buffer)
{
int      tabs[] = {40,80,120,160};
LPVM     vmptr;
int      vmcnt;
int      fl;

  SendMessage (lbhwnd, LB_RESETCONTENT, 0, 0L);
  SendMessage (lbhwnd, LB_SETTABSTOPS, 4, (DWORD)(LPINT)tabs);

  wsprintf (buffer,"VM Handle\tStatus\tHigh Lin\tClient Ptr\tVM ID");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);

  for (vmcnt=0;vmcnt<(int)sysstruct.VMCount;vmcnt++)
  {
    /* map VM control block */
    SetSelectorBase (uSelector1, sysstruct.VMHndl[vmcnt]);
    SetSelectorLimit (uSelector1, sizeof(VM));
    vmptr = (LPVM)MAKELP(uSelector1,0);

    wsprintf (buffer,"%08lX\t%08lX\t%08lX\t%08lX\t%08lX",
              sysstruct.VMHndl[vmcnt],
              vmptr->CB_VM_Status,
              vmptr->CB_High_Linear,
              vmptr->CB_Client_Pointer,
              vmptr->CB_VMID);
    SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);

    /* dump CB_ flags */
    for (fl = 0; fl < sizeof(CB_Flags)/sizeof(char *); fl++)
      if (vmptr->CB_VM_Status & (1<<fl))
      {
        wsprintf (buffer,"\t%s", (LPSTR)CB_Flags[fl]);
        SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
      }
  }
}

/* print out the list of installed Virtual Devices */
/* and everything we can find in the Device Descriptor Block about them */
void VxDListBox (HWND lbhwnd, LPSTR buffer)
{
int      tabs[] = {40,60,80,120,160,200,240,280};
LPDDB    ddbptr;
char     namebuf[9];

  SendMessage (lbhwnd, LB_RESETCONTENT, 0, 0L);
  SendMessage (lbhwnd, LB_SETTABSTOPS, 8, (DWORD)(LPINT)tabs);
  wsprintf (buffer,"VxD Name\tID\tVer\tControl\tV86 API\tPM API\tSvc Tab\tSvc Count");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);

  SetSelectorBase (uSelector1, sysstruct.VxDRoot);
  SetSelectorLimit (uSelector1, sizeof(DDB));
  ddbptr = (LPDDB)MAKELP(uSelector1, 0);

  /* walk the VxD list */
  while (1)
  {
    lstrcpyn (namebuf,ddbptr->DDB_Name,sizeof(namebuf));
    namebuf[8] = '\0';
    wsprintf (buffer,"%s\t%04X\t%d.%d\t%08lX\t%08lX\t%08lX\t%08lX\t%08lX",
             (LPSTR)namebuf,
             ddbptr->DDB_Req_Device_Number,
             ddbptr->DDB_Dev_Major_Version,
             ddbptr->DDB_Dev_Minor_Version,
             ddbptr->DDB_Control_Proc,
             ddbptr->DDB_V86_API_Proc,
             ddbptr->DDB_PM_API_Proc,
             ddbptr->DDB_Service_Table_Ptr,
             ddbptr->DDB_Service_Table_Size);
    SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);

    if (ddbptr->DDB_Next == 0)
      break;
    SetSelectorBase (uSelector1, ddbptr->DDB_Next);
    SetSelectorLimit (uSelector1, sizeof(DDB));
    ddbptr = (LPDDB)MAKELP(uSelector1, 0);
  }
}

BOOL ParsePageEntry (LPPAGE lppage, LPPGMAP pgmap)
{
  pgmap->pgatr  = (WORD)*lppage;
  pgmap->pgtype = pgmap->pgatr & PG_TYPE;
  pgmap->pgphys = *lppage & PAGE_FRAME_ADS_MASK;

  return (*lppage != NULL);
}

BOOL PageDir (LPSTR buff, LPPAGE lppage, int pdir)
{
PGMAP pgmap;
DWORD rangestart = (DWORD)pdir;
DWORD rangeend   = (DWORD)pdir;
char  pgtype[40];
char  vmpgtab[10];
DWORD vmid;

  if (ParsePageEntry (lppage, &pgmap))
  {
    rangestart <<= 22;
    rangeend   = rangestart | (LIN_PAGE | LIN_OFFSET);

    PrintPageType (&pgmap, pgtype);
    if (vmid = GetPageTableVM (rangestart))
      wsprintf (vmpgtab, "VM %d",vmid);
    else
      lstrcpy (vmpgtab,"");

    wsprintf (buff, "%x:\t%08lX\t%08lX\t%08lX - %08lX%s\t%s",
              pdir,
              pgmap.pgphys,
              MapPhysToLinear (pgmap.pgphys, P_SIZE),
              rangestart,
              rangeend,
              (LPSTR)pgtype,
              (LPSTR)vmpgtab);
    return TRUE;
  }
  return FALSE;
}

DWORD PageTable (LPSTR buff, LPPAGE lppage, int ptab, int pdir)
{
PGMAP pgmap;
char  pgtype[40]; /* Present Write Accessed User Dirty Type*/

  if (ParsePageEntry(lppage, &pgmap))
  {
    PrintPageType (&pgmap, pgtype);

    wsprintf (buff, "%X:\t%05lXxxx\t%lX%s",
              ptab,
              (DWORD)pdir << 10 | ptab,
              pgmap.pgphys >> 12,
              (LPSTR)pgtype);
    return pgmap.pgphys;
  }
  return 0xFFFFFFFF;
}

LPSTR PrintPageType (LPPGMAP entry, LPSTR pgtype)
{
  lstrcpy (pgtype, (entry->pgatr & P_PRES) ? "\t P" : "\tNP");
  lstrcat (pgtype, (entry->pgatr & P_WRITE) ? "\tW" : "\t ");
  lstrcat (pgtype, (entry->pgatr & P_ACC) ? "\tA" : "\t ");
  lstrcat (pgtype, (entry->pgatr & P_USER) ? "\tU" : "\tS");
  lstrcat (pgtype, (entry->pgatr & P_DIRTY) ? "\tD" : "\t ");

  switch (entry->pgtype)
  {
    case PgT_VM:
      lstrcat (pgtype, "\tVM");
      break;

    case PgT_SYS:
      lstrcat (pgtype, "\tSystem");
      break;

    case PgT_HOOKED:
      lstrcat (pgtype, "\tHooked");
      break;

    case PgT_RESERVED1:
      lstrcat (pgtype, "\tReserved1");
      break;

    case PgT_RESERVED2:
      lstrcat (pgtype, "\tReserved2");
      break;

    case PgT_PRIVATE:
      lstrcat (pgtype, "\tPrivate");
      break;

    case PgT_RELOCK:
      lstrcat (pgtype, "\tRelock");
      break;

    case PgT_INSTANCE:
      lstrcat (pgtype, "\tInstance");
      break;
  }
  return (pgtype);
}

/* Page Directory dump */
void PageDirListBox (HWND lbhwnd, LPSTR buffer, UINT pdir)
{
UINT     entry, lbind;
int      tabs[] = {20,60,100,164,174,184,194,204,214};
LPPAGE   ptabptr;

  SendMessage (lbhwnd, LB_RESETCONTENT, 0, 0L);
  SendMessage (lbhwnd, LB_SETTABSTOPS, 9, (DWORD)(LPINT)tabs);
  wsprintf (buffer,"PTab\tPhysical\tLinear\tRange\t\tFlags and VMM Type\tOwner");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);

  SetSelectorBase (uSelector1, sysstruct.SysCr.PDIRL);
  SetSelectorLimit (uSelector1, P_SIZE);
  ptabptr = (LPPAGE)MAKELP(uSelector1, 0);

  /* walk the Page Directory */
  for (entry = 0; entry < PTABLE_SIZE; entry++,ptabptr++)
    if (PageDir (buffer, ptabptr, entry))
    {
      lbind = LOWORD(SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer));
      if (lbind != LB_ERR && lbind != LB_ERRSPACE)
        SendMessage (lbhwnd, LB_SETITEMDATA, lbind, (DWORD)entry);
    }
  SendMessage (lbhwnd, LB_SETCURSEL, pdir, 0L);
}

BYTE ParseDescriptor (LPSEGDESCRIPTOR descptr, LPSEGDESCMAP descmap)
{
  descmap->base = MAKEBASE(descptr->Seg_Desc_Base_0_15,
                           descptr->Seg_Desc_Base_16_23,
                           descptr->Seg_Desc_Base_24_31);
  descmap->limit = MAKELIMIT(descptr->Seg_Desc_Limit_0_15,
                             descptr->Seg_Desc_Gran_Byte);
  descmap->type = descptr->Seg_Desc_Access_Right;
  descmap->dpl = (descptr->Seg_Desc_Access_Right & D_DPL3) >> 5;
  descmap->gran = descptr->Seg_Desc_Gran_Byte;
  return (descmap->type);
}

void PrintDescriptor (LPSTR buff, LPSEGDESCRIPTOR GDTSel, int index, BYTE tab)
{
#define  TASKGATE 1
#define  CALLGATE 2
#define  TRAPGATE 4
#define  INTGATE  8

char     dsctype[20];
char     attribute[40];
LPSEGDESCRIPTOR ptr = GDTSel + index;
BYTE     system = 0;
SEGDESCMAP descmap;

  if (ParseDescriptor (ptr, &descmap) & D_PRES)
    lstrcpy (attribute, "\tP");
  else
    lstrcpy (attribute, "\tNP");

  if (descmap.gran & D_GRAN_PAGE)
  {
    descmap.limit *= P_SIZE;
    descmap.limit += P_SIZE - 1;
  }

  if (descmap.type & D_SEG)                    /* segment descriptor */
  {
    if (descmap.type & D_CODE)                /* code segment */
    {
      lstrcpy (dsctype, "Code");

      if (descmap.type & D_C)
        lstrcat (attribute, "\tC");   /* conforming */
      else
        lstrcat (attribute, "\t");    /* conforming */

      if (descmap.type & D_RX)
        lstrcat (attribute, "\tRE");    /* readable, executeable */
      else
        lstrcat (attribute, "\tE");     /* executeable */
    }
    else                      /* data segment */
    {
      lstrcpy (dsctype, "Data");

      if (descmap.type & D_E)
        lstrcat (attribute, "\tEX");
      else
        lstrcat (attribute, "\t");

      if (descmap.type & D_W)
        lstrcat (attribute, "\tRW");
      else
        lstrcat (attribute, "\tR");
    }
    if (descmap.type & D_ACCESSED)
      lstrcat (attribute, "\tA");

    if (descmap.gran & D_DEF32)
      lstrcat (dsctype, "32");
    else
      lstrcat (dsctype, "16");
  }
  else                                   /* system descriptor */
  {
    switch (descmap.type & DESC_MASK)
    {
      case DESC_TYPE_286_TSS:
        lstrcpy (dsctype, "TSS16");
        break;
      case DESC_TYPE_LDT:
        lstrcpy (dsctype, "LDT");
        break;
      case DESC_TYPE_BUSY_286_TSS:
        lstrcpy (dsctype, "TSS16");
        lstrcat (attribute, "\tB");
        break;
      case DESC_TYPE_286_CALL_GATE:
        lstrcpy (dsctype, "CallG16");
        system |= CALLGATE;
        break;
      case DESC_TYPE_TASK_GATE:
        lstrcpy (dsctype, "TaskG");
        system |= TASKGATE;
        break;
      case DESC_TYPE_286_INT_GATE:
        lstrcpy (dsctype, "IntG16");
        system |= INTGATE;
        break;
      case DESC_TYPE_286_TRAP_GATE:
        lstrcpy (dsctype, "TrapG16");
        system |= TRAPGATE;
        break;
      case DESC_TYPE_386_TSS:
        lstrcpy (dsctype, "TSS32");
        break;
      case DESC_TYPE_BUSY_386_TSS:
        lstrcpy (dsctype, "TSS32");
        lstrcat (attribute, "\tB");
        break;
      case DESC_TYPE_386_CALL_GATE:
        lstrcpy (dsctype, "CallG32");
        system |= CALLGATE;
        break;
      case DESC_TYPE_386_INT_GATE:
        lstrcpy (dsctype, "IntG32");
        system |= INTGATE;
        break;
      case DESC_TYPE_386_TRAP_GATE:
        lstrcpy (dsctype, "TrapG32");
        system |= TRAPGATE;
        break;

      default:
        lstrcpy (dsctype, "Rsrved");
        break;
    }
  }

  if (!system)
    wsprintf (buff,"%x\t%s\t%08lX\t%08lX\t%x%s",
              (index * 8) | descmap.dpl | tab,
              (LPSTR)dsctype,
              descmap.base, descmap.limit, descmap.dpl,
              (LPSTR)attribute);
  else if (system & TASKGATE)
    wsprintf (buff,"%x\t%s\t%X\t\t%x%s",
              (index * 8) | descmap.dpl | tab,
              (LPSTR)dsctype,
              ((LPCALLGATEDESCRPT)ptr)->Selector, descmap.dpl,
              (LPSTR)attribute);
  else if (system & CALLGATE)
    wsprintf (buff,"%x\t%s\t%04X\t%08lX\t%x%s",
              (index * 8) | descmap.dpl | tab,
              (LPSTR)dsctype,
              ((LPCALLGATEDESCRPT)ptr)->Selector,
              MAKELONG(((LPCALLGATEDESCRPT)ptr)->Offset_O_15,
                       ((LPCALLGATEDESCRPT)ptr)->Offset_16_31),
              descmap.dpl,
              (LPSTR)attribute);
  else     /* interrupt and trap gates */
    wsprintf (buff,"%x\t%s\t%04X\t%08lX\t%x%s",
              index,
              (LPSTR)dsctype,
              ((LPCALLGATEDESCRPT)ptr)->Selector,
              MAKELONG(((LPCALLGATEDESCRPT)ptr)->Offset_O_15,
                       ((LPCALLGATEDESCRPT)ptr)->Offset_16_31),
              descmap.dpl,
              (LPSTR)attribute);
}

/* dump the Global Descriptor Table */
void GDTListBox (HWND lbhwnd, LPSTR buffer, UINT entry)
{
LPSEGDESCRIPTOR gdtptr;
int            index;
int            tabs[] = {30,60,100,145,170,180,190,205};

  SendMessage (lbhwnd, LB_RESETCONTENT, 0, 0L);
  SendMessage (lbhwnd, LB_SETTABSTOPS, 8, (DWORD)(LPINT)tabs);
  wsprintf (buffer,"Selector\tType\tBase\tLimit\tDPL\tAccess Rights");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  SetSelectorBase (uSelector1, sysstruct.SysGdt.GDTRB);
  SetSelectorLimit (uSelector1, sysstruct.SysGdt.GDTRL);

  gdtptr = (LPSEGDESCRIPTOR)MAKELP(uSelector1, 0);
  /* walk the GDT */
  for (index = 1; index < (int)(sysstruct.SysGdt.GDTRL + 1) / 8; index++)
  {
    PrintDescriptor (buffer, gdtptr, index, 0);
    SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  }
  SendMessage (lbhwnd, LB_SETCURSEL, entry, 0L);
}

/* dumps the Local Descriptor Table */
void LDTListBox (HWND lbhwnd, LPSTR buffer, UINT entry)
{
LPSEGDESCRIPTOR ldtptr;
int            index;
int            tabs[] = {30,60,100,145,160,175,185,200};

  SendMessage (lbhwnd, LB_RESETCONTENT, 0, 0L);
  SendMessage (lbhwnd, LB_SETTABSTOPS, 8, (DWORD)(LPINT)tabs);
  wsprintf (buffer,"Sel\tType\tBase\tLimit\tDPL\tAccess Rights");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);

  SetSelectorBase (uSelector1, sysstruct.SysLdt.LDTB);
  SetSelectorLimit (uSelector1, sysstruct.SysLdt.LDTSL);

  ldtptr = (LPSEGDESCRIPTOR)MAKELP(uSelector1, 0);
  /* walk the LDT */
  for (index = 0; index < (int)(sysstruct.SysLdt.LDTSL + 1) / 8; index++)
  {
    PrintDescriptor (buffer, ldtptr, index, TABLE_MASK);
    SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  }
  SendMessage (lbhwnd, WM_SETREDRAW, TRUE, 0L);
  SendMessage (lbhwnd, LB_SETCURSEL, entry, 0L);
}

/* dumps the Interrupt Descriptor Tanle */
void IDTListBox (HWND lbhwnd, LPSTR buffer, UINT entry)
{
LPSEGDESCRIPTOR gdtptr;
int            index;
int            tabs[] = {30,70,100,150,170,185,195,210};

  SendMessage (lbhwnd, LB_RESETCONTENT, 0, 0L);
  SendMessage (lbhwnd, LB_SETTABSTOPS, 8, (DWORD)(LPINT)tabs);
  wsprintf (buffer,"Sel\tType\tBase\tLimit\tDPL\tAccess Rights");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  SetSelectorBase (uSelector1, sysstruct.SysIdt.IDTRB);
  SetSelectorLimit (uSelector1, sysstruct.SysIdt.IDTRL);

  gdtptr = (LPSEGDESCRIPTOR)MAKELP(uSelector1, 0);
  /* walk the IDT */
  for (index = 0; index < (int)(sysstruct.SysIdt.IDTRL + 1) / 8; index++)
  {
    PrintDescriptor (buffer, gdtptr, index, 0);
    SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  }
  SendMessage (lbhwnd, LB_SETCURSEL, entry, 0L);
}

/* dumps the Task State Segment (never changes in Windows)*/
void TSSListBox (HWND lbhwnd, LPSTR buffer)
{
LPTSS_386 tssptr;
int   tabs[] = {25,50,75,100,125,150,175,200,225,250,275};

  SendMessage (lbhwnd, LB_RESETCONTENT, 0, 0L);
  SendMessage (lbhwnd, LB_SETTABSTOPS, 11, (DWORD)(LPINT)tabs);
  SetSelectorBase (uSelector1, sysstruct.SysTss.TSSB);
  SetSelectorLimit (uSelector1, sysstruct.SysTss.TSSL);

  tssptr = (LPTSS_386)MAKELP(uSelector1, 0);

  wsprintf (buffer,"GS = \t%04X\tFS = \t%04X\tDS = \t%04X\tSS = \t%04X\tCS = \t%04X\tES = \t%04X",
            tssptr->TSS_gs,
            tssptr->TSS_fs,
            tssptr->TSS_ds,
            tssptr->TSS_ss,
            tssptr->TSS_cs,
            tssptr->TSS_es);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"LDT = \t%04X\tCR3 = \t%08lX",
            tssptr->TSS_LDTR,
            tssptr->TSS_cr3);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"EAX = \t%08lX\tEBX = \t%08lX\tECX = \t%08lX\tEDX = \t%08lX",
            tssptr->TSS_eax,
            tssptr->TSS_ebx,
            tssptr->TSS_ecx,
            tssptr->TSS_edx);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"ESI = \t%08lX\tEDI = \t%08lX\tEBP = \t%08lX\tESP = \t%08lX",
            tssptr->TSS_esi,
            tssptr->TSS_edi,
            tssptr->TSS_ebp,
            tssptr->TSS_esp);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"EIP = \t%08lX\tEFl = \t%08lX",
            tssptr->TSS_eip,
            tssptr->TSS_EFlags);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer,"SS0 = %04X : %08lX     SS1 = %04X : %08lX    SS2 = %04X : %08lX",
            tssptr->TSS_ss0, tssptr->TSS_esp0,
            tssptr->TSS_ss1, tssptr->TSS_esp1,
            tssptr->TSS_ss2, tssptr->TSS_esp2);
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
}

BOOL IsPageEntryPhys (LPPAGE lppage, DWORD phys, LPPGMAP pgmap)
{
   ParsePageEntry (lppage, pgmap);
   return (pgmap->pgatr & P_PRES && pgmap->pgphys == phys);
}


void PhysListBox (HWND lbhwnd, LPSTR buffer, DWORD physical)
{
LPPAGE pdirptr, ptabptr;
UINT   pdir, ptab;
UINT   sel1,sel2;
PGMAP  pgmap;
char   pgtype[40];
int    tab[] = {50,80,110,120,130,140,150,170,210};
char   vmpgtab[10];
DWORD  vmid;
DWORD  linear;

  sel1 = sel2 = SELECTOROF((LPWORD)&uSelector1);
  sel1 = AllocSelector (sel1);
  sel2 = AllocSelector (sel2);
  SendMessage (lbhwnd, LB_RESETCONTENT, 0, 0L);
  SendMessage (lbhwnd, LB_SETTABSTOPS, 9, (DWORD)(LPINT)tab);

  SetSelectorBase (sel1, sysstruct.SysCr.PDIRL);
  SetSelectorLimit (sel1, P_SIZE);
  pdirptr = (LPPAGE)MAKELP(sel1, 0);

  wsprintf (buffer, "Linear\tPDir\tPTab\tAttributes and VMM types\tOwner");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  wsprintf (buffer, "");
  SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
  for (pdir = 0; pdir < PTABLE_SIZE; pdir++)
    if (*(pdirptr + pdir))
    {
      SetSelectorBase (sel2, MapPhysToLinear((*(pdirptr + pdir)) & PAGE_FRAME_ADS_MASK, 4096));
      SetSelectorLimit (sel2, P_SIZE);
      ptabptr = (LPPAGE)MAKELP(sel2, 0);
      for (ptab = 0; ptab < PTABLE_SIZE; ptab++)
        if (IsPageEntryPhys (ptabptr + ptab, physical & PAGE_FRAME_ADS_MASK,&pgmap))
        {
          PrintPageType (&pgmap, pgtype);
          linear = MAKELIN(pdir,ptab,0);
          if (vmid = GetPageTableVM (linear & LIN_DIR))
            wsprintf (vmpgtab, "VM %d",vmid);
          else
            lstrcpy (vmpgtab,"");
          wsprintf (buffer, "%08lX\t%04X\t%04X%s\t%s", linear, pdir, ptab,
                                                         (LPSTR)pgtype,
                                                         (LPSTR)vmpgtab);
          SendMessage (lbhwnd, LB_ADDSTRING, 0, (LONG)buffer);
        }
    }
  SendMessage (lbhwnd, LB_SETCURSEL, 0, 0L);
  FreeSelector (sel1);
  FreeSelector (sel2);
}

DWORD GetPageTableVM (DWORD linear)
{
int   vmcnt;
LPVM  vmptr;
UINT  sel;
DWORD vmid = 0;

  sel = SELECTOROF((LPWORD)&uSelector1);
  sel = AllocSelector (sel);
  for (vmcnt=0;vmcnt<(int)sysstruct.VMCount;vmcnt++)
  {
    /* map VM control block */
    SetSelectorBase (sel, sysstruct.VMHndl[vmcnt]);
    SetSelectorLimit (sel, sizeof(VM));
    vmptr = (LPVM)MAKELP(sel,0);

    if (vmptr->CB_High_Linear == linear)
    {
      vmid = vmptr->CB_VMID;
        break;
    }
  }
  FreeSelector (sel);
  return vmid;
}
