/* ------------------------------------------------------------------------ *\
|            |                                                               |
|  WRAPI.C   | Wrapped-API support functions.                                |
|            |                                                               |
|            | Placed in the public domain by:                               |
|            |                                                               |
|            |    Gregory C. Sarafin                                         |
|            |    In-Design                                                  |
|            |    PO Box 336                                                 |
|            |    Cranbury, NJ 08512                                         |
|            |                                                               |
|            |    CIS: 73747,3112                                            |
|            |                                                               |
|            |                                                               |
|            | * * N O T I C E * N O T I C E * N O T I C E * N O T I C E * * |
|            |                                                               |
|            | WRAPI is the original work of Gregory C. Sarafin with         |
|            | contributions by David Karasek.                               |
|            |                                                               |
|            | WRAPI comes with absolutely no warranty.                      |
|            |                                                               |
|            | WRAPI exists in the public domain.                            |
|            |                                                               |
|            | Use WRAPI at your own risk.                                   |
|            |                                                               |
|            | * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * |
|            |                                                               |
|            |                                                               |
|            | The following APIs are currently supported...                 |
|            |                                                               |
|            | _API_MSC       Microsoft 'C' (Large Model)             LIB    |
|            | _API_BC        Borland 'C'   (Large Model)             LIB    |
|            | _API_WCC       Watcom 'C'    (Large Model)             LIB    |
|            | _API_DLL       Generic 'C' DLL                         DLL    |
|            | _API_CLIP4     Clipper Summer '87                      LIB    |
|            | _API_CLIP5     Clipper 5.x  (VMM)                      LIB    |
|            | _API_FPD20     FoxPro 2.0                              PLB    |
|            | _API_FPD25     FoxPro 2.5+                             PLB    |
|            | _API_FPW25     FoxPro 2.5+ for Windows                 FLL    |
|            | _API_VBD       Visual Basic for DOS 1.0+               LIB    |
|            | _API_VBW       Visual Basic for Windows 2.0+           DLL    |
|            |                                                               |
|            |                                                               |
|            | Revison History                                               |
|            |                                                               |
|            |                                                               |
|            | #     Date      Description                                   |
|            | -     ----      -----------                                   |
|            | 1.00  01/15/94  Sanitized for public consumption              |
|            |                                                               |
\* ------------------------------------------------------------------------ */

#include <stddef.h>
#include "wrapi.h"
#include "export.h"
#include "core.h"

#ifdef _FAMILY_DLL
  static HANDLE hLibInst ;
  #ifdef _API_FOX
    extern HANDLE Inst ;
  #endif
#endif

#ifndef _FOX_DEBUG
#define _FOX_DEBUG 0  // If !0, includes code to help with FOX debugging
#endif

/* Str conversion... ------------------------------------------------------ *\

   WRAPI uses the typedef WSTR to handle all string passing between
   the host API and the core functions.  The WSTR typedef can be found
   in WRAPI.H, but it is repeated here for clarity:

   typedef struct  { // WRAPPED-API STRING STRUCTURE
     char   *cp    ; // Actual pointer to the string
     ushort  uiLen ; // Length of string
     ulong   ulHnd ; // Host API handle if applicable
   } WSTR          ;

   This structure has been designed to handle both zero-terminated strings
   such as those passed by C and Clipper as-well-as buffered strings such
   as those passed by Fox and VB.

   The wrapper functions automatically translate host API strings into
   WSTR structures before they are passed to the core.

\* ------------------------------------------------------------------------ */


    
/* ------------------------------------------------------------------------ *\

    FUNCTION: void   szwcpy   Copies a wstr to a zstr

  PARAMETERS: char*  szaTo    Address of zstr (destination)
              WSTR*  wspFr    Address of wstr (source)

      RETURN:                 Nothing

       NOTES: Like strcpy()

\* ------------------------------------------------------------------------ */
    
void szwcpy ( char* szaTo , WSTR* wspFr ) {

  char *cp = wspFr->cp ;

  #if ( ! IsWZTerm )

    ushort uiWLen = wspFr->uiLen ;

  #endif

  if ( cp == NULL )

    *szaTo = '\0' ;

  else {

    #if ( IsWZTerm )

      while ( *szaTo++ = *cp++ ) ;

    #else

      while ( uiWLen-- ) *szaTo++ = *cp++ ;

      *szaTo= '\0' ;

    #endif

    }

  return ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: void   szwncpy  Copies at most n-1 chars of a wstr to a zstr

  PARAMETERS: char*  szaTo    Address of zstr (destination)
              WSTR*  wspFr    Address of wstr (source)
              ushort uiMax    Maximum number of characters to copy

      RETURN:                 Nothing

       NOTES: Unlike the C function strncpy(), szwncpy() will always
              zero terminate the zstring.

\* ------------------------------------------------------------------------ */
    
void szwncpy ( char* szaTo , WSTR* wspFr , ushort uiMax ) {

  char *cp = wspFr->cp ;

  #if ( ! IsWZTerm )

    ushort uiWLen = wspFr->uiLen ;

  #endif

  if ( cp == NULL )

    *szaTo = '\0' ;

  else

  if ( --uiMax <= 0 )

    *szaTo = '\0' ;

  else {

    #if ( IsWZTerm )

      while ( uiMax-- && ( *szaTo++ = *cp++ ) ) ;

    #else

      while ( uiWLen-- && uiMax-- && ( *szaTo++ = *cp++ ) ) ;

    #endif

    *szaTo = '\0' ;

    }

  return ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: void   swzcpy   Copies a zstr to a wstr

  PARAMETERS: WSTR*  wspTo    Address of wstr (destination)
              char*  szaFr    Address of zstr (source)

      RETURN:                 Nothing

       NOTES: Like strcpy()

\* ------------------------------------------------------------------------ */
    
void swzcpy ( WSTR* wspTo , char* szaFr ) {

  char *cp = wspTo->cp ;

  wspTo->uiLen = 0 ;

  if ( szaFr == NULL ) {

    wspTo->cp    = NIL_S ;

    wspTo->uiLen = (ushort) STRLEN ( wspTo->cp ) ;

    }

  else
    
    while ( ( *cp++ = *szaFr++ ) && ++(*wspTo).uiLen ) ;

  return ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: void   swwcpy   Copies a wstr to a wstr

  PARAMETERS: WSTR*  wspTo    Address of wstr (destination)
              WSTR*  wspFr    Address of wstr (source)

      RETURN:                 Nothing

       NOTES: Like strcpy() - ALWAYS zero terminates!

\* ------------------------------------------------------------------------ */
    
void swwcpy ( WSTR* wspTo , WSTR* wspFr ) {

  char *cpTo = wspTo->cp , *cpFr = wspFr->cp ;
  
  ushort uiLen = wspTo->uiLen = wspFr->uiLen ;

  while ( uiLen-- ) *cpTo++ = *cpFr++ ;

  *cpTo = '\0' ;

  return ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: short  szwcmp   Compares a zstr to a wstr

  PARAMETERS: char*  sza1     Address of zstr (string 1)
              WSTR*  wsp2     Address of wstr (string 2)

      RETURN: short  iReturn  Difference between string 1 and string 2

       NOTES: Like strcmp()

\* ------------------------------------------------------------------------ */
    
short szwcmp ( char* sza1 , WSTR* wsp2 ) {

  short iReturn = 0 ;

  #if ( IsWZTerm )

    iReturn = (short) STRCMP ( sza1 , wsp2->cp ) ;

  #else

    uchar   *cp1    = (uchar*) sza1
          , *cp2    = (uchar*) wsp2->cp
          , *cp1Max = (uchar*) ( cp1 + wsp2->uiLen ) ;

    while (    *cp1
            &&  cp1 < cp1Max
            && ! ( iReturn = (short) ( (short) *cp1++ - (short) *cp2++ ) )
          ) ;

    if ( *cp1 && ( cp1 == cp1Max ) )

      iReturn = (short) *cp1 ;

  #endif

  return iReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: short  swzcmp   Compares a wstr to a zstr

  PARAMETERS: WSTR*  wsp1     Address of wstr (string 1)
              zstr*  sza2     Address of zstr (string 2)

      RETURN: short  iReturn  Difference between string 1 and string 2

       NOTES: Like strcmp()

\* ------------------------------------------------------------------------ */
    
short swzcmp ( WSTR* wsp1 , char* sza2 ) {

  short iReturn = 0 ;

  #if ( IsWZTerm )

    iReturn = (short) STRCMP ( wsp1->cp , sza2 ) ;

  #else

    uchar   *cp1    = (uchar*) wsp1->cp
          , *cp2    = (uchar*) sza2
          , *cp1Max = (uchar*) ( cp1 + wsp1->uiLen ) ;

    while (    cp1 < cp1Max
            && ! ( iReturn = (short) ( (short) (*cp1++) - (short) (*cp2++) ) )
          ) ;

  #endif

  return iReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: short  swwcmp   Compares a wstr to a wstr

  PARAMETERS: WSTR*  wsp1     Address of wstr (string 1)
              WSTR*  wsp2     Address of wstr (string 2)

      RETURN: short  iReturn  Difference between string 1 and string 2

       NOTES: Like strcmp()

\* ------------------------------------------------------------------------ */
    
short swwcmp ( WSTR* wsp1 , WSTR* wsp2 ) {

  short iReturn = 0 ;

  #if ( IsWZTerm )

    iReturn = (short) STRCMP ( wsp1->cp , wsp2->cp ) ;

  #else

    uchar   *cp1    = (uchar*) wsp1->cp
          , *cp2    = (uchar*) wsp2->cp
          , *cp1Max = (uchar*) ( cp1 + wsp1->uiLen )
          , *cp2Max = (uchar*) ( cp2 + wsp2->uiLen ) ;
          
    while (    cp1 < cp1Max
            && cp2 < cp2Max
            && ! ( iReturn = (short) ( (short) *cp1++ - (short) *cp2++ ) )
          ) ;

    if ( cp1 < cp1Max && ( cp2 == cp2Max ) )

      iReturn = (short) *cp1 ;

  #endif

  return iReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: WSTR*  ztow         Wraps a statically allocated
                                  wstr around a zstr

  PARAMETERS: char*  sz2BWrapped  Address of zstr to be wrapped

      RETURN: WSTR  &wsReturn     Address of static wstr wrapper

       NOTES: This is used by the SET_ERR_LOC macro to pass a zstr to
              the _wspXXErrLoc() function which requires a wstr.

\* ------------------------------------------------------------------------ */
    
WSTR* ztow ( char*  sz2BWrapped ) {

  static WSTR wsReturn ;

  wsReturn.cp    = sz2BWrapped ;
  wsReturn.uiLen = (ushort) STRLEN ( sz2BWrapped ) ;

  return &wsReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: char*  ctosz        Wraps a statically allocated
                                  zstr around a char

  PARAMETERS: char   c2BWrapped   Character to be wrapped

      RETURN: char*  szReturn     Address of static zstr wrapper

       NOTES: This is used by the SEND_C macro to send single char
              values back to APIs that only recognize strings

\* ------------------------------------------------------------------------ */
    
char* ctosz   ( char c2BWrapped ) {

  static char szaReturn[2] ;

  szaReturn[0] = c2BWrapped ;

  return szaReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: void   strfill       Fills a block of memory with the contents
                                  of a zstr, advancing the original pointer
                                  to the end of the filled string.

  PARAMETERS: char** cpp          Address of address of memory to fill

      RETURN:                     Nothing

       NOTES: I never liked the C standard way of catenating strings in
              a contiguos block of memory.  This works much better.
              

\* ------------------------------------------------------------------------ */
    
void strfill ( char **cpp , char *sz ) {

  while ( *(*cpp)++ = *sz++ ) ; *(*cpp)-- ;

  return ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: bool   streat       Eats a substring off the front of a
                                  master string based on a particular
                                  token.  The position of the master
                                  string is updated and the substring
                                  is stored in the passed memory location.

  PARAMETERS: char** cpp          Address of address of string to eat
              char*  sz           Adress of substring buffer (holds result)
              char   cToken       Token character
              ushort uiMaxLen     Maximum length of the substring (this is
                                  ususally passed as a sizeof() value)

      RETURN: bool   bReturn      Boolean value indicating success

       NOTES: A slightly different way of peeling substrings off the
              front of a master string.

\* ------------------------------------------------------------------------ */
    
bool streat ( char **cpp , char *sz , char cToken , ushort uiMaxLen ) {

  bool bReturn ; char c ;

  while ( --uiMaxLen && ( c = *(*cpp)++ ) ) {

    if ( (*sz++ = c ) == cToken )

      break ;

    }

  if ( bReturn = (bool) ( uiMaxLen && c && ( c == cToken ) ) )

    *--sz ;     // move back one position to token

  *sz = '\0' ;  // zero terminate

  if ( !bReturn && !uiMaxLen && ( *(*cpp) == cToken ) ) {

    // special case where token sits at max length

    (*cpp)++ ;    // skip master past token

    bReturn = 1 ; // return TRUE after all

    }

  return bReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: void   strpadr      Pads the right side of a zstr with
                                  a specified character to a specified
                                  length.

  PARAMETERS: char*  sz           Address of string to pad
              char   cPad         Pad character
              ushort uiLen        Length of resultant string

      RETURN: bool   bReturn      Boolean value indicating success

       NOTES: Nice for space padding data fields

\* ------------------------------------------------------------------------ */
    
void strpadr ( char *sz , char cPad , ushort uiLen ) {

  while ( uiLen-- && *sz ) *sz++ ;       // find terminating NULL

  uiLen++ ;                              // back up one

  while ( uiLen-- ) *sz++ = cPad ;       // pad to length

  *sz = '\0' ;                           // zero terminate at length

  return ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: char*  stringify    Converts a block of numeric types
                                  into a comma delimited string.

  PARAMETERS: void*  vpBlock      Address of memory containing numerics
              char*  szMask       Mask used to interpret the numerics

      RETURN: char*  szaReturn    Address of statically allocated zstr

       NOTES: The function stringify() will read any contiguous block of
              one or more numerics ( short , long , or double ) and convert
              them to a comma separated string.  stringify() takes a pointer
              to the memory containing the block of numbers and a mask which
              determines how many numbers are read and what type each number
              is.  Examples:

                // short , short , short , short
                
                stringify ( iArray    , "iiii"       ) ;
                
                // short , double , long
                
                stringify ( &gSpecial , "idl"        ) ;
                
                // double , skip 8 bytes , short
                
                stringify ( &gWierd   , "dxxxxxxxxi" ) ;

     WARNING: If you are stringifying structures, be aware that you must
              instruct the compiler to pack data structures - OR - you
              must allow for the 2 or 4 byte boundaries that word-aligned
              structures follow!

\* ------------------------------------------------------------------------ */

char* stringify ( void *vpBlock , char *szMask ) {

  static char szaReturn[128] ; char c, *szReturn = szaReturn ;

  MEMSET ( szReturn , '\0' , 128 ) ;

  if ( vpBlock ) { // ! NULL
  
    while ( c = *(szMask++) ) {  // step through each character in the mask

      switch ( c ) {

        case 'i' : STRCPY ( szReturn , itosz ( *((short*)vpBlock) , 10 ) ) ;
                   vpBlock = (void*) ( (size_p)vpBlock + (size_p)sizeof(short) ) ;
                   break ;

        case 'l' : STRCPY ( szReturn , ltosz ( *((long*)vpBlock) , 10 ) ) ;
                   vpBlock = (void*) ( (size_p)vpBlock + (size_p)sizeof(long) ) ;
                   break ;

        case 'd' : STRCPY ( szReturn , dtosz ( *((double*)vpBlock) , 4 ) ) ;
                   vpBlock = (void*) ( (size_p)vpBlock + (size_p)sizeof(double) ) ;
                   break ;

        default  : vpBlock = (void*) ( (size_p)vpBlock + (size_p)sizeof(char) ) ;
                   break ;

        }

      while ( *(++szReturn) ) ; // move past stringified number to next '\0'
      *(szReturn++) = ','     ; // place comma and move to next '\0'

      }

    *(--szReturn) = '\0' ;      // erase trailing comma, zero terminate

    }

  return szaReturn ;

  /* -------------------------------------------------------------------- *\

      NOTE: size_p is a typedef from WRAPI.H which corresponds to the size
            of a pointer (either ushort or ulong).  Most APIs use far
            pointers, therefore size_p usually is ulong.  The size_p
            typdef anticipates APIs with near pointers.

  \* -------------------------------------------------------------------- */

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: char*  itosz        Convert short int to zstr

  PARAMETERS: short  iValue       short integer to convert
              short  iRadix       radix value to use 

      RETURN: char*  szaReturn    Statically allocated return value

       NOTES: Like itoa(), but return value statically allocated

\* ------------------------------------------------------------------------ */
    
char* itosz ( short iValue , short iRadix ) {

  static char szaReturn [17] ;

  STRCPY ( szaReturn , ltosz ( (long) iValue , iRadix ) ) ;

  return szaReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: char*  uitosz       Convert unsinged short int to zstr

  PARAMETERS: ushort uiValue      unsigned short integer to convert
              short  iRadix       radix value to use 

      RETURN: char*  szaReturn    Statically allocated return value

       NOTES: Like ultoa(), but return value statically allocated

\* ------------------------------------------------------------------------ */
    
char* uitosz ( ushort uiValue , short iRadix ) {

  static char szaReturn [17] ;

  STRCPY ( szaReturn , ltosz ( (long) uiValue , iRadix ) ) ;

  return szaReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: char*  ltosz        Convert long int to zstr

  PARAMETERS: long   lValue       long integer to convert
              short  iRadix       radix value to use 

      RETURN: char*  szaReturn    Statically allocated return value

       NOTES: Like ltoa(), but return value statically allocated

\* ------------------------------------------------------------------------ */
    
char* ltosz ( long lValue , short iRadix ) {

  static char szaReturn [33] ;

  bool bNegative ;
  
  char szaTemp [33] , *cpIn = szaTemp , *cpOut = szaReturn ;

  if ( bNegative = (bool) ( lValue < 0L ) ) lValue = -lValue ;

  do {

    *cpIn = (char) ( ( lValue % (long) iRadix ) + '0' ) ;

    if ( *cpIn > '9' ) *cpIn += '\x07' ;
    
    lValue /= 10L ;
    
    cpIn++ ;

    }
  
    while ( lValue ) ;

  if ( bNegative ) *cpOut++ = '-' ;

  while ( cpIn != szaTemp ) *cpOut++ = *--cpIn ;

  *cpOut = '\0' ;

  return ( szaReturn ) ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: char*  ultosz       Convert unsigned long int to zstr

  PARAMETERS: ulong  ulValue      unsigned long integer to convert
              short  iRadix       radix value to use 

      RETURN: char*  szaReturn    Statically allocated return value

       NOTES: Like ultoa(), but return value statically allocated

\* ------------------------------------------------------------------------ */
    
char* ultosz ( ulong ulValue , short iRadix ) {

  static char szaReturn [33] ;

  char szaTemp [33] , *cpIn = szaTemp , *cpOut = szaReturn ;

  do {

    *cpIn = (char) ( ( ulValue % (ulong) iRadix ) + '0' ) ;

    if ( *cpIn > '9' ) *cpIn += '\x07' ;
    
    ulValue /= 10L ;
    
    cpIn++ ;

    }
  
    while ( ulValue ) ;

  while ( cpIn != szaTemp ) *cpOut++ = *--cpIn ;

  *cpOut = '\0' ;

  return ( szaReturn ) ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: char*  dtosz        Convert double float to zstr

  PARAMETERS: double dValue       double float to convert
              short  iDecimals    number of decimal places

      RETURN: char*  szaReturn    Statically allocated return value

       NOTES: a cross between itoa() and fcvt()

\* ------------------------------------------------------------------------ */
    
char* dtosz ( double dValue , short iDecimals ) {

  static char szaReturn [18] ; // sign + 15 digits + decimal point + \0

  short iIn = 0 , iOut = 0 ; bool bNegative ; char szaTemp [18] ;

  long lValue = (long) ( dValue * POW10 ( iDecimals ) + 0.5 ) ;

  if ( bNegative = (bool) ( lValue < 0 ) ) lValue = -lValue ;

  do {

    szaTemp[iIn++] = (char) ( ( lValue % 10L ) + '0' ) ;
    lValue /= 10L ;

    if ( iIn == iDecimals ) szaTemp[iIn++] = '.' ;

    } while ( lValue || iIn < iDecimals ) ;

  if ( szaTemp[iIn-1] == '.' ) szaTemp[iIn++] = '0' ;

  if ( bNegative ) szaReturn[iOut++] = '-' ;

  while ( iIn ) szaReturn[iOut++] = szaTemp[--iIn] ;

  szaReturn[iOut] = '\0' ;

  return ( szaReturn ) ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: short  sztoi        Convert zstr to short integer

  PARAMETERS: char*  sz           zstr to convert

      RETURN: short               short integer return value

       NOTES: Like atoi()

\* ------------------------------------------------------------------------ */
    
short sztoi ( char *sz ) {

  return (short) sztod ( sz ) ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: ushort sztoui       Convert zstr to unsinged short integer

  PARAMETERS: char*  sz           zstr to convert

      RETURN: ushort              unsigned short integer return value

       NOTES: Like atoi(), but unsigned

\* ------------------------------------------------------------------------ */
    
ushort sztoui ( char *sz ) {

  return (ushort) sztod ( sz ) ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: long   sztol        Convert zstr to long integer

  PARAMETERS: char*  sz           zstr to convert

      RETURN: long                long integer return value

       NOTES: Like atol()

\* ------------------------------------------------------------------------ */
    
long sztol ( char *sz ) {

  return (long) sztod ( sz ) ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: ulong  sztoul       Convert zstr to unsinged long integer

  PARAMETERS: char*  sz           zstr to convert

      RETURN: ulong               unsigned long integer return value

       NOTES: Like atol(), but unsigned

\* ------------------------------------------------------------------------ */
    
ulong sztoul ( char *sz ) {

  return (ulong) sztod ( sz ) ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: double sztod        Convert zstr to double float

  PARAMETERS: char*  sz           zstr to convert

      RETURN: double              double float return value

       NOTES: 

\* ------------------------------------------------------------------------ */
    
double sztod ( char *sz ) {

  bool   bNegative = (bool) ( *sz == '-' ) ;
  bool   bDecimal  = 0                     ;
  short  iDigits   = 0                     ;
  long   lValue    = 0L                    ;
  long   lDivisor  = 1L                    ;

  if ( bNegative ) sz++ ;

  while ( *sz ) {

    if ( *sz == '.' )

      if ( bDecimal )                       // 2nd decimal causes termination

        break ;

      else {

        bDecimal = 1 ;
        sz++         ;
        continue     ;

        }

    if ( *sz < '0' || *sz > '9' )           // non-digit causes termination

      break ;

    lValue *= 10                      ;
    lValue += (short) ( *sz++ - '0' ) ;

    if ( iDigits++ == 16 )                  // 16th digit causes termination

      break ;

    if ( bDecimal )

      lDivisor *= 10 ;

    }

  if ( bNegative ) lValue = - lValue ;

  return (double) ( (double) lValue / (double) lDivisor ) ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: double powerof10    Return 10^i

  PARAMETERS: short  i            power to which ten is raised

      RETURN: double dReturn      10^i

       NOTES: Like pow10(), used by other conversion functions

\* ------------------------------------------------------------------------ */
    
double powerof10 ( short i ) {

  double dReturn = 1.0 ;

  if ( i < 0 )
    while ( i++ ) dReturn /= 10.0 ;
  else
    while ( i-- ) dReturn *= 10.0 ;

  return dReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: double round        Round a double to specified precision

  PARAMETERS: double dValue       Value to round
              short  iDecimals    Number of decimal places

      RETURN: doible              Rounded number

       NOTES: 

\* ------------------------------------------------------------------------ */
    
double round ( double dValue , short iDecimals ) { // round double

  double dPowerOfTen = POW10 ( iDecimals ) ;

  return ( (double) (long) ( dValue * dPowerOfTen + 0.5 ) ) / dPowerOfTen;

  }


/* Memory allocation ------------------------------------------------------ */



/* ------------------------------------------------------------------------ *\

    FUNCTION: void*  WAlloc       Allocate a block of memory from heap

  PARAMETERS: ushort uiSize       Amount of memory to allocate

      RETURN: void*  vpReturn     Address of memory block (or NULL)

       NOTES: WAlloc() allocates six extra bytes in which the size and API
              handle (if any) are stored.  The extra six bytes are taken
              at the beginning of the block so the pointer returned is
              actually six bytes in from the start of the block.

\* ------------------------------------------------------------------------ */
    
void* WAlloc ( ushort uiSize ) {

  ushort    uiAllocSize = (ushort) ( uiSize + ALLOC_LEN ) ;
  char far *vpReturn = NULL ;
  ulong     ulHnd = 0L ;

  #if defined ( _API_CLIP4)

    vpReturn = (void*) _exmgrab ( uiAllocSize ) ;

  #elif defined ( _API_CLIP5 )
    
    ulHnd    = (unsigned long) _vAlloc ( uiAllocSize ) ;
    vpReturn = (void*)         _vLock  ( (HANDLE) ulHnd ) ;

  #elif defined ( _API_CLIP5X )

    vpReturn = (void*) _xalloc ( uiAllocSize ) ;

  #elif defined ( _API_FOX )

    ulHnd = _AllocHand ( uiAllocSize ) ;

    if ( ulHnd ) {
                         _HLock     ( (MHANDLE) ulHnd ) ;
      vpReturn = (void*) _HandToPtr ( (MHANDLE) ulHnd ) ;
      }

  #else

    vpReturn = (char*) malloc ( uiAllocSize ) ;

  #endif

  if ( vpReturn ) {

    vpReturn += ALLOC_LEN ;
    MEMCPY ( vpReturn - ALLOC_LEN , &uiSize , sizeof ( ushort ) ) ;
    MEMCPY ( vpReturn - ALLOC_HND , &ulHnd  , sizeof ( ulong  ) ) ;

    }

  return (void*) vpReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: void   WFree        Free a block of WAlloc()'ed memory

  PARAMETERS: void** vpp          Amount of memory to allocate

      RETURN:                     Nothing

       NOTES: WFree deallocates a block of memory previously allocated
              by WAlloc().

\* ------------------------------------------------------------------------ */
    
void WFree ( void** vpp ) {

  if ( *vpp ) {

    ushort uiLen = * (ushort*) ( (char*) *vpp - ALLOC_LEN ) ;
    ulong  ulHnd = * (ulong*)  ( (char*) *vpp - ALLOC_HND ) ;
    *vpp = ( (char*) *vpp ) - ALLOC_LEN ;

    #if defined ( _API_CLIP4 )

      _exmback ( *vpp , uiLen ) ;

    #elif defined ( _API_CLIP5 )

      _vUnLock ( (HANDLE) ulHnd ) ;
      _vFree   ( (HANDLE) ulHnd ) ;

    #elif defined ( _API_CLIP5X )
      
      _xfree ( *vpp ) ;
  
    #elif defined ( _API_FOX )

      _HUnLock  ( (MHANDLE) ulHnd ) ;
      _FreeHand ( (MHANDLE) ulHnd ) ;

    #else

      free ( *vpp ) ;

    #endif

    *vpp = NULL ;

    }

  return ;

  }



/* Device Output ---------------------------------------------------------- */



/* ------------------------------------------------------------------------ *\

    FUNCTION: short  iWPrnStart   Initialize a print context

  PARAMETERS: WPRN*  wpp          Address of wrapped print device
              short  hPRN         Handle to DOS File or a Device constant

      RETURN: short  iReturn      WRAPI error code

       NOTES: Must be done prior to printing!

\* ------------------------------------------------------------------------ */
    
short iWPrnStart ( WPRN* wpp , short hPRN ) {

  short iReturn = EC_NONE ;

  char szaSpooler[8] ;

  #ifdef _FAMILY_DLL

    FARPROC lpfnAbortProc ;

    #ifdef _API_FOX

      hLibInst = Inst ;

    #endif

  #endif

  (*wpp).bStd = (bool) ( hPRN <= PRN_STD ) ;

  if ( ! (*wpp).bStd )

    (*wpp).fh = hPRN ;

  else {

    #ifndef _FAMILY_DLL

      (*wpp).fh = F_OPEN_WO (   hPRN == PRN_STD  ? "PRN"
                              : hPRN == PRN_LPT1 ? "LPT1"
                              : hPRN == PRN_LPT2 ? "LPT2"
                              : hPRN == PRN_LPT3 ? "LPT3"
                              :                    "NUL"
                            ) ;

      if ( (*wpp).fh == -1 )

        iReturn = EC_WR_PRN ;

    #else  

      if ( (*wpp).bStd = (bool) ( hPRN == PRN_STD ) ) {

        GetProfileString ("windows" , "spooler" , "" , szaSpooler , 8 ) ;
      
        if ( *szaSpooler == 'y' || *szaSpooler == 'Y' ) {

          WriteProfileString ("windows" , "spooler" , "no" ) ;

	  		  (*wpp).bSpool = 1 ;

          }

	  	  }

		  else

			  (*wpp).bSpool = 0 ;

      if ( (*wpp).bStd )

        if ( ! ( (*wpp).hdc = hdcGetPrinterDC ( ) ) )

          iReturn = EC_GDI_DC ;

        else {

          lpfnAbortProc = MakeProcInstance ( (FARPROC) bAbortProc , hLibInst ) ;

          Escape ( (*wpp).hdc , SETABORTPROC , 0 , (LPSTR) lpfnAbortProc , NULL ) ;

          if ( ! bGDICommand ( (*wpp).hdc , STARTDOC ) )

            iReturn = EC_GDI_CMD ;

          }

    #endif

    }

  return iReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: short  iWPrnStart   End a print context

  PARAMETERS: WPRN*  wpp          Address of wrapped print device

      RETURN: short  iReturn      WRAPI error code

       NOTES: Must be done when print job is complete!

\* ------------------------------------------------------------------------ */
    
short iWPrnEnd ( WPRN* wpp ) {

  short iReturn = EC_NONE ;

  #ifndef _FAMILY_DLL

    if ( (*wpp).bStd && (*wpp).fh > 0 )

      F_CLOSE ( (*wpp).fh ) ;

  #else

    if ( (*wpp).bStd ) {

      if ( ! bGDICommand ( (*wpp).hdc , NEWFRAME ) )

        iReturn = EC_GDI_CMD ;

      else

        if ( ! bGDICommand ( (*wpp).hdc , ENDDOC ) )

          iReturn = EC_GDI_CMD ;

      if ( ! DeleteDC ( (*wpp).hdc ) )

        iReturn = EC_GDI_DC ;

			if ( (*wpp).bSpool )

				 WriteProfileString ( "windows" , "spooler" , "yes" ) ;

      }

  #endif

  return iReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: bool   bWPrnC       Send single character to print device

  PARAMETERS: WPRN*  wpp          Address of wrapped print device
              char   c            Character to print

      RETURN: bool   bReturn      Boolean value indicating success

       NOTES: 

\* ------------------------------------------------------------------------ */
    
bool bWPrnC ( WPRN* wpp , char c ) {

  bool bReturn ;

  #ifdef _FAMILY_DLL

    WSTR ws ;  char sza[2] ;

    if ( (*wpp).bStd ) {

      sza[0] = c , sza[1] = '\0' ; ws.cp = sza ; ws.uiLen = (ushort) 1 ;

      bReturn = bGDIPrint ( (*wpp).hdc , &ws ) ;

      }

     else

       if ( ! ( bReturn = (bool) ( (ushort) F_WRITE ( (*wpp).fh , &c , 1 ) == 1 ) ) )

         SET_ERR_OUT ( ) ;

  #else

    if ( ! ( bReturn = (bool) ( (ushort) F_WRITE ( (*wpp).fh , &c , 1 ) == 1 ) ) )

      SET_ERR_OUT ( ) ;

  #endif

  return bReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: bool   bWPrnSB      Send buffered string to print device

  PARAMETERS: WPRN*  wpp          Address of wrapped print device
              char*  sb           Address of character string to print
              ushort uiLen        Length of buffered string

      RETURN: bool   bReturn      Boolean value indicating success

       NOTES: 

\* ------------------------------------------------------------------------ */
    
bool bWPrnSB ( WPRN* wpp , char* sb , ushort uiLen ) {

  bool   bReturn ;

  #ifdef _FAMILY_DLL

    WSTR ws ;

    if ( (*wpp).bStd ) {

      ws.cp = sb ; ws.uiLen = uiLen ;

      bReturn = bGDIPrint ( (*wpp).hdc , &ws ) ;

      }

    else

      if ( ! ( bReturn = (bool) ( (ushort) F_WRITE ( (*wpp).fh , sb , uiLen ) == uiLen ) ) )

        SET_ERR_OUT ( ) ;

  #else

    if ( ! ( bReturn = (bool) ( (ushort) F_WRITE ( (*wpp).fh , sb , uiLen ) == uiLen ) ) )

      SET_ERR_OUT ( ) ;

  #endif

  return bReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: bool   bWPrnSZ      Send zstr to print device

  PARAMETERS: WPRN*  wpp          Address of wrapped print device
              char*  sz           Address of zstr to print

      RETURN: bool   bReturn      Boolean value indicating success

       NOTES: 

\* ------------------------------------------------------------------------ */
    
bool bWPrnSZ ( WPRN* wpp , char* sz ) {

  bool   bReturn ;   ushort uiLen = (ushort) STRLEN ( sz ) ;

  #ifdef _FAMILY_DLL

    WSTR ws ;

    if ( (*wpp).bStd ) {

      ws.cp = sz ; ws.uiLen = uiLen ;

      bReturn = bGDIPrint ( (*wpp).hdc , &ws ) ;

      }

    else

      if ( ! ( bReturn = (bool) ( (ushort) F_WRITE ( (*wpp).fh , sz , uiLen ) == uiLen ) ) )

        SET_ERR_OUT ( ) ;

  #else

    if ( ! ( bReturn = (bool) ( (ushort) F_WRITE ( (*wpp).fh , sz , uiLen ) == uiLen ) ) )

      SET_ERR_OUT ( ) ;

  #endif

  return bReturn ;

  }



/* ------------------------------------------------------------------------ *\

    FUNCTION: bool   bWPrnWSP     Send wstr to print device

  PARAMETERS: WPRN*  wpp          Address of wrapped print device
              WSTR*  wsp          Address of wstr

      RETURN: bool   bReturn      Boolean value indicating success

       NOTES: 

\* ------------------------------------------------------------------------ */
    
bool bWPrnWSP ( WPRN* wpp , WSTR* wsp ) {

  bool   bReturn = (bool) 0 ;

  ushort uiLen ;

  if ( ! IsWNIL ( wsp ) ) {

    #ifdef _FAMILY_DLL

      if ( (*wpp).bStd )

        bReturn = bGDIPrint ( (*wpp).hdc , wsp ) ;

      else {

        uiLen = wsplen ( wsp ) ;

        if ( ! ( bReturn = (bool) ( F_WRITE ( (*wpp).fh , (*wsp).cp , uiLen ) == (short) uiLen ) ) )

          SET_ERR_OUT ( ) ;

        }

    #else

      uiLen = wsplen ( wsp ) ;

      if ( ! ( bReturn = (bool) ( F_WRITE ( (*wpp).fh , (*wsp).cp , uiLen ) == uiLen ) ) )

        SET_ERR_OUT ( ) ;

    #endif

    }

  return bReturn ;

  }



/* Windows DLL stuff ------------------------------------------------------ */


#ifdef _FAMILY_DLL


  BOOL FAR PASCAL _export bAbortProc ( HDC hdcPrn , short iCode ) {

    MSG msg ;

    while ( PeekMessage ( &msg , (HWND) 0 , 0 , 0 , PM_REMOVE ) ) {
      TranslateMessage (&msg) ;
      DispatchMessage  (&msg) ;
      }

    return ( ! iCode ) ;

    }


  HDC hdcGetPrinterDC ( void ) {

    HDC hdcReturn = 0 ;

    // this is the standard GetPrinterDC() func everyone uses

    char  szaPrinter[80] , // Default printer info from WIN.INI
         *szDevice       , // Device name (e.g. "HP LaserJet III")
         *szDriver       , // Driver name (e.g. "HPPCL5A")
         *szOutput       ; // Output port (e.g. "LPT1:")

    // get the currently selected printer info from WIN.INI

    GetProfileString ("windows", "device", "", szaPrinter, 80 ) ;

    // use string token func to parse out comma delimited info string

    szDevice = strtok ( szaPrinter, ","  ) ;
    szDriver = strtok ( NULL      , ", " ) ;
    szOutput = strtok ( NULL      , ", " ) ;

    // create the device context from the printer info

    if (    ( szDevice != NULL )
         && ( szDriver != NULL )
         && ( szOutput != NULL ) )

      hdcReturn = CreateDC ( szDriver , szDevice , szOutput , NULL ) ;

    return hdcReturn ;

    }



  bool bGDICommand ( HDC hdc , short iCommand ) {

    return (bool) ( Escape ( hdc , iCommand , 0 , NULL , NULL ) > 0 ) ;

    }


  bool bGDIPrint ( HDC hdc , WSTR* wsp ) {

    // get length of string

    bool bReturn = 1 ; ushort uiLen ; void *vpStruct ;

    if ( uiLen = wsplen ( wsp ) ) {

      // allocate printer structure - string prefixed with 2 bytes for length

      vpStruct = malloc ( uiLen + 2 ) ;

      // copy length to print structure

      MEMCPY ( vpStruct , &uiLen , 2 ) ;

      // copy print data to print structure at 3rd byte

      MEMCPY ( ( (char*) vpStruct ) + 2 , (*wsp).cp , uiLen ) ;

      // send data (text and/or escape sequences) to printer

      if ( ! ( bReturn = (bool) ( Escape (   hdc
                                           , DEVICEDATA
                                           , uiLen + 2
                                           , (LPCSTR) vpStruct
                                           , NULL
                                         )
                                  > 0
                                )
             )
         )

        SET_ERR_GDI_CMD() ;

      }

    return bReturn ;

    }


  #ifndef _API_FOX

    int FAR PASCAL LibMain ( HANDLE hInstance   ,
                             WORD   wDataSeg    ,
                             WORD   cbHeapSize  ,
                             LPSTR  lpszCmdLine ) {

      if ( cbHeapSize > 0 )  UnlockData ( 0 ) ;

      hLibInst = hInstance ;

      return 1;  // return load success to Windows

      }


    int FAR PASCAL WEP (int nParameter) {

      return 1;  // return unload success to Windows

      }

  #endif

#endif



/* Clipper & Fox parameter grabbers and return senders -------------------- */



#ifdef _FAMILY_XBASE

  char cWGrab ( API_PARM ) {

    char cReturn = NIL_C ;

    if ( PARMCOUNT >= iParm )
      if ( PARMTYPE ( iParm ) == XBASE_CHAR )
        API_GRAB_C
      else
        if ( PARMTYPE ( iParm ) != XBASE_NIL )
          SET_ERR_TYPE ( ) ;

    return cReturn ;

    }

  WSTR* wspWGrab ( API_PARM ) {

    static WSTR wsReturn ;

    wsReturn.cp    = NIL_S      ;
    wsReturn.uiLen = (ushort) 1 ;
    wsReturn.ulHnd = 0L         ;

    if ( PARMCOUNT >= iParm )
      if ( PARMTYPE ( iParm ) == XBASE_STRING )
        API_GRAB_S
      else
        if ( PARMTYPE ( iParm ) != XBASE_NIL )
          SET_ERR_TYPE ( ) ;

    return &wsReturn ;

    }

  bool bWGrab ( API_PARM ) {

    bool bReturn = NIL_B ;

    if ( PARMCOUNT >= iParm )
      if ( PARMTYPE ( iParm ) == XBASE_BOOL )
        API_GRAB_B
      else
        if ( PARMTYPE ( iParm ) != XBASE_NIL )
          SET_ERR_TYPE ( ) ;

    return bReturn ;

    }

  short iWGrab ( API_PARM ) {

    short iReturn = NIL_I ;

    if ( PARMCOUNT >= iParm )
      if ( PARMTYPE ( iParm ) == XBASE_INT )
        API_GRAB_I
      else
        if ( PARMTYPE ( iParm ) != XBASE_NIL )
          SET_ERR_TYPE ( ) ;

    return iReturn ;

    }

  long lWGrab ( API_PARM ) {

    long lReturn = NIL_L ;

    if ( PARMCOUNT >= iParm )
      if ( PARMTYPE ( iParm ) == XBASE_LONG )
        API_GRAB_L
      else
        if ( PARMTYPE ( iParm ) != XBASE_NIL )
          SET_ERR_TYPE ( ) ;

    return lReturn ;

    }


  double dWGrab ( API_PARM ) {

    double dReturn = NIL_D ;

    if ( PARMCOUNT >= iParm )
      if ( PARMTYPE ( iParm ) == XBASE_DOUBLE )
        API_GRAB_D
      else
        if ( PARMTYPE ( iParm ) != XBASE_NIL )
          SET_ERR_TYPE ( ) ;

    return dReturn ;

    }

#endif



#ifdef _API_CLIPPER


  short WStrcmp ( char *sz1 , char *sz2 ) {

    short iReturn = 0 ;

    while ( *sz1 && ( ! ( iReturn = (short) ( *sz1++ - *sz2++ ) ) ) ) ;

    return iReturn ;

    }



  void vWError ( void ) {

    #ifdef _API_CLIP4

      CALL_ERR_CLIP4() ;

    #else

      static ERR gError ; short iErrCode ;

      char *cp , szaErrLoc[16] , szaErrText[EC_LEN] ;

                 iErrCode = GET_ERR_CODE ( )            ;
      STRCPY ( szaErrLoc  , GET_ERR_LOC  ( )          ) ;
      STRCPY ( szaErrText , GET_ERR_TEXT ( iErrCode ) ) ;

      cp = szaErrLoc ; while ( *++cp ) ; *cp++ = '(' ; *cp++ = ')' ; *cp = '\0' ;

      gError.uiSeverity  = 1                 ; // Warning
      gError.uiGenCode   = 0                 ;
      gError.uiOSCode    = 0                 ;
      gError.uiFlags     = 4                 ; // Bit 0: RETRY
                                               // Bit 2: DEFAULT
      gError.uiTries     = 0                 ;
      gError.uiSubCode   = (ushort) iErrCode ;
      gError.szSubSystem = SYSTEM_NAME ( )   ;
      gError.szDescript  = szaErrText        ;
      gError.szOp        = szaErrLoc         ;
      gError.szFileName  = ""                ;

      _eError ( &gError ) ;

    #endif

    return ;

    }


#endif



#ifdef _API_FOX

  #if _FOX_DEBUG

    void _PutLong ( long lValue , short iWidth ) {

      Value val ;

      val.ev_type   = 'I'    ;
      val.ev_width  = iWidth ;
      val.ev_long   = lValue ;

      _PutChr   ( '{' )  ;
      _PutValue ( &val ) ;
      _PutChr   ( '}' )  ;

      return ;

      }

    void _PutDouble ( double dValue , short iWidth , short iDecimals ) {

      Value val ;

      val.ev_type   = 'N'       ;
      val.ev_width  = iWidth    ;
      val.ev_length = iDecimals ;
      val.ev_real   = dValue    ;

      _PutChr   ( '{' )  ;
      _PutValue ( &val ) ;
      _PutChr   ( '}' )  ;

      return ;

      }

    void _PutString ( WSTR* wsp ) {

      Value val ;

      val.ev_type   = 'C'          ;
      val.ev_length = (*wsp).uiLen ;
      val.ev_real   = (*wsp).ulHnd ;

      _PutChr   ( '{' )  ;
      _PutValue ( &val ) ;
      _PutChr   ( '}' )  ;

      return ;

      }

  #endif



  void vErrTrapFox ( void ) {

    char *szFunc , szaFunc[16] ; short iError ; Value gVal ;

    if ( * ( szFunc = GET_ERR_FNAME ( ) ) ) {

      STRCPY ( szaFunc , szFunc ) ;
      szFunc = szaFunc            ;
      while ( *szFunc++ )         ;
      *--szFunc = '('             ;
      *++szFunc = ')'             ;
      *++szFunc = '\0'            ;

      if ( iError = (short) _Evaluate ( &gVal , szaFunc ) )

        _Error ( iError ) ;

      }

    return ;

    }



  /* -------------------------------------------------------------------- *\

     FoxPro does not permit passing arrays to a function or returning an
     array from a function.  Many other host APIs permit either array
     return values or typedef return values.  Fox requires that complex
     return values be stringified.  For example, the array { 1.0 , 2.5 }
     would be returned as "1.0000,2.5000".

     In an effort to make the Fox API more usable, the function
     AUNSTRING() will automatically convert a stingified value into
     an array.  The syntax for AUNSTRING is:

     AUNSTRING ( cArrName , cStringifiedValues ) --> lSuccess

     AUNSTRING will declare a new PRIVATE variable named cArrName.  If
     cArrName already exists, it is checked for the proper dimension.
     If the dimension is ok, the existing cArrName is used, otherwise,
     cArrName is RELEASEd and a new cArrName created.  Error handling
     is done through the Fox error system.

  \* -------------------------------------------------------------------- */



  void far AUNSTRING ( ParamBlk *gpFoxParm ) {

    bool bReturn = 0 ;
      
    short iCommas = 0 , iError = 0 ;

    short i , iLen , iWidth , iDecimals ; double dValue ;

    Locator gLoc ; Value gVal ; NTI iNTI ;

    char szaName[11] , *cpName , *cpString ;

    // must have 2 params of non-Null type 'C'

    if ( ( (*gpFoxParm).pCount == 2 )               &&
         ( (*gpFoxParm).p[0].val.ev_type   == 'C' ) &&
         ( (*gpFoxParm).p[1].val.ev_type   == 'C' ) &&
         ( (*gpFoxParm).p[0].val.ev_length        ) &&
         ( (*gpFoxParm).p[1].val.ev_length        )    ) {

      // lock string handles

      _HLock ( (*gpFoxParm).p[0].val.ev_handle ) ;
      _HLock ( (*gpFoxParm).p[1].val.ev_handle ) ;

      // convert string handles to far pointers

      cpName   = (char*) _HandToPtr ( (*gpFoxParm).p[0].val.ev_handle ) ;
      cpString = (char*) _HandToPtr ( (*gpFoxParm).p[1].val.ev_handle ) ;

      // count commas to determine how many elements in the array

      iLen = (*gpFoxParm).p[1].val.ev_length ;

      while ( iLen-- ) if ( *(cpString + iLen) == ',' ) iCommas++ ;

      // copy FoxPro string to a string buffer and zero terminate

      i = 0 ; iLen = (*gpFoxParm).p[0].val.ev_length ;

      while ( i++ < 10 && iLen-- ) szaName[i-1] = *cpName++ ; // 10 char max

      szaName[i-1] = '\0' ;

      // use Locator structure to create array of proper dimension

      gLoc.l_subs = (short) 1 ; gLoc.l_sub1 = (short) ( iCommas + 1 ) ;

      if ( ( iError = (short) _NewVar ( szaName , &gLoc , NV_PRIVATE ) ) < 0 ) {

        // Could not create array, let's see if it already exists

        if ( ( iNTI = _NameTableIndex ( szaName ) ) >= 0 ) {

          // Aha, the array already exists, is it the right size?

          if ( _ALen ( iNTI , AL_SUBSCRIPT1 ) == (long) ( iCommas + 1 ) &&
               _ALen ( iNTI , AL_SUBSCRIPT2 ) == (long) 0                  ) {

            // Yep, we can use it, let's reference it...

            if ( _FindVar ( iNTI , -1 , &gLoc ) ) iError = 0 ;

            }

          else {

            // Nope, release it and try to create it again!

            if ( ! ( iError = (short) _Release ( iNTI ) ) )
              iError = (short) _NewVar ( szaName , &gLoc , NV_PRIVATE ) ;

            }

          }

        }

      if ( iError >= 0 ) {

        // Unstringify into newly created array

        i = 0 ; iLen = (short) ( (*gpFoxParm).p[1].val.ev_length - 1 ) ;

        gVal.ev_type = 'N' ; gLoc.l_sub1 = 0 ;

        dValue = 0.0 ; iWidth = 0 ; iDecimals = 0 ;

        do {

          switch ( *(cpString+i) ) {

            default  : break ;
            case '.' : iDecimals = 1 ;
                       iWidth++ ;
                       break ;
            case '0' :
            case '1' :
            case '2' :
            case '3' :
            case '4' :
            case '5' :
            case '6' :
            case '7' :
            case '8' :
            case '9' : dValue *= 10 ;
                       dValue += (double) ( *(cpString+i) - '0' ) ;
                       if ( iDecimals ) iDecimals++ ;
                       iWidth++ ;
                       break ;

            }


          if ( i == iLen || *(cpString+i) == ',' ) {

            if ( iDecimals ) iDecimals-- ; // was one more than actual

            gVal.ev_width  = iWidth      ; // display width and decimals
            gVal.ev_length = iDecimals   ;

            while ( iDecimals-- )
              dValue /= 10.0             ; // set decimal place

            gVal.ev_real   = dValue      ; // value ready to be stored

            gLoc.l_sub1++                ; // move to next element in array

            // Attempt to store, save error condition

            iError = (short) _Store ( &gLoc , &gVal ) ;

            // Reset for next value

            dValue = 0.0 ; iWidth = 0 ; iDecimals = 0 ;

            }

          } while ( i++ < iLen && iError >= 0 ) ;

          bReturn = 1 ;

        }

      // Unlock strings

      _HUnLock ( (*gpFoxParm).p[0].val.ev_handle ) ;
      _HUnLock ( (*gpFoxParm).p[1].val.ev_handle ) ;

      }

    if ( iError < 0 ) _Error ( iError ) ; // FoxPro error ... goodbye

    _RetLogical ( bReturn ) ;

    return ;

    }

#endif


#ifdef _API_VBD

  void vErrTrapVBD ( void ) {

    ERR_VBD ;

    }

#endif


#ifdef _API_VBW

  void vErrTrapVBW ( void ) {

    VBSetErrorMessage ( GET_ERR_HOST() , CALL_ERR_MSG() ) ;
    VBRuntimeError    ( GET_ERR_HOST()                  ) ;

    return ;

    }

#endif


#ifdef _API_C

	void vErrTrapC ( void ) {

    #ifdef _FAMILY_DLL

      MessageBox (   (HWND)   NULL
                   , (LPCSTR) ERR_MSG
                   , (LPCSTR) SYSTEM_NAME
                   , (UINT)   MB_OK
                 ) ;

    #else

      void  (*vpFunc)() = (void*) GET_ERR_FPTR() ;

      if ( vpFunc ) vpFunc ( ) ;

    #endif

    return ;

    }

#endif



/* End-of-file ------------------------------------------------------------ */
