/*****************************************************************************/
/*	       Copyright (c) 1994 by Jyrki Salmi <jytasa@jyu.fi>             */
/*        You may modify, recompile and distribute this file freely.         */
/*****************************************************************************/

/*
   Routines used to create a linked list of file names specified on the
   command-line and in the listfiles.
*/

#include <stdio.h>
#include <stdlib.h>
#define INCL_DOSFILEMGR
#include <os2.h>
#include <ctype.h>
#include <malloc.h>
#include <string.h>
#include "typedefs.h"
#include "common.h"
#include "dir.h"
#include "tl.h"
#include "global.h"

/* Adds a file of PATH to the list TL. SIZE is the file size in bytes. */

void tl_add(TL **tl, U8 *path, U32 size) {

  U32 buf_idx;
  TE *te;

  if ((te = malloc(sizeof(TE))) == NULL) {
    fprintf(stderr, "Failed to allocate memory\n");
    exit(1);
  }
  te->path_len = strlen(path);
  if ((te->path = malloc(te->path_len + 1)) == NULL) {
    fprintf(stderr, "Failed to allocate memory\n");
    exit(1);
  }
  memcpy(te->path, path, te->path_len + 1);

  buf_idx = te->path_len + 1;
  while (buf_idx > 0) {
    buf_idx--;
    if (te->path[buf_idx] == '/' ||
	te->path[buf_idx] == '\\') {
      buf_idx++;
      break;
    }
  }
  te->name_len = te->path_len - buf_idx;
  te->name = &te->path[buf_idx];

  if (*tl == NULL) {
    /* The list is empty */
    if ((*tl = malloc(sizeof(TL))) == NULL) {
      fprintf(stderr, "Failed to allocate memory\n");
      exit(1);
    }
    (*tl)->f = te;
    (*tl)->l = te;
    (*tl)->c = te;
    (*tl)->cnt = 0;
    (*tl)->size = 0;
    te->p = NULL;
  } else {
    (*tl)->l->n = te;
    te->p = (*tl)->l;
  }
  (*tl)->cnt++;
  (*tl)->size += size;
  (*tl)->l = te;
  te->n = NULL;
}

/* Add files of PATH to the transfer list. Possible wildcards in PATH will */
/* be expanded. If -recursive option is specified, files will be searched */
/* recursively. */

void tl_expanded_add(TL **tl, U8 *path) {

  U8 expanded_path[260];
  BOOLEAN found = 0;
  DIR_ENTRY dir_entry;

  /* Search for files */
  if (dir_find_first(path, 0, &dir_entry) == 0) {
    found = 1;
    do {
      sprintf(expanded_path, "%.*s%s",
	      (int)get_dir_len(path), path,
	      dir_entry.name);
      tl_add(tl, expanded_path, *dir_entry.size);
    } while (!dir_find_next(&dir_entry));
  }
  dir_find_close(&dir_entry);

  /* Search for directories */
  if (opt_recursive) {
    if (dir_find_first(path, DIR_FIND_DIRECTORY, &dir_entry) == 0) {
      found = 1;
      do {
	if (strcmp(dir_entry.name, ".") != 0 &&
	    strcmp(dir_entry.name, "..") != 0) {
	  sprintf(expanded_path, "%.*s%s\\*",
		  (int)get_dir_len(path), path,
		  dir_entry.name);
	  tl_expanded_add(tl, expanded_path);
	}
      } while (!dir_find_next(&dir_entry));
    }
    dir_find_close(&dir_entry);
  }
  if (!found)
    tl_add(tl, path, 0);	/* If we are sending, we'll */
  				/* notify about non-existing */
				/* files later... */
}

/* Reads the files from a listfile */

void tl_read_from_list(TL **tl, U32 expand, U8 *path) {

  U8 str[512];
  U32 str_len;
  FILE *stream;

  path++; /* skip the @-character */
  if ((stream = fopen(path, "r")) == NULL) {
    perror(path);
    return;
  }
  while (fgets(str, 511, stream) != NULL) {
    if (str[0] != '\0') {
      str_len = strlen(str);
      str_len--;
      if (str[str_len] == '\n')
	str[str_len] = '\0';
      if (expand)
	tl_expanded_add(tl, str);
      else
	tl_add(tl, str, 0);
    }
  }
  fclose(stream);
}

/* Checks if file of PATH exists in the transfer list. This is used when */
/* receiving to check that the file was specified on the command-line */

BOOLEAN tl_exists(TL *tl, U8 *path) {

  TE *te;

  te = tl->f;
  while (te != NULL) {
    if (stricmp(path, te->path) == 0)
      return(1);
    te = te->n;
  }
  return(0);
}

/* Frees the memory reserved by the transfer list */

void tl_free(TL **tl) {

  TE *te1;
  TE *te2;

  if (*tl != NULL) {
    te1 = (*tl)->f;
    while (te1 != NULL) {
      te2 = te1->n;
      free(te1->path);
      free(te1);
      te1 = te2;
    }
    free(*tl);
    *tl = NULL;
  }
}

