/* acl32.c - access to LAN Server ACL's
 *
 * Author:  Kai Uwe Rommel <rommel@ars.muc.de>
 * Created: Mon Aug 08 1994
 */
 
static char *rcsid =
"$Id: acl32.c,v 2.4 1994/08/24 21:07:55 ak Exp $";
static char *rcsrev = "$Revision: 2.4 $";

/*
 * $Log: acl32.c,v $
 * Revision 2.4  1994/08/24  21:07:55  ak
 * K.U.R. fix, double slashes.
 *
 * Revision 2.3  1994/08/22 18:23:10  ak
 * .
 *
 * Revision 2.2  1994/08/21 12:31:23  ak
 * Removed test code.
 *
 * Revision 2.1  1994/08/09 14:23:58  ak
 * Added Kai Uwe Rommels ACL implementation.
 * 
 */

#include <stdlib.h>
#include <string.h>

#define INCL_NOPM
#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>

#include "acl32.h"

#define BUFFERSIZE 4096

#define UNLEN 20
#define PSTR _far16ptr

typedef struct access_list 
{
  char acl_ugname[UNLEN+1];
  char acl_pad;
  USHORT acl_access;
}
ACCLIST;

typedef struct access_info
{
  PSTR acc_resource_name;
  USHORT acc_attr;
  USHORT acc_count;
}
ACCINFO;

BOOL initialized, netapi_avail;

static USHORT (APIENTRY *_NetAccessGetInfo)(PSZ pszServer, PSZ pszResource, 
  USHORT sLevel, PVOID pbBuffer, USHORT cbBuffer, PUSHORT pcbTotalAvail);
static USHORT (APIENTRY *_NetAccessSetInfo)(PSZ pszServer, PSZ pszResource, 
  USHORT sLevel, PVOID pbBuffer, USHORT cbBuffer, USHORT sParmNum);
static USHORT (APIENTRY *_NetAccessAdd)(PSZ pszServer, 
  USHORT sLevel, PVOID pbBuffer, USHORT cbBuffer);

USHORT NetAccessGetInfo(PSZ pszServer, PSZ pszResource, USHORT sLevel, 
			PVOID pbBuffer, USHORT cbBuffer, PUSHORT pcbTotalAvail)
{
  return (USHORT)
          (_THUNK_PROLOG (4+4+2+4+2+4);
           _THUNK_FLAT (pszServer);
           _THUNK_FLAT (pszResource);
           _THUNK_SHORT (sLevel);
           _THUNK_FLAT (pbBuffer);
           _THUNK_SHORT (cbBuffer);
           _THUNK_FLAT (pcbTotalAvail);
           _THUNK_CALLI (_emx_32to16(_NetAccessGetInfo)));
}

USHORT NetAccessSetInfo(PSZ pszServer, PSZ pszResource, USHORT sLevel, 
			PVOID pbBuffer, USHORT cbBuffer, USHORT sParmNum)
{
  return (USHORT)
          (_THUNK_PROLOG (4+4+2+4+2+2);
           _THUNK_FLAT (pszServer);
           _THUNK_FLAT (pszResource);
           _THUNK_SHORT (sLevel);
           _THUNK_FLAT (pbBuffer);
           _THUNK_SHORT (cbBuffer);
           _THUNK_SHORT (sParmNum);
           _THUNK_CALLI (_emx_32to16(_NetAccessSetInfo)));
}

USHORT NetAccessAdd(PSZ pszServer, USHORT sLevel, 
		    PVOID pbBuffer, USHORT cbBuffer)
{
  return (USHORT)
          (_THUNK_PROLOG (4+2+4+2);
           _THUNK_FLAT (pszServer);
           _THUNK_SHORT (sLevel);
           _THUNK_FLAT (pbBuffer);
           _THUNK_SHORT (cbBuffer);
           _THUNK_CALLI (_emx_32to16(_NetAccessAdd)));
}

static int acl_init(void)
{
  HMODULE netapi;
  char buf[256];

  if (initialized)
    return 0;

  initialized = TRUE;

  if (DosLoadModule(buf, sizeof(buf), "NETAPI", &netapi)) 
    return -1;

  if (DosQueryProcAddr(netapi, 0, "NETACCESSGETINFO", (PFN *) &_NetAccessGetInfo) ||
      DosQueryProcAddr(netapi, 0, "NETACCESSSETINFO", (PFN *) &_NetAccessSetInfo) ||
      DosQueryProcAddr(netapi, 0, "NETACCESSADD", (PFN *) &_NetAccessAdd))
    return -1;

  netapi_avail = TRUE;

  return 0;
}

ACL *acl_alloc(void)
{
  ACL *acl;

  if (acl_init())
    return NULL;

  if ((acl = _tmalloc(sizeof(ACL))) == NULL)
    return NULL;

  memset(acl, 0, sizeof(ACL));

  if ((acl -> data = _tmalloc(BUFFERSIZE)) == NULL)
    return NULL;
  if ((acl -> text = _tmalloc(BUFFERSIZE)) == NULL)
    return NULL;

  return acl;
}

void acl_free(ACL *acl)
{
  if (acl)
  {
    if (acl -> data)
      _tfree(acl -> data);
    if (acl -> text)
      _tfree(acl -> text);

    _tfree(acl);
  }
}

static void acl_mkpath(char *buffer, char *source)
{
  char *ptr;
  static char cwd[CCHMAXPATH];
  static int cwdlen;

  if (isalpha(source[0]) && source[1] == ':')
    buffer[0] = 0; /* fully qualified names */
  else
  {
    if (cwd[0] == 0)
    {
      _getcwd2(cwd, sizeof(cwd));
      cwdlen = strlen(cwd);
    }

    if (source[0] == '/' || source[0] == '\\')
    {
      if (source[1] == '/' || source[1] == '\\')
	buffer[0] = 0; /* UNC names */
      else
      {
	strncpy(buffer, cwd, 2);
	buffer[2] = 0;
      }
    }
    else
    {
      strcpy(buffer, cwd);
      if (cwd[cwdlen - 1] != '\\' && cwd[cwdlen - 1] != '/')
	strcat(buffer, "/");
    }
  }

  strcat(buffer, source);

  for (ptr = buffer; *ptr; ptr++)
    if (*ptr == '/')
      *ptr = '\\';

  strupr(buffer);
}

int acl_load(ACL *acl, char *server, char *resource)
{
  char *path;
  USHORT bytes;
  int rc;

  if (!netapi_avail)
    return -1;

  if ((path = _tmalloc(CCHMAXPATH)) == NULL)
    return -1;

  acl_mkpath(path, resource);

  rc = NetAccessGetInfo(server, path, 1, 
			acl -> data, BUFFERSIZE, &acl -> datalen);

  _tfree(path);

  return rc;
}

int acl_store(ACL *acl, char *server, char *resource)
{
  char *path;
  ACCINFO *ai;
  char *ptr;
  int rc;

  if (!netapi_avail)
    return -1;

  if ((path = _tmalloc(CCHMAXPATH)) == NULL)
    return -1;

  if ((ai = _tmalloc(sizeof(ACCINFO))) == NULL)
    return _tfree(path), -1;

  acl_mkpath(path, resource);

  ai -> acc_resource_name = _emx_32to16(path);
  ai -> acc_attr = 0;
  ai -> acc_count = 0;

  NetAccessAdd(server, 1, ai, sizeof(ACCINFO));
  /* Ignore any errors, most probably because ACL already exists. */
  /* In any such case, try updating the existing ACL. */

  _tfree(ai);

  ai = (ACCINFO *) (acl -> data);
  ai -> acc_resource_name = _emx_32to16(path);

  rc = NetAccessSetInfo(server, path, 1, 
			acl -> data, acl -> datalen, 0);

  _tfree(path);

  return rc;
}

int acl_bin2text(ACL *acl)
{
  ACCINFO *ai;
  ACCLIST *al;
  char *ptr;
  int cnt, offs;

  ai = (ACCINFO *) (acl -> data);
  al = (ACCLIST *) (acl -> data + sizeof(ACCINFO));

  offs = sprintf(acl -> text, "ACL1:%X,%d\n", 
		 ai -> acc_attr, ai -> acc_count);

  for (cnt = 0; cnt < ai -> acc_count; cnt++)
    offs += sprintf(acl -> text + offs, "%s,%X\n", 
		    al[cnt].acl_ugname, al[cnt].acl_access);

  strcpy(acl -> text + offs, _emx_16to32(ai -> acc_resource_name));
  acl -> textlen = strlen(acl -> text);
  acl -> name = ptr;

  return 0;
}

int acl_text2bin(ACL *acl)
{
  ACCINFO *ai;
  ACCLIST *al;
  char *ptr, *ptr2;
  int cnt;

  ai = (ACCINFO *) (acl -> data);
  al = (ACCLIST *) (acl -> data + sizeof(ACCINFO));

  ai -> acc_resource_name = 0; /* set later by acl_store */

  if (sscanf(acl -> text, "ACL1:%hX,%hd", 
	     &ai -> acc_attr, &ai -> acc_count) != 2)
    return ERROR_INVALID_PARAMETER;

  ptr = strchr(acl -> text, '\n') + 1;

  for (cnt = 0; cnt < ai -> acc_count; cnt++)
  {
    ptr2 = strchr(ptr, ',');
    strncpy(al[cnt].acl_ugname, ptr, ptr2 - ptr);
    al[cnt].acl_ugname[ptr2 - ptr] = 0;
    sscanf(ptr2 + 1, "%X", &al[cnt].acl_access);
    ptr = strchr(ptr, '\n') + 1;
  }

  acl -> datalen = sizeof(ACCINFO) + ai -> acc_count * sizeof(ACCLIST);

  return 0;
}

/* end of acl32.c */
