/*
     Program PM-Info: a program for viewing GNU-style hypertext info
     documentation files.
     Copyright (C) 1992,1993,1994  Colin Jensen
     
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

     Contact addresses:
	Colin Jensen
	email: cjensen@netcom.com
	US mail: 4902 Esguerra Terrace, Fremont CA, 94555
*/

#define INCL_PM
#include <os2.h>
#include <stdlib.h>
#include <string.h>
#include "font.h"
#ifndef NDEBUG
#define NDEBUG
#endif
#include "bugme.h"

/*************************************************************/
/**** Class for dealing with HPSs in a reasonable fashion ****/
/*************************************************************/

class TempPS
{
public:
  HPS hps;
  TempPS(HWND window = HWND_DESKTOP);
  ~TempPS();
};

TempPS::TempPS(HWND window)
{
  hps = WinGetPS (window);
}

TempPS::~TempPS()
{
  WinReleasePS (hps);
}

/*************************************************************/
/**** Class for dealing with HDCs in a reasonable fashion ****/
/*************************************************************/

class TempDC
{
  void init (HPS hps);
public:
  HDC hdc;
  TempDC (HPS hps) { init (hps); }
  TempDC (const TempPS &hps) { init (hps.hps); }
  int x_resolution ();
  int y_resolution ();
};

void TempDC::init (HPS hps)
{
  hdc = GpiQueryDevice (hps);
}

int TempDC::x_resolution ()
{
  long xres;
  DevQueryCaps (hdc, CAPS_HORIZONTAL_FONT_RES, 1, &xres);
  return xres;
}

int TempDC::y_resolution ()
{
  long yres;
  DevQueryCaps (hdc, CAPS_VERTICAL_FONT_RES, 1, &yres);
  return yres;
}

/***********************************************************/
/**** Class for simplifying getting at font information ****/
/***********************************************************/

class FontInformation
{
  int font_count;
  FONTMETRICS *font_metrics;
  long xres, yres;
  const FONTMETRICS *find (const char *fontname, int pointsize,
			   unsigned short selection, 
			   unsigned short antiselection);
  static const unsigned short normal_fm_sel;
 public:
  const FONTMETRICS *find (const char *fontname, int pointsize)
    { return find (fontname, pointsize, 0, 0); }
  const FONTMETRICS *find_normal (const char *fontname, int pointsize)
    { return find (fontname, pointsize, 0, normal_fm_sel); }
  const FONTMETRICS *find_bold (const char *fontname, int pointsize)
    { return find (fontname, pointsize, FM_SEL_BOLD,
		   normal_fm_sel & ~FM_SEL_BOLD); }
  const FONTMETRICS *find_italic (const char *fontname, int pointsize)
    { return find (fontname, pointsize, FM_SEL_ITALIC, 
		   normal_fm_sel & ~FM_SEL_ITALIC); }
  FontInformation();
  ~FontInformation();
};

FontInformation::FontInformation()
{
  TempPS hps;

  // Count up how many fonts are availible in the system
  long foobar = 0;
  font_count = GpiQueryFonts (hps.hps, QF_PUBLIC, 0, &foobar, 0, 0);

  // Allocate room for all the font information
  font_metrics = (FONTMETRICS *) malloc (font_count * sizeof (*font_metrics));

  // Then redo the query to snarf information about all the fonts
  foobar = font_count;
  GpiQueryFonts (hps.hps, QF_PUBLIC, 0,
		 &foobar, sizeof (*font_metrics), font_metrics);

  // Grab information about the PS resolution for later use
  TempDC hdc (hps);
  xres = hdc.x_resolution ();
  yres = hdc.y_resolution ();
}

FontInformation::~FontInformation()
{
  if (font_metrics)
    free (font_metrics);
}

const unsigned short FontInformation::normal_fm_sel
= (FM_SEL_ITALIC 
   | FM_SEL_UNDERSCORE 
   | FM_SEL_NEGATIVE
   | FM_SEL_OUTLINE
   | FM_SEL_STRIKEOUT
   | FM_SEL_BOLD);



const FONTMETRICS *FontInformation::find (const char *fontname, int pointsize,
					  unsigned short sel,
					  unsigned short antisel)
{
  int i;

#if 0
  BUGME(("FontInformation::find(`%s', %d, %X, %X)", fontname, pointsize,
	 sel, antisel));
  BUGME(("font count: %d", font_count));
  BUGME(("X Resolution: %d", xres));
  BUGME(("Y Resolution: %d", yres));

  for (i = 0; i < font_count; i++)
    {
      FONTMETRICS *fm = font_metrics + i;
      BUGME(("Font #%d", i));
#define N(field) BUGME(("    Field " #field " = %d", fm->field))
#define H(field) BUGME(("    Field " #field " = %X", fm->field))
#define S(field) BUGME(("    Field " #field " = `%s'", fm->field))
      S(szFamilyname);
      S(szFacename);
      N(sNominalPointSize);
      H(fsSelection);
      H(fsDefn);
      N(sXDeviceRes);
      N(sYDeviceRes);
    }
#endif


  // Search for an exact match bitmap font
  BUGME(("%s:%d: font_metrics is at %p", __FILE__, __LINE__, font_metrics));
  for (i = 0; i < font_count; i++)
    {
      FONTMETRICS *fm = font_metrics + i;
      // Reject inappropriate fonts
      if (strcmp (fm->szFacename, fontname)      // wrong font
	  || (fm->fsSelection & sel) != sel      // wrong selections
	  || (fm->fsSelection & antisel)	 // wrong selections
	  || (fm->fsDefn & FM_DEFN_OUTLINE)      // vector font
	  || fm->sXDeviceRes != xres             // wrong x resolution
	  || fm->sYDeviceRes != yres             // wrong y resolution
	  || fm->sNominalPointSize != pointsize) // wrong point size
	continue;
      BUGME(("FontInformation::find: Selecting font #%d (bitmap)", i));
      BUGME(("pointer is %p", fm));
      return fm;
    }

  // Search for an exact match vector font
  for (i = 0; i < font_count; i++)
    {
      FONTMETRICS *fm = font_metrics + i;
      // Reject inappropriate fonts
      if (strcmp (fm->szFacename, fontname)      // wrong font
	  || (fm->fsSelection & sel) != sel      // wrong selections
	  || (fm->fsSelection & antisel)	 // wrong selections
	  || !(fm->fsDefn & FM_DEFN_OUTLINE)     // bitmap font
	  || fm->sNominalPointSize != pointsize) // wrong point size
	continue;
      BUGME(("FontInformation::find: Selecting font #%d (exact vector)", i));
      return fm;
    }

  // Search for an acceptible vector font
  // (We could try harder to find the best match amongst this group)
  for (i = 0; i < font_count; i++)
    {
      FONTMETRICS *fm = font_metrics + i;
      // Reject inappropriate fonts
      if (strcmp (fm->szFacename, fontname)      // wrong font
	  || (fm->fsSelection & sel) != sel      // wrong selections
	  || (fm->fsSelection & antisel)	 // wrong selections
	  || !(fm->fsDefn & FM_DEFN_OUTLINE)     // bitmap font
	  || pointsize < fm->sMinimumPointSize   // font too big
	  || pointsize > fm->sMaximumPointSize)  // font too small
	continue;
      BUGME(("FontInformation::find: Selecting font #%d (good vector)", i));
      return fm;
    }

  // Search for any matching vector font - we'll forcibly scale it
  for (i = 0; i < font_count; i++)
    {
      FONTMETRICS *fm = font_metrics + i;
      // Reject inappropriate fonts
      if (strcmp (fm->szFacename, fontname)      // wrong font
	  || (fm->fsSelection & sel) != sel      // wrong selections
	  || (fm->fsSelection & antisel)	 // wrong selections
	  || !(fm->fsDefn & FM_DEFN_OUTLINE))    // bitmap font
	continue;
      BUGME(("FontInformation::find: Selecting font #%d (poor vector match)", 
	     i));
      return fm;
    }

  // If the requester was looking for a BOLD font, try to return
  // a non-bold version
  if ((sel & FM_SEL_BOLD) && !(antisel & FM_SEL_BOLD))
    {
      const FONTMETRICS *fm = find (fontname, pointsize, sel & ~FM_SEL_BOLD,
				    antisel & FM_SEL_BOLD);
      if (fm)
	return fm;
    }

  // If the requester was looking for a ITALIC font, try to return
  // a non-italic version
  if ((sel & FM_SEL_ITALIC) && !(antisel & FM_SEL_ITALIC))
    {
      const FONTMETRICS *fm = find (fontname, pointsize, sel & ~FM_SEL_ITALIC,
				    antisel & FM_SEL_ITALIC);
      if (fm)
	return fm;
    }

  // If the requester was looking for a BOLD ITALIC font, try to return
  // a non-bold-italic version
#define BOLDIT (FM_SEL_ITALIC | FM_SEL_BOLD)
  if ((sel & BOLDIT) && !(antisel & BOLDIT))
    {
      const FONTMETRICS *fm = find (fontname, pointsize, sel & ~BOLDIT,
				    antisel & BOLDIT);
      if (fm)
	return fm;
    }

  // No match -- too damn bad
  BUGME(("FontInformation::find: No matchable font found"));
  return 0;
}

static FontInformation *font_information;

static void font_information_init ()
{
  static once = 0;
  if (once)
    return;
  once = 1;
  BUGME(("%s:%d: initializing font information", __FILE__, __LINE__));
  font_information = new FontInformation;
}

/***********************************************************/
/**** Class for keeping track of which LCIDs are in use ****/
/***********************************************************/

LCID_Tracker::LCID_Tracker ()
{
  lcid_in_use[0] = 1;
  int i;
  for (i = 1; i < 256; i++)
    lcid_in_use[i] = 0;
}

int LCID_Tracker::get ()
{
  int i;
  for (i = 1; i < 256; i++)
    {
      if (!lcid_in_use[i])
	{
	  lcid_in_use[i] = 1;
	  BUGME(("%s:%d: LCID_Tracker::get: returning %d",
		 __FILE__, __LINE__, i));
	  return i;
	}
    }
  BUGME(("%s:%d: LCID_Tracker::get: returning -1", __FILE__, __LINE__));
  return -1;
}

void LCID_Tracker::free (int lcid)
{
  if (1 <= lcid && lcid <= 255)
    lcid_in_use[lcid] = 0;
}

/*************************************************************/
/**** A class for storing all the fonts we want in an HPS ****/
/*************************************************************/

FontList::FontList ()
: metrics ( (FATTRS *) malloc (sizeof (FATTRS) * 256) ),
  fmetrics ( (FONTMETRICS *) malloc (sizeof (FONTMETRICS) * 256) ),
  pointsizes ( (int *) malloc (sizeof (int) * 256) )
{
}

FontList::~FontList ()
{
  if (metrics) ::free (metrics);
  if (fmetrics) ::free (fmetrics);
  if (pointsizes) ::free (pointsizes);
}

int FontList::get (const FONTMETRICS *fm, int pointsize)
{
  FATTRS fa;

  // Convert the fm into a suitable fa
  memset (&fa, 0, sizeof (fa));
  fa.usRecordLength = sizeof (fa);
  fa.lMatch = fm->lMatch;
  strcpy (fa.szFacename, fm->szFacename);
  fa.idRegistry = fm->idRegistry;
  if (!(fm->fsDefn & FM_DEFN_OUTLINE)) 
    {
      fa.lMaxBaselineExt = fm->lMaxBaselineExt;
      fa.lAveCharWidth = fm->lAveCharWidth;
    }
  else
    fa.fsType = FATTR_TYPE_KERNING;

  // Obtain an lcid to use
  int id = ancestor::get ();
  if (id == -1)
    return LCID_DEFAULT;

  // Check that this is an allowable font selection
  TempPS hps;
  int rc = GpiCreateLogFont (hps.hps, 0, id, &fa);
  if (rc != FONT_MATCH)
    {
      free (id);
      return LCID_DEFAULT;
    }

  // Save away the metrics for future use
  metrics[id] = fa;
  fmetrics[id] = *fm;
  pointsizes[id] = pointsize;

  return id;
}

void FontList::restore (HPS hps)
{
  int i;
  BUGME(("FontList::restore: entry"));
  for (i = 1; i < 256; i++)
    if (lcid_in_use[i])
      {
	BUGME(("Restoring logical font #%d", i));
	GpiCreateLogFont (hps, 0, i, &metrics[i]);
      }
  BUGME(("FontList::restore: exit"));
}

int FontList::make_font (const char *fontname, int pointsize)
{
  BUGME(("%s:%d: FontList::make_font (`%s', %d)", __FILE__, __LINE__, 
	 fontname, pointsize));
  font_information_init ();
  const FONTMETRICS *fm = font_information->find_normal (fontname, pointsize);
  if (fm == 0)
    return LCID_DEFAULT;
  return get (fm, pointsize);
}

int FontList::make_italic_font (const char *fontname, int pointsize)
{
  font_information_init ();
  const FONTMETRICS *fm = font_information->find_italic (fontname, pointsize);
  if (fm == 0)
    return LCID_DEFAULT;
  return get (fm, pointsize);
}

int FontList::make_bold_font (const char *fontname, int pointsize)
{
  font_information_init ();
  const FONTMETRICS *fm = font_information->find_bold (fontname, pointsize);
  if (fm == 0)
    return LCID_DEFAULT;
  return get (fm, pointsize);
}

void FontList::select_font (HPS hps, int lcid)
{
  if (lcid < 0 || 255 < lcid || !lcid_in_use[lcid])
    return; // Invalid ID
  GpiSetCharSet (hps, lcid);

  // For vector fonts, arrange all the hps settings appropriately
  FONTMETRICS *fm = &fmetrics[lcid];

  // Scale to the correct point size
  if (fm->fsDefn & FM_DEFN_OUTLINE)
    {
      TempDC hdc(hps);
      SIZEF box;
      box.cx = ((hdc.x_resolution () * pointsizes[lcid]) << 16) / 720;
      box.cy = ((hdc.y_resolution () * pointsizes[lcid]) << 16) / 720;
      GpiSetCharBox (hps, &box);
    }
}
