
/*
     Program PM-Info: a program for viewing GNU-style hypertext info
     documentation files.
     Copyright (C) 1992,1993,1994  Colin Jensen
     
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

     Contact addresses:
	Colin Jensen
	email: cjensen@netcom.com
	US mail: 4902 Esguerra Terrace, Fremont CA, 94555
*/

// Don't even think about compiling this file without a BSD/Torek stdio.h!

#define INCL_DOSPROCESS
#define INCL_DOSQUEUES
#define INCL_DOSFILEMGR
#include <os2.h>
#include <stdio.h>
#include <memory.h>
#include <stdlib.h>
#include "fzoo.h"
#include "bugme.h"

FILE *fopen_cmd_output(const char **cmd, int *return_code);

int file_exists(const char *filename, const char *extension)
{
    char fullname[strlen(filename) + strlen(extension) + 1];
    sprintf(fullname, "%s%s", filename, extension);
    FILE *f = fopen(fullname, "rb");
    if (f == NULL) return 0;
    fclose(f);
    return 1;
}

FILE *fopen_zoo(const char *zfile, const char *afile)
{
    if (zfile == NULL || afile == NULL) return NULL;
    if (!file_exists(zfile, ".zoo")) return NULL;
    // I would like to strangle whoever ported zoo!
    // It ignores everything but its first argument
    // so I have to mash all the arguments in the icky way cmd.exe does
    char *zooargs_fmt = "xpq %s.zoo %s";
    char zooargs[strlen(zooargs_fmt) - 4 + strlen(zfile) + strlen(afile) + 1];
    sprintf(zooargs, zooargs_fmt, zfile, afile);
    const char *cmd[] = {"zoo.exe", "zoo", zooargs, NULL};
    int return_code;
    FILE *zoo_file = fopen_cmd_output(cmd, &return_code);
    if (zoo_file == NULL) return NULL;
    if (return_code != 0) {
	BUGME(("fopen_zoo: bad return code"));
	fclose(zoo_file);
	return NULL;
    }
    return zoo_file;
}

FILE *fopen_zip(const char *zfile, const char *afile)
{
    if (zfile == NULL || afile == NULL) return NULL;
    if (!file_exists(zfile, ".zip")) return NULL;
    char *zipargs_fmt = "-c %s.zip %s";
    char zipargs[strlen(zipargs_fmt) - 4 + strlen(zfile) + strlen(afile) + 1];
    sprintf(zipargs, zipargs_fmt, zfile, afile);
    const char *cmd[] = {"unzip.exe", "unzip", zipargs, NULL};
    int return_code;
    FILE *zip_file = fopen_cmd_output(cmd, &return_code);
    if (zip_file == NULL) return NULL;
    if (return_code != 0) {
	BUGME(("fopen_zip: bad return code"));
	fclose(zip_file);
	return NULL;
    }
    return zip_file;
}

static FILE *Z(const char *file, const char *progname, const char *args)
{
    char program[strlen(progname)+5];
    sprintf(program, "%s.exe", progname);
    char actual_args[strlen(args) + strlen(file) + 1];
    sprintf(actual_args, args, file);
    const char *cmd[] = {program, progname, actual_args, NULL};
    int return_code;
    FILE *zfile = fopen_cmd_output(cmd, &return_code);
    if (zfile == NULL) return NULL;
    if (return_code != 0) {
	fclose(zfile);
	return NULL;
    }
    return zfile;
}

FILE *fopen_compress(const char *file)
{
    if (file == NULL) return NULL;
    if (!file_exists(file, ".Z")) return NULL;
    FILE *f;
    f = Z(file, "zcat", "%s.Z");
    if (f != NULL) return f;
    f = Z(file, "uncompress", "-c %s.Z");
    if (f != NULL) return f;
    f = Z(file, "uncompre", "-c %s.Z");
    return f;
}

static int calc_cmd_string(const char **cmd)
{
    int i = 0;
    int len = 1;
    while (cmd[i] != NULL) len += 1 + strlen(cmd[i++]);
    return len;
}

static void fmt_cmd_string(char *cmdstr, const char **cmd)
{
    BUGME(("fmt_cmd_string"));
    while (cmd[0] != NULL) {
	BUGME(("fmt_cmd_string: adding \"%s\"", cmd[0]));
	strcpy(cmdstr, cmd[0]);
	cmdstr += strlen(cmdstr) + 1;
	cmd++;
    }
    *cmdstr = '\0';
}

#ifdef DEBUG
static void debug_cmd_string(const char *cmd_string)
{
    BUGME(("debug_cmd_string(...)"));
    while(*cmd_string != '\0') {
	BUGME(("debug_cmd_string: Arg: \"%s\"", cmd_string));
	cmd_string += strlen(cmd_string) + 1;
    }
    BUGME(("debug_cmd_string: End of string"));
}
#endif

typedef struct mbuf_struct MBuf;
struct mbuf_struct {
    int size, position;
    MBuf *next;
    char buf[0];
};

FILE *convert_buf_to_file(MBuf *buffer);

FILE *fopen_cmd_output(const char **cmd, int *return_code)
{
    BUGME(("fopen_cmd_output((\"%s\", ...), ...)", cmd[0]));
    int len = calc_cmd_string(cmd+1);
    BUGME(("fopen_cmd_output: cmd string length is %d", len));
    char cmd_string[len];
    fmt_cmd_string(cmd_string, cmd+1);
    BUGME(("fopen_cmd_output: start of cmd looks like: \"%s\"", cmd_string));
    HFILE pipe_input, pipe_output;
#define PIPESIZE 1024
    int code = DosCreatePipe(&pipe_output, &pipe_input, PIPESIZE);
    if (code != 0) {
	BUGME(("fopen_cmd_output: pipe open failed"));
	return NULL;
    }
    BUGME(("fopen_cmd_output: pipe opened: %d -> %d", 
	   pipe_input, pipe_output));

    char *piper = "\n";
    ULONG written;
    code = DosWrite(pipe_input, piper, strlen(piper), &written);
    if (code != 0) {
	BUGME(("fopen_cmd_output: WARNING: DosWrite returned a %d", code));
    } else if (written != strlen(piper)) {
	BUGME(("fopen_cmd_output: WARNING: "
	       "DosWrite wrote %d bytes (%d requested)",
	       written, strlen(piper)));
    }

    HFILE stdout_handle = 1;
    
    // Save the stdout handle so we can restore it later
    // even though we could care less about stdout, if stdout ever gets
    // closed, OS/2 will reuse the handle next time we DosCreatePipe!
    HFILE saved_stdout_handle = 0;
    saved_stdout_handle = ~saved_stdout_handle;
    code = DosDupHandle(stdout_handle, &saved_stdout_handle);
    if (code != 0) {
	BUGME(("fopen_cmd_output: Error duping stdout:"
	       "DosDupHandle returned %d", code));
	return NULL;
    }

    code = DosDupHandle(pipe_input, &stdout_handle);
    if (code != 0) {
	BUGME(("fopen_cmd_output: failed to dup handle"));
	DosClose(saved_stdout_handle);
	return NULL;
    }
    BUGME(("fopen_cmd_output: dup handle %d to %d", 
	   pipe_input, stdout_handle));
    code = DosClose(pipe_input);
    if (code != 0) {
	BUGME(("fopen_cmd_output: Warning: DosClose returned %d", code));
    }
    RESULTCODES result;
#ifdef DEBUG
    debug_cmd_string(cmd_string);
#endif
    code = DosExecPgm(NULL, 0, EXEC_ASYNCRESULT, (PSZ) cmd_string,
		      NULL, &result, (PSZ) cmd[0]);
    if (code != 0) {
	BUGME(("fopen_cmd_output: Unable to execute program \"%s\" (%d)", 
	       cmd[0], code));
	DosDupHandle(saved_stdout_handle, &stdout_handle);
	DosClose(saved_stdout_handle);
	return NULL;
    }
    BUGME(("fopen_cmd_output: program exec started; pid: %d", 
	   * (int *) &result));

    code = DosDupHandle(saved_stdout_handle, &stdout_handle);
    if (code != 0) {
	BUGME(("fopen_cmd_output: WARNING: Trouble restoring stdout: "
	       "DosDupHandle returned %d", code));
    }
    code = DosClose(saved_stdout_handle);
    if (code != 0) {
	BUGME(("fopen_cmd_output: WARNING: err closing duped stdout: "
	       "DosClose returned %d", code));
    }

    /* Now snarf that sucker into memory */
    MBuf *buffers = NULL;
    int buffered_length = 0;
    BUGME(("fopen_cmd_output: Reading buffers"));
    for(;;) {
	// Do be sure to allocate a *LOT* of stack for yourself
	MBuf *buf;
	buf = (MBuf *) alloca(sizeof(MBuf) + PIPESIZE);
	buf->next = buffers;
	buf->size = 0;
	code = DosRead(pipe_output, (PVOID) &buf->buf, PIPESIZE, 
		       (PULONG) &buf->size);
	if (code != 0) {
	    BUGME(("fopen_cmd_output: Read has a problem: %d", code));
	    break;
	}
	if (buf->size == 0) break;
	buffered_length += buf->size;
	buffers = buf;
    }
    BUGME(("fopen_cmd_output: Read %d total bytes", buffered_length));
    
    code = DosClose(pipe_output);
    if (code != 0) {
	BUGME(("fopen_cmd_output: Warning: DosClose returned %d", code));
    }

    /* Get the exit code of the subprocess */
    PID pid;
    code = DosWaitChild(DCWA_PROCESS, DCWW_WAIT, &result, &pid, 
			* (int *) &result);
    if (code != 0) {
	BUGME(("fopen_cmd_output: WARNING: DosWaitChild returned %d", code));
	*return_code = -1;
    } else if (result.codeTerminate != TC_EXIT) {
	BUGME(("fopen_cmd_output: WARNING: Program exited abnormally: %d",
	       result.codeTerminate));
	*return_code = -1;
    } else {
	BUGME(("fopen_cmd_output: Program exited: Code %d", 
	       result.codeResult));
	*return_code = result.codeResult;
    }

    /* Now take the buffer chain, and make a contiguous region out of it */
    MBuf *memory = (MBuf *) malloc(sizeof(MBuf) + buffered_length);
    if (memory == NULL) {
	BUGME(("fopen_cmd_output: Out of memory: Tried to alloc %d bytes",
	       sizeof(memory) + buffered_length));
	return NULL;
    }
    memory->size = buffered_length;
    memory->next = NULL; // Of course
    char *m = memory->buf + memory->size;
    while(buffers != NULL) {
	m -= buffers->size;
	memcpy(m, buffers->buf, buffers->size);
	buffers = buffers->next;
    }

#ifdef DEBUG
    if (buffered_length < 200) {
	BUGME(("fopen_cmd_out: ================ START OF READ FILE\n"));
	fwrite(memory->buf, memory->size, 1, fnote);
	BUGME(("\nfopen_cmd_out: ================ END OF READ FILE\n"));
    }
#endif

    return convert_buf_to_file(memory);
}


static int mbuf_read(void *buf, char *rbuf, int rsize)
{
    BUGME(("mbuf_read(%X, %X, %d)", buf, rbuf, rsize));
    MBuf *mbuf = (MBuf *) buf;
    if (mbuf->size == 0) return 0;
    int maxsize = mbuf->size - mbuf->position;
    if (rsize > maxsize) rsize = maxsize;
    memcpy(rbuf, mbuf->buf + mbuf->position, rsize);
    mbuf->position += rsize;
    BUGME(("mbuf_read: actual size read: %d", rsize));
    return rsize;
}

static fpos_t mbuf_seek(void *buf, fpos_t position, int seek_type)
{
    BUGME(("mbuf_seek(%X, %d, %d)", buf, position, seek_type));
    MBuf *mbuf = (MBuf *) buf;
    if (mbuf->size == 0) return 0;
    int new_position;
    switch(seek_type) {
    case SEEK_SET:
	new_position = position;
	break;
    case SEEK_CUR:
	new_position = mbuf->position + position;
	break;
    case SEEK_END:
	new_position = mbuf->size - position;
	break;
    default:
	return -1;
    }
    if (new_position < 0) position = 0;
    if (new_position >= mbuf->size) new_position = mbuf->size - 1;
    return mbuf->position = new_position;
}

static int mbuf_close(void *buf)
{
    BUGME(("mbuf_close(%X)", buf));
    free(buf);
    return 0;
}

FILE *convert_buf_to_file(MBuf *buffer)
{
    BUGME(("convert_buf_to_file(%X)", buffer));
    buffer->position = 0;
    FILE *f = funopen(buffer, mbuf_read, NULL, mbuf_seek, mbuf_close);
    if (f == NULL) {
	BUGME(("convert_buf_to_file: failed"));
	free(buffer);
    }
    BUGME(("convert_buf_to_file: returning"));
    return f;
}
