/*
     Library PMIO: A library for stdio/pcio on a PM screen
     Copyright (C) 1994  Colin Jensen
     
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

     Contact addresses:
	Colin Jensen
	email: cjensen@netcom.com
	US mail: 4902 Esguerra Terrace, Fremont CA, 94555
*/

#define INCL_WIN
#define INCL_GPI
#define INCL_DOSSEMAPHORES
#include <os2.h>

#include "pmman.h"
#include "bugme.h"
#include "stdwin.h"
#include "font.h"
#include "frame.h"
#include "pmio3.h"

#define max(a,b) (((a) > (b)) ? (a) : (b))
#define min(a,b) (((a) < (b)) ? (a) : (b))
#define LIMIT(v,a,b) (v) = max((a),min((b),(v)))

static int pccolor_to_pmcolor (int color)
{
  static int pmcolors[] =
    {
      CLR_BLACK,
      CLR_DARKBLUE,
      CLR_DARKGREEN,
      CLR_DARKCYAN,
      CLR_DARKRED,
      CLR_DARKPINK,
      CLR_BROWN,
      CLR_NEUTRAL,
      CLR_DARKGRAY,
      CLR_BLUE,
      CLR_GREEN,
      CLR_CYAN,
      CLR_RED,
      CLR_PINK,
      CLR_YELLOW,
      CLR_WHITE
      };
  return pmcolors [color & 0x0F];
}


MyWin::MyWin(IO_Interface *ainterface)
    : StdWinPlus(),
      view (this),
      eventq_size(0),
      eventq_head(0),
      eventq_tail(0),
      interface (ainterface),
      delay_output(0)
{
}


void MyWin::Init(HAB hab)
{
    scroll_position = 0;
    StdWin::Init(hab, "StdWin", CS_SIZEREDRAW,
		 FCF_TITLEBAR | FCF_SYSMENU | FCF_SIZEBORDER | FCF_MINBUTTON
		 | FCF_SHELLPOSITION | FCF_TASKLIST | FCF_ICON);
}


void MyWin::RefreshLine(int line, int min, int max)
{
  RECTL rect;
  line -= scroll_position;
  rect.xLeft = min * char_width;
  rect.xRight = (max+1) * char_width;
  rect.yTop = client_height - char_height * line;
  rect.yBottom = client_height - char_height * (line + 1);
  WinInvalidateRect(window, &rect, FALSE);
}

MRESULT MyWin::MsgVscroll(USHORT, SHORT slider, USHORT cmd, int *)
{
    SHORT new_scroll_position = scroll_position;

    switch(cmd) {
    case SB_LINEUP:
	new_scroll_position--;
	break;
    case SB_LINEDOWN:
	new_scroll_position++;
	break;
    case SB_PAGEUP:
	new_scroll_position -= max(1,client_char_height);
	break;
    case SB_PAGEDOWN:
	new_scroll_position += max(1,client_char_height);
	break;
    case SB_SLIDERTRACK:
	new_scroll_position = slider;
	break;
    }
    LIMIT(new_scroll_position, 0, max_scroll_position);
    if (new_scroll_position != scroll_position) {
	WinScrollWindow(window, 0, 
	    (new_scroll_position - scroll_position) * char_height,
	    NULL,NULL,0,NULL, SW_INVALIDATERGN);
	scroll_position = new_scroll_position;
	WinSendMsg(vscrollbar, SBM_SETPOS,
	    MPFROMSHORT(scroll_position), NULL);
	WinUpdateWindow(window);
    }
    return 0;
}

MRESULT MyWin::MsgPresParamChanged(MPARAM, MPARAM, 
				   int *dodefault)
{
    GetFontInfo();
    client_char_height = client_height / char_height;
    recalc_max_scroll_position();
    recalc_scrollbar();
    WinInvalidateRect(window, NULL, FALSE);
    *dodefault = 1;
    return 0;
}

MRESULT MyWin::MsgChar(MPARAM mp1, MPARAM mp2, int *)
{
  interface->post ();
  if (eventq_size >= MAXEVENTS)
    return 0;
  Event *e = &eventq[eventq_head++];
  eventq_size++;
  if (eventq_head >= MAXEVENTS)
    eventq_head = 0;
  e->flags = SHORT1FROMMP(mp1);
  e->repeat = CHAR3FROMMP(mp1);
  e->scancode = CHAR4FROMMP(mp1);
  e->ch = SHORT1FROMMP(mp2);
  e->vk = SHORT2FROMMP(mp2);
  return 0;
}

void MyWin::flush_dirty_lines ()
{
  if (delay_output)
    return;
  char min, max;
  for (int y = 0; y < view->height(); y++)
    if (view->getdirt(y, &min, &max))
      RefreshLine (y, min, max);
}

MRESULT MyWin::MsgPaint(int *)
{
    RECTL rect;
    HPS hps = WinBeginPaint(window, 0, &rect);
    font_list.restore (hps);
    font_list.select_font (hps, font_id);
    POINTL pt;
    int top,bottom;
    int i;

    GpiErase(hps);
    top = (client_height - rect.yTop) / char_height;
    bottom = (client_height - rect.yBottom) / char_height + 1;
    pt.x = 0;
    pt.y = client_height - char_height * (1+top)+ char_descender;
    top += scroll_position;
    bottom += scroll_position;
    LIMIT(top,scroll_position,view->height());
    LIMIT(bottom,scroll_position,view->height());
    const int width = view->width ();
    CHARBUNDLE charattr;
    charattr.usBackMixMode = FM_OVERPAINT;
    GpiSetAttrs (hps, PRIM_CHAR, CBB_BACK_MIX_MODE, 0, (PBUNDLE) &charattr);
    for (i = top; i < bottom; i++) 
      {
	const char *line = view->line (i);
	const char *attr = view->attrs (i);
	GpiMove(hps, &pt);
	int next_segment;
	for (int j = 0; j < width; j = next_segment)
	  {
	    charattr.lColor = pccolor_to_pmcolor (attr[j] & 0x0F);
	    charattr.lBackColor = pccolor_to_pmcolor (attr[j] >> 4);
	    GpiSetAttrs (hps, PRIM_CHAR, CBB_COLOR | CBB_BACK_COLOR, 0, 
			 (PBUNDLE) &charattr);
	    // Calculate the start of the next piece of text that
	    // will require different attributes than the ones we just set
	    for (next_segment = j+1; next_segment < width; next_segment++)
		if (attr[j] != attr[next_segment])
		  break;
#if 0
	    GpiCharString(hps, next_segment - j, (PSZ) line + j);
#else
	    // In *some* modes of my S3 adapter using Courier or VIO fonts,
	    // GpiCharString royally screws up the current position.
	    // I suspect this is an S3 driver bug since the behavior is not
	    // consistent (i.e. two adjacent identical lines do not
	    // misbehave in the same way).
	    // At any rate, the workaround is to not rely on the GpiCharString
	    // setting the "current position" attributute correctly; instead
	    // we calculate the current position explicitly. -- cjensen
	    POINTL pts[TXTBOX_COUNT];
	    POINTL cur_pos;
	    GpiQueryTextBox (hps, next_segment - j, (PSZ) line + j,
			     TXTBOX_COUNT, pts);
	    GpiQueryCurrentPosition (hps, &cur_pos);
	    GpiCharString(hps, next_segment - j, (PSZ) line + j);
	    cur_pos.x += pts[TXTBOX_TOPRIGHT].x;
	    GpiSetCurrentPosition (hps, &cur_pos);
#endif
	  }
	pt.y -= char_height;
    }
    WinEndPaint(hps);
    return 0;
}

void MyWin::recalc_scrollbar(void)
{
    WinSendMsg(vscrollbar, SBM_SETSCROLLBAR, MPFROM2SHORT(scroll_position, 0), 
	MPFROM2SHORT(0, max_scroll_position));
    WinSendMsg(vscrollbar, SBM_SETTHUMBSIZE, 
	       MPFROM2SHORT(client_char_height, view->height()), 0);
    WinEnableWindow(vscrollbar, (max_scroll_position == 0) ? FALSE : TRUE);
}

MRESULT MyWin::MsgSize(SHORT, SHORT, SHORT scxnew, SHORT scynew, int *)
{
  client_height = scynew;
  client_width = scxnew;
  client_char_height = client_height / char_height;
  recalc_max_scroll_position();
  recalc_scrollbar();
  return 0;
}

void MyWin::GetFontInfo()
{
    HPS hps;
    FONTMETRICS fm;

    hps = WinGetPS(window);
    font_list.restore (hps);
    font_list.select_font (hps, font_id);
    GpiQueryFontMetrics(hps, sizeof(fm), &fm);
    WinReleasePS(hps);

    char_width = fm.lAveCharWidth;
    char_height = fm.lMaxBaselineExt;
    char_descender = fm.lMaxDescender;
}

MRESULT MyWin::MsgCreate(PVOID, PCREATESTRUCT, int *)
{
  // Initialize the external interface
  interface->window = window;
  interface->post ();
  
  // Then release the calling process

  font_id = font_list.make_font ("System VIO", 120);
  if (font_id == LCID_DEFAULT)
    font_id = font_list.make_font ("Courier", 100);
  BUGME(("Font id is %d", font_id));

  vscrollbar = WinWindowFromID(WinQueryWindow(window, QW_PARENT), 
			       FID_VERTSCROLL);
  GetFontInfo();

  frame_control.set_absolute_size (view.width () * char_width,
				   view.height () * char_height);
  frame_control.attach (WinQueryWindow (window, QW_PARENT));

  return 0;
}

int MyWin::physical_scroll (int count)
{
  WinScrollWindow (window, 0, count * char_height,
		   NULL, NULL, 0, NULL, SW_INVALIDATERGN);
  return 1;
}
