
/*
     Program PM-Info: a program for viewing GNU-style hypertext info
     documentation files.
     Copyright (C) 1992,1993,1994  Colin Jensen
     
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

     Contact addresses:
	Colin Jensen
	email: cjensen@netcom.com
	US mail: 4902 Esguerra Terrace, Fremont CA, 94555
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#define INCL_GPIPRIMITIVES
#include <os2.h>

#ifndef NDEBUG
//#define NDEBUG
#endif
#include "bugme.h"
#include "history.h"
#include "attr.h"
#include "view.h"
#include "infofile.h"
#include "fzoo.h"


char *expand_tabs(char *buf)
{
    char *p;
    int strcount;
    char *dp, *sp;

    // Trim off trailing nl and cr
    p = buf + strlen(buf) - 1;
    while (p >= buf && isspace(*p)) *p-- = '\0';
    // Decide how many chars this is, after tab expansion
    for (strcount = 0, p = buf; *p != '\0'; p++) {
	if (*p == '\t') {
	    strcount += 8 - ((p - buf) % 8);
	} else {
	    strcount++;
	}
    }
    // Alloc a permanent place for the expanded string
    p = (char *) malloc(strcount+1);
    // Expand the string into its permanent destination
    for (sp = buf, dp = p; *sp != '\0'; sp++) {
	if (*sp == '\t') {
	    strcount = 8 - ((sp - buf) % 8);
	    while (strcount-- > 0) *dp++ = ' ';
	} else {
	    *dp++ = *sp;
	}
    }
    *dp = '\0';
    return p;
}

char *fgetline_noexpand(FILE *f)
{
    char buf[1024];
    char *p;
    p = fgets(buf, sizeof(buf), f);
    if (p == NULL) return NULL;
    return strdup(buf);
}

char *fgetline(FILE *f)
{
    char *s = fgetline_noexpand(f);
    if (s == NULL) return NULL;
    char *r = expand_tabs(s);
    free(s);
    return r;
}


int IsFile(const char *name)
{
    BUGME(("IsFile(``%s'')", name));
    FILE *f = fopen(name, "rb");
    if (f == NULL) {
	BUGME(("IsFile: file not found"));
	return 0;
    }
    fclose(f);
    BUGME(("IsFile: file exists"));
    return 1;
}

int IsInfoFile(const char *name)
{
    int len = strlen(name);
    char namebuf[len+5];
    strcpy(namebuf, name);
    char *ext = namebuf + len;
    if (IsFile(namebuf)) return 1;
    strcpy(ext, ".zoo");
    if (IsFile(namebuf)) return 1;
    strcpy(ext, ".zip");
    if (IsFile(namebuf)) return 1;
    strcpy(ext, ".Z");
    if (IsFile(namebuf)) return 1;
    strcpy(ext, ".z");  // Just in case IBM gets smarter
    if (IsFile(namebuf)) return 1;
    return 0;
}

FILE *infoopen_2(const char *part_filename, const char *filename,
		 const char *search_path)
{
    if (part_filename == NULL || filename == NULL) return NULL;

    char part_with_path[strlen(search_path) + strlen(part_filename) + 1];
    char file_with_path[strlen(search_path) + strlen(filename) + 1];
    sprintf(part_with_path, "%s%s", search_path, part_filename);
    sprintf(file_with_path, "%s%s", search_path, filename);

    FILE *f;
    f = fopen(part_with_path, "rt");
    if (f != NULL) return f;

    f = fopen_zoo(file_with_path, part_filename);
    if (f != NULL) return f;

    f = fopen_zip(file_with_path, part_filename);
    if (f != NULL) return f;

    f = fopen_compress(part_with_path);
    if (f != NULL) return f;

    return NULL;
}

typedef struct path_list_struct PathList;
struct path_list_struct {
    PathList *next;
    const char *path;
};

PathList *path_list = NULL;

char *most_preferred_info_directory = NULL;
extern const char *default_file_name;

static void force_info_path(const char *filename)
{
    // First, parse out the filename and pathname components
    char *last_slash = strchr(filename, '\\');
    const char *last_colon = (filename[0] == '\0') ? NULL 
	: (filename[1] == ':') ? filename+1 : NULL;
    char dirname_buf[strlen(filename)];
    char *dirname = dirname_buf;
    if (last_slash != NULL) {
	int path_len = last_slash - filename;
	strcpy(dirname, filename);
	dirname[path_len] = '\0';
	filename += path_len;
    } else if (last_colon != NULL) {
	strcpy(dirname, filename);
	filename = last_colon+1;
	dirname[2] = '\0';
    } else {
	filename = dirname;
	dirname = "";
    }
    most_preferred_info_directory = strdup(dirname);

    char *extention = strstr(filename, ".zoo");
    if (extention != NULL && strlen(extention) == 4) {
	*extention = '\0';
	default_file_name = strdup(filename);
	return;
    }

    extention = strstr(filename, ".zip");
    if (extention != NULL && strlen(extention) == 4) {
	*extention = '\0';
	default_file_name = strdup(filename);
	return;
    }

    extention = strstr(filename, ".Z");
    if (extention != NULL && strlen(extention) == 2) {
	*extention = '\0';
	default_file_name = strdup(filename);
	return;
    }

    extention = strstr(filename, ".z");
    if (extention != NULL && strlen(extention) == 2) {
	*extention = '\0';
	default_file_name = strdup(filename);
	return;
    }

    default_file_name = strdup(filename);
}

void add_info_file_or_path(const char *new_path)
{
    BUGME(("add_info_file_or_path(``%s'')", new_path));
    if (new_path == NULL) return;
    if (IsInfoFile(new_path)) {
	BUGME(("add_info_file_or_path: forcing path"));
	force_info_path(new_path);
	return;
    }
    add_info_path(new_path);
}

void add_info_path(const char *new_path)
{
    BUGME(("add_info_path(``%s'')", new_path));
    if (new_path == NULL) return;
    char new_path_tmp[strlen(new_path) + 2];
    const char *end_of_path = new_path + strlen(new_path) - 1;
    if (end_of_path > new_path 
	&& *end_of_path != '\\' && *end_of_path != ':') {
	sprintf(new_path_tmp, "%s/", new_path);
	new_path = new_path_tmp;
    }
    PathList *path_element = (PathList *) malloc(sizeof(PathList));
    path_element->next = path_list;
    path_element->path = strdup(new_path);
    path_list = path_element;
}

FILE *infoopen(const char *part_filename, const char *filename)
{
    static int once = 0;
    if (!once) {
	once = 1;
	// Place current directory last in search path for info files
	add_info_path("");
	// If we have a specially requested info directory, let it be last
	if (most_preferred_info_directory != NULL) {
	    add_info_path(most_preferred_info_directory);
	}
    }
    PathList *path;
    for (path = path_list; path != NULL; path = path->next) {
	FILE *f = infoopen_2(part_filename, filename, path->path);
	if (f != NULL) return f;
    }
    return NULL;
}

FILE *InfoFile::infoopen(const char *part_filename, const char *filename)
{
    if (cache_infofile != NULL
	&& cache_filename != NULL
	&& cache_part_filename != NULL
	&& !strcmp(filename, cache_filename)
	&& !strcmp(part_filename, cache_part_filename)) {
	// We already have the right file open.  So just rewind it.
	fseek(cache_infofile, 0L, SEEK_SET);
	return cache_infofile;
    } else {
	if (cache_filename != NULL) {
	    free(cache_filename);
	    cache_filename = NULL;
	}
	if (cache_part_filename != NULL) {
	    free(cache_part_filename);
	    cache_part_filename = NULL;
	}
	if (cache_infofile != NULL) {
	    fclose(cache_infofile);
	}
	cache_infofile = ::infoopen(part_filename, filename);
	if (cache_infofile != NULL) {
	    cache_filename = strdup(filename);
	    cache_part_filename = strdup(part_filename);
	}
	return cache_infofile;
    }
}

// Currently, we're using a non-ANSI toupper
#define Toupper(c) (isalpha(c) ? toupper(c) : (c))

int nodename_ncmp(const char *n1, const char *n2, int length)
{
    if (n1 == NULL) return (n2 == NULL) ? 0 : 1;
    if (n2 == NULL) return -1;
    while (length > 0 && *n1 != '\0' && *n2 != '\0') {
	char c1 = Toupper(*n1);
	char c2 = Toupper(*n2);
	if (c1 > c2) return 1;
	if (c2 > c1) return -1;
	length--;
	n1++;
	n2++;
    }
    if (length == 0) return 0;
    if (*n1 == '\0' && *n2 == '\0') return 0;
    if (*n1 == '\0') return -1;
    return 1;
}

int nodename_cmp(const char *n1, const char *n2)
{
    if (n1 == NULL) return (n2 == NULL) ? 0 : 1;
    if (n2 == NULL) return -1;
    while (*n1 != '\0' && *n2 != '\0') {
	char c1 = Toupper(*n1);
	char c2 = Toupper(*n2);
	if (c1 > c2) return 1;
	if (c2 > c1) return -1;
	n1++;
	n2++;
    }
    if (*n1 == '\0' && *n2 == '\0') return 0;
    if (*n1 == '\0') return -1;
    return 1;
}

char *extract_nodename(const char *nodeline)
{
    BUGME(("extract_nodename(\"%s\")", nodeline));
    if (nodeline == NULL) return NULL;
    while(isspace(*nodeline)) nodeline++;
    const char *start = nodeline;
    if (*nodeline == '(') {
	while(*nodeline != '\0' && *nodeline != ')') nodeline++;
    }
    while (*nodeline != '\0'
	   && strchr("\t\n,.", *nodeline) == NULL) nodeline++;
    int len = nodeline - start;
    char *dup = (char *) malloc(len + 1);
    if (dup == NULL) return NULL;
    memcpy(dup, start, len);
    dup[len] = '\0';
    BUGME(("extract_nodename: returning \"%s\"", dup));
    return dup;
}


SplitList *splitlist_cons(SplitRec split, SplitList *next)
{
    SplitList *list = (SplitList *) malloc(sizeof(SplitList));
    list->split = split;
    list->next = next;
    return list;
}



void InfoFile::open(const char *filename)
{
    if (most_recent_filename != NULL) free(most_recent_filename);
    most_recent_filename = strdup(filename);
    BUGME(("InfoFile::open(%s)", filename));
    infofile = infoopen(filename, filename);
    if (infofile == NULL) {
	okval = 0;
	BUGME(("failed to open"));
	return;
    }
    okval = 1;
    split_file = 0; /* Hopefully... */
    tags_table_availible = 0;
    /* See if we can find an indirect table */
    BUGME(("Looking to see if we have a split file"));
    char buf[1024];
    char *p;
    for(;;) {
	p = fgets(buf, sizeof(buf), infofile);
	if (p == NULL) return;
	if (*p != '\x1F') continue;
	p = fgets(buf, sizeof(buf), infofile);
	if (p == NULL) return;
	const char * const indirect = "Indirect:";
	if (nodename_ncmp(p, indirect, strlen(indirect))) continue;
	/* Drat! It is a split file */
	split_file = 1;
	break;
    }
    BUGME(("Split file detected"));
    char buf2[1024];
    int position;
    int match;
    SplitRec splitrec;
    splitlist = NULL;
    for(;;) {
	p = fgets(buf, sizeof(buf), infofile);
	if (p == NULL || *p == '\x1F') break;
	match = sscanf(buf, "%[^:]: %d", buf2, &position);
	if (match != 2) continue;
	splitrec.filename = strdup(buf2);
	splitrec.position = position;
	splitlist = splitlist_cons(splitrec, splitlist);
    }

#ifdef DEBUG
    SplitList *sl;
    for (sl = splitlist; sl != NULL; sl = sl->next) {
	BUGME(("Subfile ``%s'' at offset %ld", sl->split.filename,
		sl->split.position));
    }
#endif

    // Now look for a tags table -- We will need it for large files
    BUGME(("Seeking tag table"));
    fseek(infofile, 0L, SEEK_SET);
    for(;;) {
	p = fgets(buf, sizeof(buf), infofile);
	if (p == NULL) return;
	if (strchr(p, '\x1F') == NULL) continue;
	p = fgets(buf, sizeof(buf), infofile);
	if (p == NULL) return;
	const char * const tag_table_name = "Tag Table:";
	if (nodename_ncmp(p, tag_table_name, strlen(tag_table_name))) continue;
	// Yes, we have a tag table
	break;
    }
    tags_table_availible = 1;
    BUGME(("Got a tag table"));
    tags_table = NULL;
    SplitRec tag;
    for(;;) {
	p = fgets(buf, sizeof(buf), infofile);
	if (p == NULL || *p == '\x1F') break;
	match = sscanf(buf, "Node: %[^\x7F]\x7F %d", buf2, &position);
	if (match != 2) continue;
	tag.filename = strdup(buf2);
	tag.position = position;
	tags_table = splitlist_cons(tag, tags_table);
    }

#ifdef DEBUG
    for (sl = tags_table; sl != NULL; sl = sl->next) {
	BUGME(("Node ``%s'' at offset %ld", sl->split.filename,
		sl->split.position));
    }
#endif
}

InfoFile::InfoFile(const char *filename)
     : history(32),
       most_recent_filename(NULL),
       cache_infofile(NULL),
       cache_filename(NULL),
       cache_part_filename(NULL)
{
    open(filename);
}

InfoFile::~InfoFile(void)
{
    if (!ok()) return;
    if (cache_infofile != NULL) fclose(cache_infofile);
    if (cache_filename != NULL) free(cache_filename);
    if (cache_part_filename != NULL) free(cache_part_filename);
    if (most_recent_filename != NULL) free(most_recent_filename);
}

View *InfoFile::getnode_attempt(const char *nodename)
{
    char *p, *pnode;
    int lines_searched = 0;
    char buf[1024];

    BUGME(("InfoFile::getnode_attempt(``%s'', ...)", nodename));

    for(;;) {
	// Get a line
	p = fgets(buf, sizeof(buf), infofile);
	if (p == NULL) {
	    BUGME(("InfoFile::getnode_attempt: Hit end of file"));
	    goto attempt_fail;
	}
	lines_searched++;

	// Wait until we have a new page
	if (*p != '\x1F') continue;
	BUGME(("InfoFile::getnode_attempt: Found a page marker"));

	// Get the startup line
	p = fgetline_noexpand(infofile);
	if (p == NULL) {
	    BUGME(("InfoFile::getnode_attempt: Hit end of file"));
	    goto attempt_fail;
	}
	lines_searched++;

	// If it is not a nodeline, try some more
	BUGME(("InfoFile::getnode_attempt: Checking nodeline: ``%s''", p));
	if (nodename_ncmp(p,"File:",5)) {
	    BUGME(("InfoFile::getnode_attempt: Not a nodeline"));
	    free(p);
	    continue;
	}

	// Get the node name
	BUGME(("InfoFile::getnode_attempt: Checking for node name"));
	pnode = strstr(p, "Node:");
	if (pnode == NULL) {
	    BUGME(("InfoFile::getnode_attempt: No nodename on nodeline"));
	    free(p);
	    continue;
	}
	pnode += 5;
	while (isspace(*pnode)) pnode++;

	// Extract the nodename
	char *extract_name = extract_nodename(pnode);
	if (extract_name == NULL) {
	    free(p);
	    continue;
	}

	// Give up if wrong name
	BUGME(("InfoFile::getnode_attempt: Checking nodename"));
	if (nodename_cmp(extract_name, nodename)) {
	    BUGME(("InfoFile::getnode_attempt: Wrong nodename"));
	    free(p);
	    free(extract_name);
	    continue;
	}
	free(extract_name);

	// Otherwise, read the page
	BUGME(("InfoFile::getnode_attempt: Reading page..."));
	View *view = new SimpleView;
	view->append_allocated(expand_tabs(p));
	free(p);
	for(;;) {
	    p = fgetline(infofile);
	    if (p == NULL || *p == '\x1F') break;
	    view->append_allocated(p);
	}
	BUGME(("InfoFile::getnode_attempt: Done."));
	return view;
    }
    BUGME(("InfoFile::getnode_attempt: Done."));
    return NULL;
attempt_fail:
    BUGME(("attempt_fail"));
    return NULL;
}

View *InfoFile::get_last_node()
{
    char *current, *previous;
    current = history.pop();
    if (current == NULL) return NULL;
    previous = history.pop();
    if (previous == NULL) {
	history.push(current);
	free(current);
	return NULL;
    }
    View *view = getnode(previous);
    if (view == NULL) {
	history.push(previous);
	history.push(current);
    }
    free(previous);
    free(current);
    return view;
}

View *InfoFile::getnode(const char *anodename)
{
    BUGME(("InfoFile::getnode(\"%s\")", anodename));

    char cpynodename[strlen(anodename)+1];
    char *nodename = cpynodename;
    while(isspace(*anodename)) anodename++;
    char *cpy = nodename;
    if (*anodename == '(') {
	while (*anodename != '\0' && *anodename != ')') {
	    *cpy++ = *anodename++;
	}
    }
    while (*anodename != '\0' && strchr("\t\n,.", *anodename) == NULL) {
	*cpy++ = *anodename++;
    }
    *cpy = '\0';

    if (cpynodename[0] == '\(') {
	history.push(cpynodename);
    } else {
	char tempbuf[strlen(cpynodename)+strlen(most_recent_filename)+3];
	sprintf(tempbuf, "(%s)%s", most_recent_filename, cpynodename);
	history.push(tempbuf);
    }

    if (*nodename == '\(') {
	BUGME(("getnode: Node name may contain file component: ``%s''", 
	       nodename));
	// Uh-oh.  New file
	char *tail = strchr(nodename, ')');
	if (tail != NULL && tail > nodename + 1) {
	    BUGME(("getnode: Extracting filename"));
	    char newfilename[tail-nodename];
	    nodename++;
	    strncpy(newfilename, nodename, tail-nodename);
	    newfilename[tail-nodename] = '\0';
	    BUGME(("getnode: Filename is ``%s''", newfilename));
	    nodename = tail+1;
	    BUGME(("getnode: Opening ``%s''", newfilename));
	    open(newfilename);
	}
    }
    if (!ok()) {
fail:
        BUGME(("getnode: Failed"));
	return NULL;
    }

    View *view = NULL;

    if (*nodename == '\0') nodename = "Top";

    // Goto start of file
    if (split_file) {
	long min_pos = 0, max_pos = 2048;
	if (tags_table_availible) {
	    SplitList *tag;
	    for (tag = tags_table; tag != NULL; tag = tag->next) {
		if (!nodename_cmp(nodename, tag->split.filename)) break;
	    }
	    if (tag == NULL) goto fail;
	    min_pos = tag->split.position - 1024;
	    if (min_pos < 0) min_pos = 0;
	    max_pos = tag->split.position + 1024;
	    if (max_pos < 1024) max_pos = 1024;
	}
	SplitList *sl;
	SplitList *prev = NULL;
	for (sl = splitlist; sl != NULL; sl = sl->next) {
	    BUGME(("Trying %s", sl->split.filename));
	    if (tags_table_availible) {
		BUGME(("Checking bounds: %ld in [%ld,%ld]",
			      sl->split.position, min_pos, max_pos));
		if (prev == NULL) {
		    if (sl->split.position > max_pos) continue;
		} else if (min_pos > prev->split.position) {
		    // No way - we're past the point of possibility
		    break;
		} else if (sl->split.position > max_pos) {
		    // No way - we haven't gone far enough yet
	    	    prev = sl;
	    	    continue;
		} // Otherwise, try this node
		prev = sl;
	    }
	    infofile = infoopen(sl->split.filename, most_recent_filename);
	    if (infofile == NULL) continue;
	    fseek(infofile, 0, SEEK_SET);
	    view = getnode_attempt(nodename);
	    if (view != NULL) break;
	}
    } else {
	fseek(infofile, 0, SEEK_SET);
	view = getnode_attempt(nodename);
    }
    if (view == NULL) goto fail;

    add_attributes(view);
    return view;
}

void InfoFile::add_attributes(View *view)
{
    BUGME(("InfoFile::add_attributes(...)"));
    // Add menu highlighting
    int i;
    const char *line;
    BUGME(("InfoFile::add_attributes: Searching for menu"));
    for (i=1; i < view->length(); i++) {
	line = view->line(i);
	if (line == NULL) continue;
	while(isspace(*line)) line++;
	if (*line != '*') continue;
	line++;
	while(isspace(*line)) line++;
	if (nodename_ncmp(line, "Menu", 4)) continue;
	line += 4;
	while(isspace(*line)) line++;
	if (*line == ':'); break;
    }
    if (i < view->length()) {
	BUGME(("InfoFile::add_attributes: Found start of menu"));
	view->attr.
	    add_attribute(new RedAttr(TextRegion(TextPoint(i,0), 
						 TextPoint(i,INT_MAX))));
	i++;

	const char *start, *end;
	for (; i < view->length(); i++) {
	    start = line = view->line(i);
	    if (line == NULL) continue;
	    if (*line != '*') continue;
	    const char *start_of_nodename = line+1;
	    BUGME(("InfoFile::add_attributes: Search for colon in ``%s''",
		   line));
	    end = strstr(line, ":");
	    BUGME(("InfoFile::add_attributes: Search returns ``%s''", end));
	    if (end == NULL) continue;
	    BUGME(("InfoFile::add_attributes: Menu line detected: ``%s''",
		   start));
	    BUGME(("InfoFile::add_attributes: At colon: ``%s''", end));
	    if (end[1] == ':') {
	      BUGME(("Ka-ping!"));
		end++;
	    } else {
	      BUGME(("Whiff!"));
		// Find the end of the node name
		end++;
		for(;;) {
		    if (*end == '\0') break;
		    if (strchr("\t\n,.", *end)) break;
		    if (*end == '(') {
			while (*end != '\0' && *end != ')') end++;
		    } else {
			end++;
		    }
		}
	    }

	    int ref_name_len = end-start_of_nodename+1;
	    char node_ref_name[ref_name_len+1];
	    strncpy(node_ref_name, start_of_nodename, ref_name_len);
	    node_ref_name[ref_name_len] = '\0';
	    
	    view->attr.
		add_attribute(new MenuItem(TextRegion(TextPoint(i,0), 
						      TextPoint(i,
								end-start)),
					   node_ref_name));
	}
    }


    // Now find and highlight cross-references
    BUGME(("InfoFile::add_attributes: Searching for cross refs"));
    int view_length = view->length();
    for (i=1; i < view_length; i++) {
	const char *line_start;
	line_start = line = view->line(i);
	const char *p;
	for(;;) {
	    p = strstr(line, "*note");
	    if (p == NULL) {
		p = strstr(line, "*Note");
	    } else {
		char *p2 = strstr(line, "*Note");
		if (p2 != NULL && p2 < p) p = p2;
	    }
	    if (p == NULL) break;
	    TextPoint start_point(i,p-line_start);
	    line = p + 5;
	    // Skip whitespace
	    for(;;) {
		if (*line == '\0') {
		    i++;
		    if (i >= view_length) break;
		    line_start = line = view->line(i);
		} else if (isspace(*line)) {
		    while(isspace(*line)) line++;
		} else {
		    break;
		}
	    }
	    // Skip over node name
	    if (line == NULL) continue;
	    const char *start_of_nodename = line;
	    for(;;) {
		if (*line == '\0') break;
		if (strchr("\t\n,.", *line) != NULL) break;
		if (*line == '(') {
		    while (*line != ')' && *line != '\0') line++;
		} else {
		    line++;
		}
	    }

	    int ref_name_len = line-start_of_nodename;
	    char node_ref_name[ref_name_len+1];
	    strncpy(node_ref_name, start_of_nodename, ref_name_len);
	    node_ref_name[ref_name_len] = '\0';
	    
	    BUGME(("InfoFile::add_attributes: Node ref spotted: ``%s''...",
		   node_ref_name));
	    view->attr.add_attribute
		(new ReferenceItem(TextRegion(start_point,
					      TextPoint(i,line-line_start)),
				   node_ref_name));
	    BUGME(("InfoFile::add_attributes: ... Added"));
	}
    }
}
