/*
 * This program is accepts events and prints information about those
 * events.  This is a useful learning tool for finding out what events
 * are generated by which keys.
 */

#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <ctype.h>
#include <pmio.h>
#define INCL_WIN
#include <os2.h>

void show_key (const PMIOEvent e)
{
  const PMIOEventKey k = e.e.key;
  struct bitnames {
    unsigned short bitpattern;
    const char * bitname;
  } flagbits[] = {
    { KC_CHAR, "KC_CHAR" },
    { KC_SCANCODE, "KC_SCANCODE" },
    { KC_VIRTUALKEY, "KC_VIRTUALKEY" },
    { KC_KEYUP, "KC_KEYUP" },
    { KC_PREVDOWN, "KC_PREVDOWN" },
    { KC_DEADKEY, "KC_DEADKEY" },
    { KC_COMPOSITE, "KC_COMPOSITE" },
    { KC_INVALIDCOMP, "KC_INVALIDCOMP" },
    { KC_LONEKEY, "KC_LONEKEY" },
    { KC_SHIFT, "KC_SHIFT" },
    { KC_ALT, "KC_ALT" },
    { KC_CTRL, "KC_CTRL" }
  };
  struct bitnames vkbits[] = {
    { VK_BUTTON1, "VK_BUTTON1" },
    { VK_BUTTON2, "VK_BUTTON2" },
    { VK_BUTTON3, "VK_BUTTON3" },
    { VK_BREAK, "VK_BREAK" },
    { VK_BACKSPACE, "VK_BACKSPACE" },
    { VK_TAB, "VK_TAB" },
    { VK_BACKTAB, "VK_BACKTAB" },
    { VK_NEWLINE, "VK_NEWLINE" },
    { VK_SHIFT, "VK_SHIFT" },
    { VK_CTRL, "VK_CTRL" },
    { VK_ALT, "VK_ALT" },
    { VK_ALTGRAF, "VK_ALTGRAF" },
    { VK_PAUSE, "VK_PAUSE" },
    { VK_CAPSLOCK, "VK_CAPSLOCK" },
    { VK_ESC, "VK_ESC" },
    { VK_SPACE, "VK_SPACE" },
    { VK_PAGEUP, "VK_PAGEUP" },
    { VK_PAGEDOWN, "VK_PAGEDOWN" },
    { VK_END, "VK_END" },
    { VK_HOME, "VK_HOME" },
    { VK_LEFT, "VK_LEFT" },
    { VK_UP, "VK_UP" },
    { VK_RIGHT, "VK_RIGHT" },
    { VK_DOWN, "VK_DOWN" },
    { VK_PRINTSCRN, "VK_PRINTSCRN" },
    { VK_INSERT, "VK_INSERT" },
    { VK_DELETE, "VK_DELETE" },
    { VK_SCRLLOCK, "VK_SCRLLOCK" },
    { VK_NUMLOCK, "VK_NUMLOCK" },
    { VK_ENTER, "VK_ENTER" },
    { VK_SYSRQ, "VK_SYSRQ" },
    { VK_F1, "VK_F1" },
    { VK_F2, "VK_F2" },
    { VK_F3, "VK_F3" },
    { VK_F4, "VK_F4" },
    { VK_F5, "VK_F5" },
    { VK_F6, "VK_F6" },
    { VK_F7, "VK_F7" },
    { VK_F8, "VK_F8" },
    { VK_F9, "VK_F9" },
    { VK_F10, "VK_F10" },
    { VK_F11, "VK_F11" },
    { VK_F12, "VK_F12" },
    { VK_F13, "VK_F13" },
    { VK_F14, "VK_F14" },
    { VK_F15, "VK_F15" },
    { VK_F16, "VK_F16" },
    { VK_F17, "VK_F17" },
    { VK_F18, "VK_F18" },
    { VK_F19, "VK_F19" },
    { VK_F20, "VK_F20" },
    { VK_F21, "VK_F21" },
    { VK_F22, "VK_F22" },
    { VK_F23, "VK_F23" },
    { VK_F24, "VK_F24" },
    { VK_ENDDRAG, "VK_ENDDRAG" }
  };
  const char * const function_key_name[256] =
    {
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
      "Shift Tab", /* 15 */
      /* Row 1 of the ALT keyboard... qwerty...p */
      "Alt-Q", "Alt-W", "Alt-E", "Alt-R", "Alt-T", "Alt-Y", "Alt-U", "Alt-I",
      "Alt-O", "Alt-P", 0, 0, 0, 0,
      /* Row 2 of the ALT Keyboard... asdfg...l */
      "Alt-A", "Alt-S", "Alt-D", "Alt-F", "Alt-G",
      "Alt-H", "Alt-J", "Alt-K", "Alt-L", 0, 0, 0, 0, 0,
      /* Row 3 of the ALT Keyboard... zxcv...m */
      "Alt-Z", "Alt-X", "Alt-C", "Alt-V", "Alt-B", "Alt-N", "Alt-M",
      0, 0, 0, 0, 0, 0, 0, 0,
      /* F1 through F10 */
      "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10",
      0, 0,
      /* Misc keys */
      "Home", "Up Arrow", "PgUp", 0, "Left Arrow", 0, "Right Arrow", 0,
      "End", "Down Arrow", "PgDn", "Ins", "Del",
      /* Shift F1 through Shift F10 */
      "Shift-F1", "Shift-F2", "Shift-F3", "Shift-F4", "Shift-F5",
      "Shift-F6", "Shift-F7", "Shift-F8", "Shift-F9", "Shift-F10",
      /* Control F1 through Control F10 */
      "Ctrl-F1", "Ctrl-F2", "Ctrl-F3", "Ctrl-F4", "Ctrl-F5",
      "Ctrl-F6", "Ctrl-F7", "Ctrl-F8", "Ctrl-F9", "Ctrl-F10",
      /* Alt F1 through Control F10 */
      "Alt-F1", "Alt-F2", "Alt-F3", "Alt-F4", "Alt-F5",
      "Alt-F6", "Alt-F7", "Alt-F8", "Alt-F9", "Alt-F10",
      /* Misc */
      "Ctrl-PrtSc", "Ctrl-Left Arrow", "Ctrl-Right Arrow", "Ctrl-End",
      "Ctrl-PgDn", "Ctrl-Home",
      /* Alt-1 through Alt-= */
      "Alt-1", "Alt-2", "Alt-3", "Alt-4", "Alt-5", "Alt-6",
      "Alt-7", "Alt-8", "Alt-9", "Alt-0", "Alt-minus", "Alt-Equal",
      /* Misc */
      "Ctrl-PgUp",
      /* Don't use these keys in real programs -- not everyone has them! */
      "F11", "F12", "Shift-F11", "Shift-F12", "Ctrl-F11", "Ctrl-F12",
      "Alt-F11", "Alt-F12"
      };
  int i;

  printf ("flags: 0x%04X = ", k.flags);
  for (i = 0; i < sizeof (flagbits) / sizeof (flagbits[0]); i++)
    if (k.flags & flagbits[i].bitpattern)
      printf ("%s ", flagbits[i].bitname);
  printf ("\n");
  printf ("  repeat=%d", k.repeat_count);
  if (k.flags & KC_SCANCODE)
    printf (", scancode=%02X", k.scancode);
  printf (", ch=%02X", k.charcode);
  if (isprint (k.ch))
    {
      printf (" '");
      set_fg (YELLOW);
      set_bg (RED);
      printf ("%c", k.ch);
      set_fg (BLACK);
      set_bg (WHITE);
      printf ("' ");
    }
  if (k.flags & KC_VIRTUALKEY)
    {
      printf (", vk=%02X", k.virtualkeycode);
      for (i = 0; i < sizeof (vkbits) / sizeof (vkbits[0]); i++)
	if (k.virtualkeycode == vkbits[i].bitpattern)
	  printf (" %s", vkbits[i].bitname);
    }
  char c1, c2;
  pmio_event_convert (&e, &c1, &c2);
  if (c1 != 0 || c2 != 0)
    {
      printf (", getch=");
      set_fg (YELLOW);
      set_bg (RED);
      printf ("0x%02X", (unsigned) c1);
      if (c1 == 0)
	if (function_key_name[c2] != 0)
	  printf (" %s", function_key_name [(unsigned)c2]);
	else
	  printf (" 0x%02X", (unsigned) c2);
      set_fg (BLACK);
      set_bg (WHITE);
    }
  printf ("\n");
}

int main ()
{
  start_pmio ();
  set_fg (YELLOW);
  set_bg (BLUE);
  printf ("Key test program.  Press ESC twice to quit.");
  clreol ();
  set_fg (BLACK);
  set_bg (WHITE);
  set_window (1,2, get_screen_width (), get_screen_height ());
  
  while (!pmio_event_availible ())
    {
      printf (".");
      usleep (250000);
    }
  printf ("\n");
  int vkesc = 0;
  pmio_batch_on ();
  for (;;)
    {
      PMIOEvent e;
      if (!pmio_event_availible ())
	{
	  // Tell pmio to only bother repainting the screen once
	  // we have processed and printed all availible events
	  pmio_batch_off ();
	  pmio_get_event (&e);
	  pmio_batch_on ();
	}
      else
	pmio_get_event (&e);
      switch (e.event_type)
	{
	case pmio_event_char:
	  // If two down-key presses in a row are from the ESC key, quit
	  // the program
	  if (!(e.e.key.flags & KC_KEYUP))
	    if ((e.e.key.flags & KC_VIRTUALKEY)
		&& (e.e.key.virtualkeycode == VK_ESC))
	      if (vkesc)
		return 0;
	      else
		vkesc = 1;
	    else
	      vkesc = 0;
	  // If we're not quitting, show the key in it's gory detail
	  show_key (e);
	  break;
	default:
	  printf ("Unknown event type!!!\n");
	  break;
	}
    }
  return 0;
}
