/*      --- xydriver.c  V1.0 Jan 1994 (windows version) ---
**
**  Copyright(C) 1994 by MarshallSoft Computing, Inc.
**  All rights are reserved.
**
**  NOTES:
**  (1) All file OPEN and CLOSE operations are performed here, not in the
**      calling program.
**  (2) The more channels being run, the slower the maximum baud rate !
*/

#define DEBUG 0

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "windows.h"
#include "pcl4w.h"
#include "xycodes.h"
#include "errcodes.h"
#include "ascii.h"
#include "xydriver.h"
#include "term_io.h"
#include "file_io.h"
#include "crc.h"
#include "dir_io.h"
#include "info.h"
#include "file_io.h"
#include "xyerror.h"
#include "constants.h"

extern HWND hMainWnd;

/*** xyDriver functions & variables */

long atol();                      /* converts ascii to long integer */
static char Temp[255];            /* temporary buffer */
static int GotPacketNbrComp;      /* packet # received */
static unsigned short CheckSum;   /* 1 byte checksum */

/*** xyDriver state variables ***/

static int  xyActive=FALSE;        /* TRUE if initialized */
static int  xyAttempt;             /* current attempt number */
static int  xyBatchFlag;           /* if TRUE, send filename in packet 0 */
static char xyBuffer[1024];        /* data packet buffer */
static unsigned short xyCheckSum;  /* computed checksum or CRC byte */
static int  xyRetry;               /* retry count */
static int  xyDoneFlag;            /* TRUE when done */
static int  xyEmptyFlag;           /* TRUE when filename is empty */
static int  xyEOTflag;             /* if TRUE, EOT was received */
static int  xyErrorCode;           /* last error code */
static int  xyErrorState;          /* state in which last error occured */
static char xyFilename[64];        /* current filename */
static long xyFileSize;            /* file size */
static char xyFileSpec[64];        /* current file specification */
static int  xyFirstPacket;         /* first packet number (0 or 1) */
static int  xyHandle;              /* file Handle */
static int  xyGotPacketNbr;        /* packet number actually received */
static int  xyLastPacket;          /* last packet number */
static int  xyMaxtry;              /* maximum retry limit */
static int  xyNAKs;                /* number of NAKs received */
static char xyNCGchar;             /* NAK, 'C', or 'G' */
static int  xyNumber1K;            /* total # 1K packets */
static int  xyNumber128;           /* total # 128 byte packets */
static int  xyOneKflag;            /* if TRUE, use 1K blocks when possible */
static int  xyPacket;              /* current packet number */
static int  xyPacketNbr;           /* expected last packet # */
static int  xyPacketSize;          /* curent packet size (128 or 1024) */
static char xyPacketType;          /* packet type (SOH,STX, etc.) */
static int  xyPort;                /* serial port */
static unsigned short xyRxCheckSum;/* received checksum or CRC byte */
static int  xyReturn;              /* return state */
static int  xyState;               /* current driver state */
static long xyTime;                /* last time returned by GetTickCount */
static int  xyIndex;               /* index into packet receive buffer */
static int  xyLastState;           /* last driver state */
static int  xyStateCount;          /* # times last state called */
#if DEBUG
static long LastTime;              /* last time xyDriver was called */
#endif

/*** xyDriver functions ***/

int xyAbort(void)
{int i;
 xyErrorCode = NO_ERROR;
 xyDoneFlag = TRUE;
 xyErrorState = 0;
 xyState = XY_IDLE;
 for(i=0;i<6;i++) if(PutChar(xyPort,CAN)==-1) break;
 SioRxFlush(xyPort);
 return(0);
}

int xyGetErrorCode(void)
{
 xyDriver();
 return( xyErrorCode );
}

int xyGetErrorState(void)
{
 xyDriver();
 return( xyErrorState );
}

char *xyGetFilename(void)
{
 xyDriver();
 return( &xyFilename[0] );
}

int xyGetPacket(void)
{
 xyDriver();
 return( xyPacket );
}

int xyGetState(void)
{
 xyDriver();
 return( xyState );
}

long xyGetFileLength()
{
 xyDriver();
 return( xyFileSize );
}

int xyInit(Port)
int Port;
{
#if DEBUG
 sprintf(Temp,"xyInit: Port=COM%d\n",Port+1);
 DisplayLine(Temp);
 xyLastState = -1;
 xyStateCount = 0;
#endif
 xyActive = TRUE;
 xyPort = Port;
 xyState = XY_IDLE;
 xyDoneFlag = TRUE;
 xyHandle = -1;
 xyFilename[0] = '\0';
 xyErrorCode = NO_ERROR;
 InitCRC();
 return(0);
}

int xyGetNAKs(void)
{
 xyDriver();
 return( xyNAKs );
}

int xyStartTx(Filename,OneKflag,BatchFlag)
char *Filename;     /* File name */
int OneKflag;       /* use 1K packets */
int BatchFlag;      /* use batch mode (YMODEM) */
{
#if DEBUG
 sprintf(Temp,"xyStartTx:Filename='%s' OneKflag=%d BatchFlag=%d",
   Filename,OneKflag,BatchFlag);
#endif
 xyErrorCode = NO_ERROR;
 xyDoneFlag = FALSE;
 xyErrorState = 0;
 strcpy(&xyFileSpec[0],Filename);
 xyRetry = 0;
 xyTime = 0L;
 xyOneKflag = OneKflag;
 xyBatchFlag = BatchFlag;
 xyNAKs = 0;
 xyPacket = -1;
 xyReturn = XY_IDLE;
 xyState = TX_START;
 xyFileSize = 0;
 xyDriver();
 return(0);
} /* xyStartTx */

int xyStartRx(Filename,NCGchar,BatchFlag)
char *Filename;     /* file name */
char NCGchar;       /* should be NAK, 'C', or 'G' */
int BatchFlag;      /* batch flag (YMODEM) */
{
#if DEBUG
 sprintf(Temp,"xyStartRx:Filename='%s' NCGchar=%c BatchFlag=%d\n",
   Filename,NCGchar,BatchFlag);
 DisplayLine(Temp);
#endif
 xyErrorCode = NO_ERROR;
 xyDoneFlag = FALSE;
 xyErrorState = 0;
 strcpy(&xyFilename[0],Filename);
 xyRetry = 0;
 xyTime = 0L;
 xyOneKflag = FALSE;
 xyBatchFlag = BatchFlag;
 xyNCGchar = NCGchar;
 xyNAKs = 0;
 xyPacket = -1;
 xyReturn = XY_IDLE;
 xyState = RX_START;
 xyFileSize = 0;
 xyDriver();
 return(0);
} /* end xyStartRx */

#define DATA_TRIES  4

int xyDriver()
{int i, k;          /* local integers */
 int Code;          /* return code */
 int Len;           /* string length */
 char c;
 if(!xyActive) return(-1);
#if DEBUG
 if(xyLastState!=xyState)
   {sprintf(Temp,"%ld: State = %d\n",GetTickCount(),xyState);
    DisplayLine(Temp);
   }
#endif
 if(xyLastState==xyState) xyStateCount++;
 else
   {xyStateCount = 1;
    xyLastState = xyState;
   }
 /* process according to state */
 switch(xyState)
   {

    case XY_IDLE:

      return IDLE;

    /*** Transmit states ***/

    case TX_START:

      if(FindFirst(&xyFileSpec[0],&xyFilename[0]))
        {/* got match ! */
         xyState = TX_NEWFILE;
        }
      else
        {/* no file found */
         SetXYerror(NO_SUCH_FILE_ERROR);
        }
      break;

    case TX_NEWFILE:

      if(xyBatchFlag&&(xyFilename[0]=='\0')) xyEmptyFlag = TRUE;
      else xyEmptyFlag = FALSE;
      /* compute # blocks */
      if(xyEmptyFlag)
         {/* empty file ! */
          xyNumber1K = 0;
          xyNumber128 = 0;
         }
      else
         {/* file NOT empty */
         xyHandle = _lopen(&xyFilename[0],OF_READ|OF_SHARE_DENY_WRITE);
         if(xyHandle<0)
           {SetXYerror(CANNOT_OPEN_ERROR);
            break;
           }
          xyFileSize = FileLength(xyHandle);
          ComputeSize(xyFileSize,xyOneKflag,&xyNumber1K,&xyNumber128);
         }
      /* clear comm port ( there may be several NAKs queued up ) */
      SioRxFlush(xyPort);
      xyRetry = 0;
      /* call TX_WAIT4_NCG as subroutine, then return to TX_COMPUTE_PACKETS */
      xyState = TX_WAIT4_NCG;
      xyReturn = TX_COMPUTE_PACKETS;
      xyTime = (long) GetTickCount();
      InfoFileName(xyFilename);
      InfoFileSize(xyFileSize);
#if 1
sprintf(Temp,"TX: Filename='%s' Size=%ld #1K=%u #128=%u\n",
  &xyFilename[0],xyFileSize,xyNumber1K,xyNumber128);
DisplayLine(Temp);
#endif
      break;

    case TX_COMPUTE_PACKETS:

      /* compute first packet */
      if(xyBatchFlag) xyFirstPacket = 0;
      else xyFirstPacket = 1;
      /* compute last packet # */
      xyLastPacket = xyNumber1K + xyNumber128;
      /* loop over all packets */
      xyPacket = xyFirstPacket;
      xyState = TX_READ_DISK;
      break;

    case TX_READ_DISK:

      /* load up Buffer */
      if(xyEmptyFlag)
        {/* empty packet # 0 */
         xyPacketSize = 128;
         for(i=0;i<128;i++) xyBuffer[i] = '\0';
        }
      else if(xyPacket==0)
        {/* Filename packet ! */
         xyPacketSize = 128;
         k = 0;
         for(i=0;i<strlen(&xyFilename[0]);i++)
            xyBuffer[k++] = xyFilename[i];
         xyBuffer[k++] = '\0';
         sprintf(Temp,"%ld",xyFileSize);
         for(i=0;i<strlen(Temp);i++) xyBuffer[k++] = Temp[i];
         while(k<128) xyBuffer[k++] = '\0';
        }
      else /* xyPacket > 0 */
       {/* DATA Packet: use 1K or 128 byte block ? */
        if(xyPacket<=xyNumber1K) xyPacketSize = 1024;
        else xyPacketSize = 128;
        /* read next block from disk */
        Code = _lread(xyHandle,&xyBuffer[0],xyPacketSize);
        if(Code<=0)
          {SetXYerror(DISK_READ_ERROR);
           break;
          }
        /* pad with ^Z's */
        for(i=Code;i<xyPacketSize;i++) xyBuffer[i] = 0x1a;
       }
      xyState = TX_PREPARE_PACKET;
      break;

    case TX_PREPARE_PACKET:

      /* get this packet ready */
      if(xyPacketSize==1024) xyPacketType = STX;
      else xyPacketType = SOH;
      xyPacketNbr = xyPacket & 0x00ff;
      /* make up to MAXTRY attempts to send this packet */
      xyAttempt = 1;
      xyMaxtry = 5;
      xyState = TX_SEND_PACKET;
      break;

    case TX_SEND_PACKET:

      /* send SOH/STX  */
      Code = PutChar(xyPort,xyPacketType);
      /* send packet # */
      Code = PutChar(xyPort,(char)xyPacketNbr);
      /* send 1's complement of packet */
      Code = PutChar(xyPort,(char)(255-xyPacketNbr));
      /* send data */
      xyCheckSum = 0;
      xyIndex = 0;
      xyRetry = 0;
      xyTime = (long)GetTickCount();
      xyState = TX_SEND_DATA;
      break;

    case TX_SEND_DATA:

      /* send entire packet */
      while(xyIndex<xyPacketSize)
        {c = xyBuffer[xyIndex];
         Code = PutChar(xyPort,c);
         if(Code==-1)
           {if((long)GetTickCount() > xyTime + (long)SIX_SECS) SetXYerror(TIMED_OUT_ERROR);
            break;
           }
         xyIndex++;
         if(xyNCGchar==NAK) xyCheckSum += c;
         else xyCheckSum = UpdateCRC(xyCheckSum, c);
        }
      xyState = TX_SEND_CHECKSUM;
      break;

   case TX_SEND_CHECKSUM:

      /* send checksum */
#if DEBUG
sprintf(Temp,"Checksum = %xH (%c)\n",xyCheckSum,xyNCGchar);
DisplayLine(Temp);
#endif
      if(xyNCGchar==NAK)
          {Code = PutChar(xyPort,(char)(xyCheckSum & 0x00ff));
          }
      else
          {Code = PutChar(xyPort,(char)((xyCheckSum>>8) & 0x00ff));
           Code = PutChar(xyPort,(char)(xyCheckSum & 0x00ff));
          }
      /* no ACK to wait for if 'G' */
      if(xyNCGchar=='G')
         {xyState = TX_CHECK4_RESTART;
         }
      else
         {xyTime = (long)GetTickCount();
          /* flush any accumulated chars [1/7/94]*/
          SioRxFlush(xyPort);          /*[1/7/94]*/
          xyState = TX_WAIT4_PACKET_ACK;
         }
      break;

    case TX_WAIT4_PACKET_ACK:

      /* get receivers ACK */
      Code = GetChar(xyPort);
      if(Code==-1)
        {if((long)GetTickCount() > xyTime + (long)TEN_SECS) SetXYerror(TIMED_OUT_ERROR);
         break;
        }
      /* char should be ACK, NAK, or CAN */
      if((char)Code==CAN) SetXYerror(CANCELLED_ERROR);
      else if((char)Code==ACK)
        {xyState = TX_CHECK4_RESTART;
#if DEBUG
sprintf(Temp,"Packet %d ACKed\n",xyPacket);
DisplayLine(Temp);
#endif
        }
      else if((char)Code==NAK)
        {xyAttempt++;
         xyNAKs++;
#if DEBUG
sprintf(Temp,"Packet %d NAKed\n",xyPacket);
DisplayLine(Temp);
#endif
         if(xyAttempt < xyMaxtry) xyState =  TX_SEND_PACKET;
         else SetXYerror(RETRIES_ERROR);
        }
      else
        {
#if DEBUG
sprintf(Temp,"Packet %d expects ACK/NAK not %xH\n",xyPacket,(char)Code);
DisplayLine(Temp);
#endif
         SetXYerror(OUT_OF_SYNC_ERROR);
        }
      break;

    case TX_CHECK4_RESTART:

      /* must 'restart' after each non-null packet # 0 */
      if(!xyEmptyFlag&&(xyPacket==0))
        {
#if DEBUG
DisplayLine("'Restart' after packet 0\n");
#endif
         xyTime = (long)GetTickCount();
         xyRetry = 0;
         /* call TX_WAIT4_NCG & return to TX_NEXT_PACKET */
         xyState = TX_WAIT4_NCG;
         xyReturn = TX_NEXT_PACKET;
        }
      else xyState = TX_NEXT_PACKET;
      break;

    case TX_NEXT_PACKET:

      /* another packet ? */
      if(xyPacket < xyLastPacket)
        {xyState = TX_READ_DISK;
         xyPacket++;
        }
      else
        {/* no more packets */
         _lclose(xyHandle);
         if(xyEmptyFlag)
           {/* all done if just sent null filename packet */
            SetXYerror(NO_ERROR);
           }
         else
           {xyRetry = 0;
            xyState = TX_SEND_EOT;
           }
        }
      break;

    case TX_SEND_EOT:

      /* transmit EOT */
      xyTime = (long)GetTickCount();
      PutChar(xyPort,EOT);
      xyState = TX_WAIT4_EOT_ACK;
      break;

    case TX_WAIT4_EOT_ACK:

      if(GetTickCount() < xyTime + (long)THREE_SECS) break;
      Code = GetChar(xyPort);
      if(Code==-1)
        {xyRetry++;
         if(xyRetry==10) SetXYerror(NO_EOT_ACK_ERROR);
         else xyState = TX_SEND_EOT;
         break;
        }
      /* received character */
      if((char)Code==ACK)
        {/* transfer complete */
#if DEBUG
sprintf(Temp,"EOT was ACKed\n");
DisplayLine(Temp);
#endif
         if(xyBatchFlag)
            {/* another file ? */
             if(!FindNext(&xyFilename[0]))
               {/* send empty file */
                xyFilename[0] = '\0';
               }
             xyState = TX_NEWFILE;
             break;
            }
         SetXYerror(NO_ERROR);
        }
      break;

    case TX_WAIT4_NCG:  /* call as "subroutine" */

      if(GetTickCount() < xyTime + (long)THREE_SECS) break;
      /* incoming character ? */
      Code = GetChar(xyPort);
      if(Code==-1)
        {/* no incoming */
         xyRetry++;
         /* quit after 60 seconds */
         if(xyRetry > 20) SetXYerror(NO_STARTUP_CHAR_ERROR);
         xyTime = (long)GetTickCount();
         break;
        }
      /* received a byte */
      if( ((char)Code==NAK) || ((char)Code=='C') || ((char)Code=='G') )
        {xyNCGchar = (char)Code;
#if DEBUG
         sprintf(Temp,"Received NCG = %c\n",xyNCGchar);
         DisplayLine(Temp);
#endif
         /* return to 'caller' */
         xyState = xyReturn;
        }
      else SetXYerror(NOT_NCG_ERROR);
      break;

     /*** Receive states ***/

    case RX_START:

      /* Send NAKs, 'C's, or 'G's */
      xyEOTflag = FALSE;
      /* clear comm port */
      SioRxFlush(xyPort);
      xyRetry = 0;
      /* call RX_SEND_NCG as subroutine */
      xyReturn = RX_OPEN_FILE;
      xyState = RX_SEND_NCG;
      break;

    case RX_OPEN_FILE:

      /* open file unless BatchFlag is on */
      if(xyBatchFlag) xyFirstPacket = 0;
      else
        {/* BatchFlag is FALSE: Start with packet 1 */
         xyFirstPacket = 1;
         /* open file passed in Filename[] for write */
         xyHandle = _lcreat(&xyFilename[0],0);
         if(xyHandle<0)
           {SetXYerror(CANNOT_OPEN_ERROR);
            break;
           }
         }
      xyState = RX_READY4_PACKETS;
      break;

    case RX_READY4_PACKETS:

      /* get each packet ready in turn */
      xyPacket = xyFirstPacket;
      xyState = RX_GET_PACKET ;
      break;

    case RX_GET_PACKET:

      /* get next packet */
      xyPacketNbr =  xyPacket & 0x00ff;
      xyAttempt = 1;
      xyMaxtry = 5;
      xyTime = (long)GetTickCount();
      xyState = RX_WAIT4_SOH_STX;
      break;

    case RX_WAIT4_SOH_STX:

      /* wait for SOH / STX / CAN / EOT */
      Code = GetChar(xyPort);
      if(Code==-1)
        {if(GetTickCount() > xyTime + (long)SIX_SECS) SetXYerror(TIMED_OUT_ERROR);
         break;
        }
      /* received char should be SOH, STX, CAN, or EOT */
      if((char)Code==SOH)
        {xyPacketSize = 128;
         xyState = RX_GET_PACKET_NBR;
        }
      else if((char)Code==STX)
        {xyPacketSize = 1024;
         xyState = RX_GET_PACKET_NBR;
        }
      else if((char)Code==CAN) SetXYerror(CANCELLED_ERROR);
      else if((char)Code==EOT)
        {/* ACK the EOT */
         PutChar(xyPort,ACK);
         xyEOTflag = TRUE;
         /* close the file */
         _lclose(xyHandle);
         /* restart receiver if BatchFlag is TRUE */
         if(xyBatchFlag) xyState = RX_START;
         else SetXYerror(NO_ERROR);
        }
      else SetXYerror(OUT_OF_SYNC_ERROR);
      break;

    case RX_GET_PACKET_NBR:

      xyTime = (long)GetTickCount();
      xyState = RX_GET_PACKET_NBR;
      /* receive packet # */
      Code = GetChar(xyPort);
      if(Code==-1)
        {if(GetTickCount() > xyTime + (long)SIX_SECS) SetXYerror(TIMED_OUT_ERROR);
         break;
        }
      xyGotPacketNbr = 0x00ff & Code;
      xyTime = (long)GetTickCount();
      xyState = RX_GET_COMPLIMENT;
      break;

    case RX_GET_COMPLIMENT:

      /* receive 1's complement */
      Code = GetChar(xyPort);
      if(Code==-1)
        {if(GetTickCount() > xyTime + (long)SIX_SECS) SetXYerror(TIMED_OUT_ERROR);
         break;
        }
      GotPacketNbrComp = 0x00ff & Code;
      /* verify packet number */
      if(xyGotPacketNbr+GotPacketNbrComp!=255)
        {SetXYerror(BAD_PACKET_NBR_ERROR);
         break;
        }
      /* ready to receive data */
      xyRetry = 0;
      xyCheckSum = 0;
      xyIndex = 0;
      xyState = RX_GET_DATA;
      xyTime = (long)GetTickCount();
      break;

    case RX_GET_DATA:

      /* get next packet */
      while(xyIndex<xyPacketSize)
        {/* get next byte */
         Code = GetChar(xyPort);
         if(Code==-1)
           {if((long)GetTickCount() > xyTime + (long)TEN_SECS) SetXYerror(TIMED_OUT_ERROR);
            break;
           }
         /* save data in buffer */
         xyBuffer[xyIndex++] = (char)Code;
         /* compute checksum or CRC */
         if(xyNCGchar==NAK) xyCheckSum = (xyCheckSum + Code) & 0x00ff;
         else xyCheckSum = UpdateCRC(xyCheckSum,(unsigned char)(0x00ff&Code));
        }
      /* got all data ? */
      if(xyIndex==xyPacketSize)
        {xyTime = (long)GetTickCount();
         xyState = RX_GET_CRC;
        }
      break;

    case RX_GET_CRC:

      /* receive CRC/checksum */
      Code = GetChar(xyPort);
      if(Code==-1)
        {if(GetTickCount() > xyTime + (long)SIX_SECS) SetXYerror(TIMED_OUT_ERROR);
         break;
        }
      xyRxCheckSum = Code & 0x00ff;
      if(xyNCGchar==NAK) xyState = RX_TEST_CRC;
      else
        {xyTime = (long)GetTickCount();
         xyState = RX_GET_2ND_CRC;
        }
      break;

    case RX_GET_2ND_CRC:

      /* receive 2nd CRC byte */
      Code = GetChar(xyPort);
      if(Code==-1)
        {if(GetTickCount() > xyTime + (long)SIX_SECS) SetXYerror(TIMED_OUT_ERROR);
         break;
        }
      CheckSum = Code & 0x00ff;
      xyRxCheckSum = (xyRxCheckSum<<8) | CheckSum;
      xyState = RX_TEST_CRC;
      break;

    case RX_TEST_CRC:

      /* don't send ACK if 'G' */
      if(xyNCGchar=='G')
        {xyState = RX_PACKET_OK;
         break;
        }
      /* packet # and checksum OK ? */
      if((xyCheckSum==xyRxCheckSum)
       &&(xyGotPacketNbr==xyPacketNbr))
        {/* ACK the packet */
         PutChar(xyPort,ACK);
         xyState = RX_PACKET_OK;
         break;
        }
      else
        {/* bad packet */
         Code = PutChar(xyPort,NAK);

#if DEBUG
if(xyCheckSum!=xyRxCheckSum)
  {
   sprintf(Temp,"Packet %d: BAD Checksum\n",xyPacket);
   DisplayLine(Temp);
   sprintf(Temp,"xyCheckSum=%xH xyRxCheckSum=%xH\n",xyCheckSum,xyRxCheckSum);
   DisplayLine(Temp);
  }
else
  {sprintf(Temp,"Packet %d: BAD Packet #\n",xyPacket);
   DisplayLine(Temp);
   sprintf(Temp,"xyPacketNbr=%d xyGotPacketNbr=%d\n",xyPacketNbr,xyGotPacketNbr);
   DisplayLine(Temp);
  }
sprintf(Temp,"NCGchar=%c PacketSize=%d\n",xyNCGchar,xyPacketSize);
DisplayLine(Temp);
#endif

         xyTime = (long)GetTickCount();
         xyState = RX_WAIT4_SOH_STX;
        }
      break;

    case RX_PACKET_OK:

      if(xyPacket==0)
        {/* copy Filename */
         strcpy(&xyFilename[0],&xyBuffer[0]);
         /* done if null packet 0 */
         if(xyFilename[0]=='\0')
            {/* Batch Transfer Complete */
             SetXYerror(NO_ERROR);
             break;
            }
         }
      /* process this packet */
      if(xyPacket==0)
        {/* open file using filename in packet 0 */
         xyHandle = _lcreat(&xyFilename[0],0);
         if(xyHandle<0)
           {SetXYerror(CANNOT_OPEN_ERROR);
            break;
           }
         /* get file length */
         Len = strlen(&xyBuffer[0]);
         xyFileSize = atol(&xyBuffer[1+Len]);
         /* 'restart' after packet 0 (call RX_SEND_NCG as subroutine) */
         xyPacket++;
         xyReturn = RX_GET_PACKET;
         xyState = RX_SEND_NCG;
        }
      else /* DATA packet */
        {/* write Buffer to disk */
         if(xyBatchFlag)
            {if(xyFileSize < (long)xyPacketSize)
                  i = (int) xyFileSize;
             else i = xyPacketSize;
             i = _lwrite(xyHandle,&xyBuffer[0],i);
             xyFileSize -= (long)i;
            }
         else _lwrite(xyHandle,&xyBuffer[0],xyPacketSize);
         /* go back for next packet */
         xyPacket++;
         xyState = RX_GET_PACKET ;
        } /* end -- else */
      break;

    case RX_SEND_NCG:  /* call as 'subroutine' */

      /* send NAK, 'C', or 'G' */
      xyTime = (long)GetTickCount();
      PutChar(xyPort,xyNCGchar);
      xyState = RX_WAIT4_INCOMING;
      break;

    case RX_WAIT4_INCOMING:

      /* wait 1 second first */
      if(GetTickCount() < xyTime + (long)ONE_SEC) break;
      /* read incoming */
      Code = GetChar(xyPort);
      if(Code==-1)
        {/* no incoming */
         xyRetry++;
         /* stop attempting CRC/'G' after 1st 4 tries */
         if((xyNCGchar!=NAK)&&(xyRetry==5)) xyNCGchar = NAK;
         xyState = RX_SEND_NCG;
         break;
        }
      /* no error -- must be incoming byte -- push byte back onto queue ! */

#if DEBUG
sprintf(Temp,"Received %c (%x) on port COM%d\n",Code,Code,1+xyPort);
DisplayLine(Temp);
#endif

      SioUnGetc(xyPort,(char)Code);

      /* return to caller */
      xyState = xyReturn;
      break;

    default:

       sprintf(Temp,"INTERNAL ERROR: Bad state = %d\n",xyState);
       DisplayLine(Temp);
       SetXYerror(INTERNAL_ERROR);
       break;
   } /* end switch */
 if(xyDoneFlag) return IDLE;
 else return !IDLE;
} /* end xyDriver */


int SetXYerror(Code)
int Code;
{int i;
 char ErrorText[80];
 xyErrorCode = Code;
 xyDoneFlag = TRUE;
 xyErrorState = xyState;
 xyState = XY_IDLE;
#if 0
 if(Code!=NO_ERROR) for(i=0;i<128;i++) if(SioPutc(xyPort,CAN)<0) break;
#else
 if(Code!=NO_ERROR) SioPutc(xyPort,CAN);
#endif
 SioRxFlush(xyPort);
 if(Code)
   {xyError(Code,ErrorText);
    sprintf(Temp,"xyDriver: Error=%d (%s) State=%d Count=%d\n",
     Code,ErrorText,xyLastState,xyStateCount);
    DisplayLine(Temp);
   }
 return(Code);
} /* end SetXYerror */