/********************************************************************
** FILENAME: namelist.c                     VERSION: 1.10
**
** DESCRIPTION: Creates a userlist for Telegard BBS Systems
**
** NOTES:
**      Requires a Telegard v2.7 NAMES.LST file.  Compiled with
**      Turbo C++ v3.0.
**
**      Copyright John Kristoff, 1993.  All rights reserved.
**
** AUTHOR: John Kristoff                START DATE: 4/26/93
**         Internet: jkristof@xroads.chigate.com
**                   jkristof@mica.meddean.luc.edu
**         FidoNet:  1:115/743
**
** CHANGES:
**      Coding style improved.  Added commenting, ANSI C conformity,
**      error checking, program usage.  Fixed possible NULL
**      pointer error.
**
** VERSION  DATE      WHO  DETAIL
** 1.00     4/26/93   JK   Initial development, basic coding
** 1.10     11/07/93  JK   Compartmentalization, coding improved
*/

#include <stdio.h>                     /* Standard i/o */
#include <stdlib.h>                    /* EXIT_SUCCESS/EXIT_FAILURE */
#include "telegard.h"                  /* Telegard structures */

#define VERS      "1.10"               /* version of program */
#define BACKSPACE "\b \b"              /* Used in EraseString */
#define FILEPATH  81                   /* Length of path + filename + 1 */

int main( int argc, char * argv[] );   /* If you don't know, YUSC */
void Usage( char * ProgName );         /* Display help screen */
char * Tg2C( char * Pstring );         /* Convert Telegard string to C */
void EraseString( char * String );     /* Finds length, then erases */
char * FixPath( char * OldPath );      /* i.e. Change C:\SUB to C:\SUB\ */
systatrec * TGdat( char * BBSpath );   /* Open and return STATUS.DAT */


int main( int argc, char * argv[] )
{
    FILE * InFile;                     /* File pointer for NAMES.LST */
    FILE * OutFile;                    /* File pointer to output file */

    char SourceFile[FILEPATH];         /* Full filepath of output file */
    char DestFile[FILEPATH];           /* Full filepath of NAMES.LST */
    int TotalUsers = 0;                /* Counter for total users */
    char UserName[37];                 /* Username w/o TG codes */
    smalrec UserList;                  /* NAMES.LST struct */
    systatrec * Status;                /* Pointer to STATUS.DAT struct */
    char * BBSpath;                    /* Pointer to BBS path string */

    if( argc != 3 )                    /* BBS path & output file req'd */
    {
        Usage( argv[0] );
        exit( EXIT_FAILURE );
    }

    BBSpath = FixPath( argv[1] );      /* i.e. Change C:\BBS to C:\BBS\ */ 
    Status = TGdat( BBSpath );         /* Return STATUS.DAT struct */


    /* --- Create source data file and destination output file --- */

    sprintf( SourceFile, "%s\\NAMES.LST", Tg2C(Status->gfilepath) );
    strcpy( DestFile, argv[2] );


    /* --- Open source data file and destination output file --- */

    if( (InFile = fopen(SourceFile, "rb")) == NULL )
    {
        fprintf( stderr, "(%d) ERROR: Cannot open %s\n",
                   __LINE__, SourceFile );
        exit( EXIT_FAILURE );
    }

    if( (OutFile = fopen(DestFile, "w")) == NULL )
    {
        fprintf( stderr, "(%d) ERROR: Cannot create %s\n",
                   __LINE__, DestFile );
        exit( EXIT_FAILURE );
    }


    /* --- Cycle through source data file and create output --- */

    while( fread(&UserList, sizeof(smalrec), 1, InFile) != NULL )
    {
        strcpy( UserName, Tg2C(UserList.name) );
        printf( UserName );
        fprintf( OutFile, "%d\t%s\n", UserList.number, UserName );
        ++TotalUsers;
        EraseString( UserName );
    }


    /* --- Check for EOF error, data file may be corrupted --- */

    if( !feof(InFile) )
    {
        fprintf( stderr, "(%d) ERROR: Cannot read %s\n",
                   __LINE__, SourceFile  );
        exit( EXIT_FAILURE );
    }

   fclose( InFile );
   fclose( OutFile );

   printf( "Output complete, %d users processed!\n", TotalUsers );

   return( EXIT_SUCCESS );
}



/********************************************************************
** NAME:            void Usage( char * ProgName )
**
** DESCRIPTION:     Display help screen on program usage
**
** PARAMETERS:
**    char * ProgName        Program .EXE filename that was used
**
** RETURN:
**    N/A
**
*/

void
Usage( char * ProgName )
{
    fprintf( stderr, "\n"
              "NAMELIST v" VERS ", " __DATE__
              " - Creates Telegard userlist\n"
              "Copyright John Kristoff, 1993.  All rights reserved.\n"
              "\n"
              "Usage: %s <BBS directory> <output file>\n"
              "\n", ProgName );
}



/********************************************************************
** NAME:            char * FixPath( char * OldPath )
**
** DESCRIPTION:     Sets string to include trailing '\' if needed
**
** PARAMETERS:
**    char * OldPath         directory path to be set
**
** RETURN:
**    char NewPath[80]       converted directory string
**
*/

char *
FixPath( char * OldPath )
{
    static char NewPath[80];

    strcpy( NewPath, OldPath );

    if( NewPath[strlen(NewPath) - 1] != '\\' )
    {
        strcat( NewPath, "\\" );
    }

    return( NewPath );
}



/********************************************************************
** NAME:            systatrec * TGdat( char * BBSpath )
**
** DESCRIPTION:     Open TG STATUS.DAT, return ptr to it's struct
**
** PARAMETERS:
**    char * BBSpath         BBS directory containing STATUS.DAT
**
** RETURN:
**    systatrec * Status     Ptr to STATUS.DAT structure
**
** NOTES:
**    requires FILEPATH #define'd as filename + path length + '\0'
*/

systatrec *
TGdat( char * BBSpath )
{
    FILE * StatusFile;                 /* File pointer to STATUS.DAT */
    static systatrec * Status;         /* ptr to STATUS.DAT structure */
    char PathToStatus[FILEPATH];       /* Path + filename for STATUS.DAT */
        

    /* --- Create path + filename for STATUS.DAT --- */

    strcpy( PathToStatus, BBSpath );
    strcat( PathToStatus, "STATUS.DAT" );


    if( (StatusFile = fopen(PathToStatus, "rb")) == NULL )
    {
        fprintf( stderr, "(%d) ERROR: Cannot open %s\n",
                   __LINE__, PathToStatus );
        exit( EXIT_FAILURE );
    }


    /* --- Allocate memory for STATUS.DAT structure ptr --- */

    Status = ( systatrec * ) malloc( sizeof(systatrec) );

    if( Status == NULL )
    {
        fprintf( stderr, "(%d) ERROR: Unable to allocate memory\n" );
        exit( EXIT_FAILURE );
    }


    /* --- Read systatrec structure into memory at Status ptr --- */

    if( fread(Status, sizeof(systatrec), 1, StatusFile) == NULL )
    {
        fprintf( stderr, "(%d) ERROR: Cannot read %s\n",
                   __LINE__, PathToStatus );
        exit( EXIT_FAILURE );
    }

    fclose( StatusFile );

    return( Status );
}



/********************************************************************
** NAME:            char * Tg2C( char * Pstring )
**
** DESCRIPTION:     Convert TG Pascal strings to C format
**
** PARAMETERS:
**    char * Pstring         Telegard/Pascal formatted string
**
** RETURN:
**    static char * Cstring  Converted string in C format
**
** NOTES:
**    Telegard stores "heart" codes for color display in it's
**    strings.  0x03 is the ASCII code for the heart symbol.
**    We must check for the heart codes in the while loop so
**    we don't put them in the converted C string.
*/

char *
Tg2C( char * Pstring )
{
    register int Length = Pstring[0];  /* Pascal stores str length here */

    static char Cstring[256];          /* Converted string */

    register int p = 1;                /* Array position in Pstring */
    register int c = 0;                /* Array position in Cstring */


    /* --- Loop through TG string, convert to C string --- */

    while( p <= Length )
    {
        if( Pstring[p] == 0x03 || Pstring[p] == '^')
        {
            ++p;
        }

        else
        {
            Cstring[c++] = Pstring[p];
        }

        ++p;
    }

    Cstring[c] = '\0';                 /* NULL terminate string */

    return( Cstring );
}



/********************************************************************
** NAME:            void EraseString( char * String )
**
** DESCRIPTION:     Erase string, put cursor at string beginning.
**
** PARAMETERS:
**    char * String          String to be erased
**
** RETURN:
**    N/A
**
** NOTES: BACKSPACE = "\b \b" #define'd elsewhere
**
*/

void
EraseString( char * String )
{
    unsigned int Length = strlen( String );
    int Counter = 0;

    for( Counter = 0; Length > Counter; Counter++ )
    {
        printf( BACKSPACE );
    }
}
