/*********************************************************************
** FILENAME: lastword.c                     VERSION: 1.01
**
** DESCRIPTION: Simple quote door for Bulletin Board Systems
**
** NOTES:
**      This program was written using Opendoors v4.1 C library
**      kit.  It was linked to the small memory model of OpenDoors.
**
**      Copyright John Kristoff, 1993.  All rights reserved.
**
** AUTHOR: John Kristoff                START DATE: 11/14/93
**         Internet: jkristof@xroads.chigate.com
**                   jkristof@mica.meddean.luc.edu
**         FidoNet:  1:115/743
**
** CHANGES:
**      Fixed Logic errors including display of error message in
**      WriteFile().
**
** VERSION  DATE    WHO DETAIL
** 1.00     14Nov93 JK  Initial design and coding
** 1.01     16Nov93 JK  Minor bug fixes
**/

#include <stdio.h>                     /* Standard i/o */
#include <stdlib.h>                    /* EXIT_SUCCESS */
#include "opendoors.h"                 /* Opendoors functions */

#define VERS       "1.01"              /* Program version */

#define LINES      5                   /* Max lines per quote */
#define LINELEN    70                  /* Max line length <=75 */

#define BORDER     L_BLUE, D_BLACK     /* Quote box color */
#define TEXT       D_CYAN, D_BLACK     /* Quote text color */
#define NORMAL     D_GREY, D_BLACK     /* Normal text color */
#define HIGHLIGHT  L_CYAN, D_BLACK     /* Highlighted text color */

typedef struct
{
    char Quote[LINES][LINELEN];
    char User[36];
} LASTWORD;                            /* Quote and username data */

int main( int argc, char * argv[] );   /* If you don't know, YUSC */
void ProgramInfo( void );              /* Display title, author, etc. */
void ShowQuote( char * DataFile );     /* Display previous Quote */
void AddQuote( char * DataFile );      /* Get quote from user */

/* --- I/O functions --- */

LASTWORD * ReadFile( char * DataFile );
void WriteFile( LASTWORD * Lastword, char * DataFile );

/* --- Memory allocation functions --- */

LASTWORD * GetMem( void );

int
main( int argc, char * argv[] )
{
    char DataFile[81] = "LASTWORD.DAT";


    /* --- OpenDoors configuration, initialization, start-up info --- */

    strcpy( od_registered_to, "John Kristoff" );
    od_registration_key = 0000000000;
    od_control.od_clear_on_exit = FALSE;
    od_control.od_nocopyright = TRUE;


    /* --- Get drop file path and datafile if specified --- */

    if( argc > 1 )
    {
        strncpy( od_control.info_path, argv[1], 59 );
    }

    if( argc > 2 )
    {
        strncpy( DataFile, argv[2], 81 );
    }


    ProgramInfo();                     /* Display copyright info */

    ShowQuote( DataFile );             /* Show existing data to user */

    od_set_colour( NORMAL );
    od_disp_str( "Would you like to add some lastwords (Y/N):" );

    od_set_colour( HIGHLIGHT );

    if( od_get_answer( "YN" ) != 'N' )
    {
        od_disp_str( "Yes\n\r" );
        AddQuote( DataFile );          /* Get input, put it in datafile */
    }

    else
    {
        od_disp_str( "No\n\r" );
    }

    od_exit( 10, FALSE );              /* Normal exit */
    return( EXIT_SUCCESS );
}



/********************************************************************
** NAME:            void ProgramInfo( void )
**
** DESCRIPTION:     Display program name, copyright, author, etc.
**
** PARAMETERS:
**    N/A
**
** RETURN:
**    N/A
**
*/

void
ProgramInfo( void )
{
    od_set_colour( NORMAL );
    od_printf( "Lastword v" VERS ", " __DATE__
               " - Leave a quote to other users\n\r"
               "Copyright John Kristoff, 1993.  All rights reserved.\n\r"
               "\n\r" );
}



/********************************************************************
** NAME:            void ShowQuote( char * DataFile )
**
** DESCRIPTION:     Display existing quote from datafile if exists
**
** PARAMETERS:
**    char * DataFile       Path and filename for data file
**
** RETURN:
**    N/A
**
*/

void
ShowQuote( char * DataFile )
{
    LASTWORD * Lastword = NULL;        /* Pointer to quote struct */
    int CurrentLine = 0;               /* Where we are in Quote[] */


    /* --- Assume data file doesn't exist if ReadFile() fails --- */

    if( (Lastword = ReadFile( DataFile )) == NULL )
    {
        od_disp_str( "No Lastwords on file\n\r" );
    }

    else
    {
        od_set_colour( NORMAL );
        od_printf( "These Lastwords left by ");
        od_set_colour( HIGHLIGHT );
        od_printf( "%s\n\r", Lastword->User );
    

        /* --- Display top border --- */
    
        od_set_colour( BORDER );
        od_putch( '' );
        od_repeat( '', LINELEN + 1 );
        od_putch( '' );
        od_disp_str( "\n\r" );
        od_set_colour( TEXT );   
    

        /* --- Cycle through quote --- */
    
        while( (Lastword->Quote[CurrentLine][0] != NULL)
            && (CurrentLine < LINES) )
        {
            od_printf( "  %s\n\r", Lastword->Quote[CurrentLine++] );
        }
    

        /* --- Display bottom border --- */
    
        od_set_colour( BORDER );
        od_putch( '' );
        od_repeat( '', LINELEN + 1 );
        od_putch( '' );
        od_disp_str( "\n\r" );    
    }                                  /* else */
}



/********************************************************************
** NAME:            void AddQuote( char * DataFile )
**
** DESCRIPTION:     Get quote from user
**
** PARAMETERS:
**    char * DataFile       Path and filename for data file
**
** RETURN:
**    N/A
**
*/

void
AddQuote( char * DataFile )
{
    LASTWORD * Lastword = NULL;        /* Pointer to quote struct */
    char QuoteLine[LINELEN] = "";      /* Initialize to empty string */
    int LineCount = 0;                 /* Keep track of current line */
    int BlankLine = FALSE;             /* If input line is empty */

    Lastword = GetMem();               /* Allocate memory for struct */

    od_set_colour( NORMAL );
    od_disp_str( "Enter up to " );
    od_set_colour( HIGHLIGHT );
    od_printf( "%d", LINES );
    od_set_colour( NORMAL );
    od_disp_str( " lines.  A blank line ends.\n\r" );
    od_disp_str( "\n\r" );


    /* --- Cycle through input routine --- */

    while( (LineCount < LINES) && (BlankLine != TRUE) )
    {
        od_set_colour( HIGHLIGHT );
        od_printf( "%d: ", LineCount + 1 );
        od_set_colour( NORMAL );
        od_input_str( QuoteLine, LINELEN - 1, 32, 127 );

        if( (QuoteLine[0] != '\0') || (QuoteLine[0] != NULL) )
        {
            strncpy( Lastword->Quote[LineCount++], QuoteLine, LINELEN );
        }        

        else
        {
            BlankLine = TRUE;
        }
    }


    /* --- If input was entered, save it --- */

    if( LineCount != 0 )
    {
        strcpy( Lastword->User, od_control.user_name );
        WriteFile( Lastword, DataFile );
    }
}



/********************************************************************
** NAME:            LASTWORD * ReadFile( char * DataFile )
**
** DESCRIPTION:     Read data file, return pointer to struct
**
** PARAMETERS:
**    char * DataFile       Path and filename for data file
**
** RETURN:
**    LASTWORD * tmp        Pointer to LASTWORD struct
**
*/

LASTWORD *
ReadFile( char * DataFile )
{
    FILE * fp = NULL;                  /* Data file containing quote */
    LASTWORD * tmp = NULL;             /* Struct read from data file */


    /* --- Open data file, allocate memory, read data in --- */

    if( (fp = fopen( DataFile, "rb" )) != NULL )
    {
        tmp = GetMem();
        fread( tmp, sizeof(LASTWORD), 1, fp );
        fclose( fp );
    }

    return( tmp );                     /* Return ptr to struct or NULL */
}



/********************************************************************
** NAME:            void WriteFile( LASTWORD *, char * DataFile )
**
** DESCRIPTION:     Write LASTWORD struct to DataFile
**
** PARAMETERS:
**    LASTWORD * Lastword   Pointer to LASTWORD struct data
**    char * DataFile       Path and filename for data file
**
** RETURN:
**    N/A
**
*/

void
WriteFile( LASTWORD * Lastword, char * DataFile )
{
    FILE * fp = NULL;                  /* Data file containing quote */

    if( (fp = fopen( DataFile, "wb" )) == NULL )
    {
        od_printf( "(%d) ERROR: Cannot open %s\n\r",
                    __LINE__, DataFile );
        od_exit( 15, FALSE );
    }

    if( fwrite(Lastword, sizeof(LASTWORD), 1, fp) == NULL )
    {
        od_printf( "(%d) ERROR: Cannot write %s\n\r",
                    __LINE__, DataFile );
        od_exit( 15, FALSE );
    }

    fclose( fp );
}



/********************************************************************
** NAME:            LASTWORD * GetMem( void )
**
** DESCRIPTION:     Allocate memory for LASTWORD struct
**
** PARAMETERS:
**    N/A
**
** RETURN:
**    LASTWORD * tmp        Pointer to allocated struct
**
*/

LASTWORD *
GetMem( void )
{
    LASTWORD * tmp = NULL;             /* Ptr to pre-allocated struct */

    tmp = ( LASTWORD * ) calloc( 1, sizeof(LASTWORD) );
    if( tmp == NULL )
    {
        od_printf( "(%d) ERROR: Memory error\n\r", __LINE__ );
        od_exit( 15, FALSE );
    }

    return( tmp );
}
