/*
 * File......: TBPRINT.PRG
 * Author....: Martin Colloby
 * BBS.......: The Dark Knight Returns
 * Net/Node..: 050/069
 * User Name.: Martin Colloby
 * Date......: 18/4/93
 * Revision..: 1.0
 *
 * This is an original work by Martin Colloby and is placed in the public
 * domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */


/*  $DOC$
 *  $FUNCNAME$
 *      GT_PRINTTBROWSE()
 *  $CATEGORY$
 *      Printer
 *  $ONELINER$
 *      Print the contents of a TBrowse object
 *  $SYNTAX$
 *      GT_PrintTBrowse( oBrowse , cScope )
 *  $ARGUMENTS$
 *      oObject - TBrowse object
 *      cScope  - Scope of TBrowse
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      Prints the contents of the TBrowse object, using all of the TBrowse
 *      whizzy bits like column headers and footers.
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *      GT_LIB.CH
 *  $END$
 */
*
#include "GT_lib.ch"

FUNCTION GT_PrintTBrowse( oBrowse , cScope )

/*****************************************************************************
 Purpose - Print the contents of the TBrowse object
 Returns - None
 Author  - Log
 Created - 30/12/92
******************************************************************************
 Parameters - oBrowse - TBrowse object
              cScope  - First incidence of scope
 Privates   - None
 Locals     - None
 Externals  - None
*****************************************************************************/

LOCAL aColHeading := {}
LOCAL aColumn     := {}
LOCAL aWidths     := GT_ColWidth( oBrowse )
LOCAL cBColSep    := IIF( oBrowse:colSep  == NIL , " " , oBrowse:colSep  )
LOCAL cBFootSep   := IIF( oBrowse:footSep == NIL , " " , oBrowse:footSep )
LOCAL cBHeadSep   := IIF( oBrowse:headSep == NIL , " " , oBrowse:headSep )
LOCAL cColSep     := ""
LOCAL cFootSep    := ""
LOCAL cHeadSep    := ""
LOCAL cLine       := ""
LOCAL nColRows    := 0
LOCAL nCount      := 0
LOCAL nCount1     := 0
LOCAL nLines      := 0
LOCAL nMaxLines   := 0
LOCAL oColumn
LOCAL xValue

GT_NextRow( 1 , "" , 5 )

* Get the column headers
FOR nCount := 1 TO oBrowse:colCount()
    cColSep := IIF( oBrowse:getColumn( nCount ):colSep == NIL , cBColSep , oBrowse:getColumn( nCount ):colSep )
    aColumn := GT_ColHead( oBrowse:getcolumn( nCount ):Heading(), aWidths[ nCount ] , cColSep )
    AADD( aColHeading , aColumn )
    nColRows := MAX( nColRows , LEN( aColumn ) )
NEXT nCount

* Output the column titles
FOR nCount := 1 TO nColRows
    FOR nCount1 := 1 TO oBrowse:colCount()
        cColSep := IIF( oBrowse:getColumn( nCount1 ):colSep == NIL , cBColSep , oBrowse:getColumn( nCount1 ):colSep )
        cLine += IIF( LEN( aColHeading[ nCount1 ] ) >= nCount , ;
                      aColHeading[ nCount1 , nCount ] , ;
                      SPACE( aWidths[ nCount1 ] + LEN( cColSep ) ) )
    NEXT nCount1
    * Next two lines should print the column header, and clear the string
    GT_NextRow( 1 , cLine , 5 )
    cLine := ""
NEXT nCount

* Output the column headings
cLine := ""
FOR nCount := 1 TO oBrowse:colCount()
    cHeadSep := IIF( oBrowse:getColumn( nCount ):headSep == NIL , cBHeadSep , oBrowse:getColumn( nCount ):headSep )
    cLine += REPLICATE( LEFT( cHeadSep , 1 ) , oBrowse:getColumn( nCount ):width ) + cHeadSep
NEXT nCount
cLine += RIGHT( cHeadSep , 1 )
IF ALLTRIM( cLine ) != ""
    GT_NextRow( 1 , cLine , 5 )
ENDIF

* Go to the first record
MovePos( "First" , cScope )

* Skip through the records in the TBrowse.
DO WHILE .T.
    * Build up a string containing the values
    cLine := ""
    FOR nCount := 1 TO oBrowse:colCount
        cColSep := IIF( oBrowse:getColumn( nCount ):colSep == NIL , cBColSep , oBrowse:getColumn( nCount ):colSep )
        cLine += IIF( nCount == 1 , " " , cColSep ) + ;
                 GT_X2Char( EVAL( oBrowse:getColumn( nCount ):block ) , aWidths[nCount] )
    NEXT nCount

    * Print this row
    GT_NextRow( 1 , cLine , 5 )

    * Evaluate the skipblock for the browse by asking it to skip one record.
    * If it returns a value other than 1, the pointer must be at the end of
    * the scope.
    IF EVAL( oBrowse:skipBlock , 1 ) != 1
        EXIT
    ENDIF

ENDDO

cLine := ""
FOR nCount := 1 TO oBrowse:colCount()
    cFootSep := IIF( oBrowse:getColumn( nCount ):footSep == NIL , cBFootSep , oBrowse:getColumn( nCount ):footSep )
    cLine += REPLICATE( LEFT( cFootSep , 1 ) , oBrowse:getColumn( nCount ):width ) + cFootSep
NEXT nCount
cLine += RIGHT( cFootSep , 1 )
IF ALLTRIM( cLine ) != ""
    GT_NextRow( 1 , cLine , 5 )
ENDIF

* Go back top the top of the browse
oBrowse:goTop()

RETURN NIL
*
*!************************************************************
*!             FUNCTION getColumnHead
*!   Purpose : Create array for column header, storing seperate lines
*!             of header (parts seperated by semicolons) in seperate
*!             array elements, centering based on column width
*!   In      : cHeading - column header which may contain semicolons.
*!           : nColWidth - width for column, to be used for centering.
*!   Out     : array for this column heading.
*!************************************************************
STATIC FUNCTION GT_ColHead( cHeading , nColWidth , cColSep )

LOCAL aColumn := {}     // array to be filled out and returned.
LOCAL cChar   := ""     // temp. holder for each character.
LOCAL cTemp   := ""     // temp. holder for each header element.
LOCAL nCount  := 0      // counter

DO WHILE ++nCount <= LEN( cHeading )
    cChar := SUBSTR( cHeading , nCount , 1 )
    IF cChar == ";"
        AADD( aColumn , PADC( cTemp , nColWidth + LEN( cColSep ) ) )
        cTemp := ''
    ELSE
        cTemp += cChar
    ENDIF
ENDDO

* Pad the title with spaces to the right, so that the title will appear
* at the left of the column
AADD( aColumn , PADR( cTemp, nColWidth + LEN( cColSep ) ) )

RETURN aColumn
*
*!************************************************************
*!             FUNCTION GetColumnLen
*!   Purpose : Create array containing column widths for tbrowse,
*!             determined by looking at column width, header, footer
*!             and block length.
*!   In      : oBrowse - tbrowse object
*!   Out     : aLen - array containing width of each column in tbrowse.
*!************************************************************
STATIC FUNCTION GT_ColWidth( oBrowse )

LOCAL aLen     := {}    // array of widths to return.
LOCAL nCount   := 0
LOCAL nThisLen := 0     // variable to hold current column width.

FOR nCount := 1 TO oBrowse:colCount()
    // get the Column Width
    IF oBrowse:getcolumn( nCount ):width == NIL
        // if column width not set by user column width is largest of
        //  len of heading, len of footing, or len of eval( block )
        nThisLen := MAX( LEN( GT_X2Char( EVAL( oBrowse:getcolumn( nCount ):Block() ) ) ) , ;
                         MAX( HeadLen( oBrowse:getcolumn( nCount ):Heading() ) , ;
                              HeadLen( oBrowse:getcolumn( nCount ):Footing() )   ;
                             )                                              ;
                        )
    ELSE
        nThisLen := oBrowse:getcolumn( nCount ):width
    ENDIF

    AADD( aLen , nThisLen )
NEXT nCount

RETURN aLen
*
*!************************************************************
*!             FUNCTION HeadLen
*!   Purpose : Returns max width of header which may contain semicolons.
*!   In      : cHeading - header to determine max length of.
*!   Out     : nMaxLen - maximum length of cHeading
*!************************************************************
STATIC FUNCTION HeadLen( cHeading )

LOCAL cChar   := ""     // current character
LOCAL nCount  := 0      // loop counter
LOCAL nLen    := 0      // current segment width.
LOCAL nMaxLen := 0      // holder of max width found so far

DEFAULT cHeading TO ""

DO WHILE ++nCount <= LEN( cHeading )
    cChar := SUBSTR( cHeading , nCount , 1 )
    IF cChar == ';'
        nMaxLen := MAX( nMaxLen , nLen )
        nLen := 0
    ELSE
        nLen ++
    ENDIF
ENDDO

RETURN nMaxLen
*
