/*
 * File......: BLATBOX.PRG
 * Author....: Martin Colloby
 * BBS.......: The Dark Knight Returns
 * Net/Node..: 050/069
 * User Name.: Martin Colloby
 * Date......: 18/4/93
 * Revision..: 1.0
 *
 * This is an original work by Martin Colloby and is placed in the public
 * domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */


/*  $DOC$
 *  $FUNCNAME$
 *      GT_BLATBOX()
 *  $CATEGORY$
 *      Video
 *  $ONELINER$
 *      Draw a double lined box with a shadow around a text string
 *  $SYNTAX$
 *      GT_BlatBox( aString , nRow , nCol , cColour )
 *  $ARGUMENTS$
 *      aString - Array containg string to be output - each element will
 *                appear on a separate line
 *      nRow    - Top row of box
 *      nCol    - Left column of box ( -1 to centre )
 *      cColour - Colour string
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      Draws a double lined box on screen, with a shadow around the box.
 *      The text in aString will appear in the box, with each element
 *      appearing on a new line.
 *  $EXAMPLES$
 *      Show the two text lines "Line1" and "Line2" in a box horizontally
 *      centred from line 10, in White on Blue :
 *
 *          GT_BlatBox( { "Line 1" , "Line 2" } , 10 , -1 , "W+/B" )
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *      GT_LIB.CH
 *  $END$
 */

*
#include "GT_LIB.ch"

FUNCTION GT_BlatBox( aString , nRow , nCol , nColour )

/****************************************************************************
 Purpose - Output a box around a string
 Returns - Length of longest line
 Author  - Martin Colloby
 Created - 12/9/90
 Edited  - 29/5/91 by Martin Colloby - Added nRow parameter
           26/6/91 by Martin Colloby - Added return value
******************************************************************************
 Parameters - aString   - Array containing string
              nRow      - Row at which to start box
 Private    - cOldColor - Temporary storage for colour setup
              nCount    - Counting variable
              nLCol     - Screen column for output
              nLength   - Length of longest line in aString
              nLines    - Number of lines in aString
              nLRow     - Screen row to output next line to
 Externals  - None

 N.B. Tilde (~) character can be used to split string onto > 1 line
****************************************************************************/

LOCAL cOldColor := ""
LOCAL nCount   := 0
LOCAL nCursor  := SETCURSOR( 0 )
LOCAL nLCol    := 0
LOCAL nLength  := 0
LOCAL nLines   := LEN( aString )
LOCAL nLRow    := 0

* Find the length of the longest line in the string
FOR nCount := 1 TO nLines
    nLength := MAX( nLength , LEN( aString[nCount] ) + 1 )
NEXT nCount

* Starting column is such that box is centred on screen
nLCol := INT( ( MAXCOL() + 1 - nLength + 2 ) / 2 ) - 1

* If nRow is -1 the box is to be placed such that the last line is at row 22
IF nRow == -1
    nLRow := MAXROW() - 2 - nLines - 2
ELSE
    * If nRow is such that the bottom of the box would go off the edge of the
    * screen, flag a warning
    IF nRow > MAXROW() - 2 - nLines - 2
        GT_Warning( { "GT_BlatBox Error - nRow is too close to bottom of screen !" } )
    ELSE
        NLRow := nRow
    ENDIF
ENDIF

* Draw a shadow
GT_Shadow( nLRow , nLcol , nLRow + nLines + 1 , nLcol + nLength + 2 )

GT_ColorSet( nColour )
@ nLRow , nLCol , nLRow + nLines + 1 , nLCol + nLength + 2 BOX B_DOUBLE

FOR nCount := 1 TO nLines
    nLRow ++
    * Draw the middle lines of the box, with the string in it
    @ nLRow, nLCol + 1 SAY PADR( " " + aString[nCount] , nLength + 1 )
NEXT cCount

SETCURSOR( nCursor )

RETURN( nLength )
*
