/*
 * File......: LLFLOCK.C
 * Author....: Dave Pearson
 * BBS.......: The Dark Knight Returns
 * Net/Node..: 050/069
 * User Name.: Dave Pearson
 * Date......: 23/03/93
 * Revision..: 1.0
 *
 * This is an original work by Dave Pearson and is placed in the public
 * domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

// NOTE: This code has been written for and compiled with Borland C++
//       Version 3.1
//

#include "gt_mem.h"
#include "Extend.H"

Boolean _FileLocking(int, char, long, long);

#define LOCK_TYPE_LOCK          0
#define LOCK_TYPE_UNLOCK        1

/*  $DOC$
 *  $FUNCNAME$
 *      GT_FLOCK()
 *  $CATEGORY$
 *      File I/O
 *  $ONELINER$
 *      Lock a region of a low level file.
 *  $SYNTAX$
 *      GT_FLock(<nHandle>,<nOffset>,<nLength>) --> lLocked
 *  $ARGUMENTS$
 *      <nHandle> is a numeric file handle.
 *
 *      <nOffset> is the byte position at which the lock will start. This
 *                is a zero based value.
 *
 *      <nLength> is the length of the region to lock.
 *
 *  $RETURNS$
 *      A logical value, .T. if the region was locked, .F. if not.
 *  $DESCRIPTION$
 *      GT_FLock() can be used to lock a region of a low level file.
 *
 *      To use the low-level locking/un-locking functions you will first
 *      need to open the file with fopen(), or perhaps create a file with
 *      fcreate(). Once open you can use the file handle returned by
 *      Clipper to carry out a lock.
 *
 *      When you have locked an area of the file, be sure to store the
 *      offset and length, these values will be required by GT_FUnLock().
 *
 *  $EXAMPLES$
 *
 *      #include "fileio.ch"
 *
 *      if (nFile := fopen("LIFE.42",FO_SHARED)) != F_ERROR
 *
 *         // Lock a region in the file at position 10 (don't forget,
 *         // the offset starts at 0, not 1!
 *
 *         if GT_FLock(nFile,10,2)
 *
 *            // Make some changes to the file.
 *
 *            GT_FUnLock(nFile,10,2)
 *         endif
 *         fclose(nFile)
 *      endif
 *
 *  $SEEALSO$
 *      GT_FUNLOCK()
 *  $END$
 */

CLIPPER GT_FLock()
{
        Boolean Result = FALSE;

        if (PCOUNT >= 3)
        {
                if (ISNUM(1) && ISNUM(2) && ISNUM(3))
                {
                        Result = _FileLocking(_parni(1),LOCK_TYPE_LOCK,_parnl(2),_parnl(3));
                }
        }
        _retl(Result);
}

/*  $DOC$
 *  $FUNCNAME$
 *      GT_FUNLOCK()
 *  $CATEGORY$
 *      File I/O
 *  $ONELINER$
 *      Unlock a region of a low level file.
 *  $SYNTAX$
 *      GT_FUnLock(<nHandle>,<nOffset>,<nLength>) --> lUnLocked
 *  $ARGUMENTS$
 *      <nHandle> is a numeric file handle.
 *
 *      <nOffset> is the byte position at which the unlock will start. This
 *                is a zero based value.
 *
 *      <nLength> is the length of the region to unlock.
 *
 *  $RETURNS$
 *      A logical value, .T. if the region was unlocked, .F. if not.
 *  $DESCRIPTION$
 *      GT_FUnLock() can be used to unlock a region of a low level file
 *      that has been locked withg GT_FLock().
 *
 *      To use the low-level locking/un-locking functions you will first
 *      need to open the file with fopen(), or perhaps create a file with
 *      fcreate(). Once open you can use the file handle returned by
 *      Clipper to carry out a lock.
 *
 *  $EXAMPLES$
 *
 *      #include "fileio.ch"
 *
 *      if (nFile := fopen("LIFE.42",FO_SHARED)) != F_ERROR
 *
 *         // Lock a region in the file at position 10 (don't forget,
 *         // the offset starts at 0, not 1!
 *
 *         if GT_FLock(nFile,10,2)
 *
 *            // Make some changes to the file.
 *              .
 *              .
 *              .
 *            // Now unlock the same region.
 *
 *            GT_FUnLock(nFile,10,2)
 *         endif
 *         fclose(nFile)
 *      endif
 *
 *  $SEEALSO$
 *      GT_FLOCK()
 *  $END$
 */

CLIPPER GT_FUnLock()
{
        Boolean Result = FALSE;

        if (PCOUNT >= 3)
        {
                if (ISNUM(1) && ISNUM(2) && ISNUM(3))
                {
                        Result = _FileLocking(_parni(1),LOCK_TYPE_UNLOCK,_parnl(2),_parnl(3));
                }
        }
        _retl(Result);
}

/*****************************************************************************
* Function: _FileLocking()                                                   *
* Syntax..: Boolean _FileLocking(int Handle, char LockType, long Offset,     *
* ........:                      long Length)                                *
* Usage...: Internal function to do the actual locking.                      *
* By......: David A Pearson                                                  *
*****************************************************************************/

static Boolean _FileLocking(int Handle, char LockType, long Offset, long Length)
{
        int     HiOffset = _GT_FP_SEG(Offset);
        int     LoOffset = _GT_FP_OFF(Offset);
        int     HiLength = _GT_FP_SEG(Length);
        int     LoLength = _GT_FP_OFF(Length);

        asm     Mov     AH,0x5C;
        asm     Mov     AL,LockType;
        asm     Mov     BX,Handle;
        asm     Mov     CX,HiOffset;
        asm     Mov     DX,LoOffset;
        asm     Mov     SI,HiLength;
        asm     Mov     DI,LoLength;
        asm     Int     0x21;
        return((_FLAGS & 1) == 0);
}
