/*
    File......: GT_GetList.prg
    Author....: Martin Bryant
    BBS.......: The Dark Knight Returns
    Net/Node..: 050/069
    User Name.: Martin Bryant
    Date......: 09/03/93
    Revision..: 1.0

    This is an original work by Martin Bryant and is placed
    in the public domain.

    Modification history:
    ---------------------

    Rev 1.0 09/03/93
    PD Revision.
*/
/*  $DOC$
 *  $FUNCNAME$
 *      GT_GETLIST()
 *  $CATEGORY$
 *      General
 *  $ONELINER$
 *      User list for a Get.
 *  $SYNTAX$
 *      GT_GetList(<oGet>,<cType>,<bReturn>,<bValid>, ;
 *          [<bFind>],[<bFor>],[<nTop>],[<nLeft>]) -> lSuccess
 *  $ARGUMENTS$
 *      <oGet> is the Get Object passed to the post block.
 *
 *      <cType> is the type of window to use. 'A' uses an
 *      array list and 'D' uses a Browse on a DBF file.
 *
 *      <bReturn> block to extract the information to put
 *      into the Get field. It will be passed the element
 *      of the array or the record number (current) in the
 *      Browse case.
 *
 *      <bValid> the block stating whether the result in the
 *      Get field is valid.
 *
 *      <bFind> defines which option to start on. Should
 *      return a number for 'A' windows or a value to seek
 *      with the browse. For Browses this will act as a
 *      while clause against the index.
 *
 *      <bFor> for clause. Only for Browse lists.
 *
 *      <nTop> and <nLeft> specify the top left corner of
 *      the window.
 *  $RETURNS$
 *      Logical valid status.
 *  $DESCRIPTION$
 *      Allow a PostBlock on a get to display the user
 *      a list of legal options from an array or an open
 *      datafile. Allows while/for options and displays the
 *      list on screen. When an option is selected the list
 *      is removed.
 *      Requires extra information in :Cargo
 *  $EXAMPLES$
 *      oGet:PostBlock := { | oGet | Suppliers->(Gt_GetList( ;
 *          oGet, ;                         // Get object
 *          'D', ;                          // Datafile
 *          { | | Field->Code }, ;          // Return
 *          { | cData | DBSEEK(cData) }, ;  // Valid
 *          { | cData | RTRIM(cData) }, ;   // Seek/while
 *          { | | Field->Qty > 0 }, ;       // For
 *          02, ;                           // Top
 *          02)) }                          // Left
 *
 *      aGets[01][01]:Cargo := { ;
 *          {'Code','Supplier'}, ;          // Titles
 *          {{ | | Field->SuppCode }, ;
 *              { | | Field->SuppName }}, ; // List fields
 *          {}}                             // Only used with array
 *
 *  $SEEALSO$
 *      Gt_Choose() GT_BROWSE()
 *  $INCLUDE$
 *
 *  $END$
 */

#include "GTClippe.ch"
MEMVAR uData
MEMVAR p_highcolours
MEMVAR p_errcolors

#define LISTTITLE   01
#define LISTDATA    02
#define LISTVALID   03

FUNCTION GT_GetList(oGet,cType,bReturn,bValid,bFind,bFor, ;
    nTop,nLeft)

LOCAL aCargoData := {'',{},{}}
LOCAL bIndex := NIL
LOCAL cColour := SETCOLOR()
LOCAL cScreen := SAVESCREEN(00,00,MAXROW(),MAXCOL())
LOCAL lFound := .F.
LOCAL nBottom := -1
LOCAL nCount := -1
LOCAL nRight := -1
LOCAL nSelect := 0
LOCAL nStart := -1

PRIVATE uData := NIL

Default oGet to NIL
Default cType to ''
Default bReturn to { | uData | uData }
Default bValid to { | uData, nSelect | (nSelect > 0) }
Default bFind to NIL
Default bFor to NIL
Default nTop to 04
Default nLeft to 02

//  Any object ?
IF oGet = NIL .OR. EMPTY(cType)
    RETURN(.T.)
ENDIF

//  Extract data
uData := EVAL(oGet:Block)

//  Set cargo array
aCargoData := oGet:Cargo

//  Valid ?
IF EVAL(bValid,uData,0,aCargoData[LISTDATA])
    IF cType = 'A'
//        EVAL(oGet:Block,EVAL(bReturn, ;
//            aCargoData[LISTDATA][nSelect]))
    ELSE
        EVAL(oGet:Block,EVAL(bReturn))
    ENDIF
    RETURN(.T.)
ENDIF

//  Which sort of list ?
IF cType = 'A'

    // Bottom
    nBottom := MIN(nTop+03+LEN(aCargoData[LISTDATA]), ;
        MAXROW()-04)

    // Right
    nRight := MIN(nLeft+01+MAX(LEN(aCargoData[LISTTITLE]), ;
        LEN(aCargoData[LISTDATA][01])),MAXCOL()-02)

    // Start ?
    nStart := IF(bFind=NIL,1,EVAL(bFind,uData))

    // List array
    SETCOLOR(p_HighColors)
    nSelect := GT_Choose(aCargoData[LISTDATA], ;
        aCargoData[LISTTITLE],nTop,nLeft,nBottom,nRight, ;
        aCargoData[LISTVALID],nStart)
    SETCOLOR(cColour)

    // Restore screen
    RESTSCREEN(00,00,MAXROW(),MAXCOL(),cScreen)

    // Use ?
    IF nSelect > 0

        EVAL(oGet:Block,EVAL(bReturn, ;
            aCargoData[LISTDATA][nSelect]))

    ENDIF

ELSE

    // Bottom
    nBottom := MIN(nTop+LASTREC()+03,MAXROW()-04)

    // Right
    nRight := nLeft + LEN(aCargoData[LISTDATA])
    FOR nCount := 1 TO LEN(aCargoData[LISTDATA])

        nRight += MAX( ;
            LEN(EVAL(aCargoData[LISTDATA][nCount])), ;
            LEN(aCargoData[LISTTITLE][nCount]))

    NEXT
    IF nRight > MAXCOL() - 02
        nRight := MAXCOL() - 02
    ENDIF

    // Fix data file
    DO CASE
        CASE bFind = NIL .AND. bFor = NIL
            // All
            DBGOTOP()
            lFound := .NOT. EOF()

        CASE bFind = NIL
            // For condition only
            DBGOTOP()
            DBEVAL({ | | NIL },NIL,{ | | .NOT. EVAL(bFor) })
            lFound := EVAL(bFor)

        CASE bFor = NIL
            // While condition only
            lFound := DBSEEK(EVAL(bFind,uData))

        OTHERWISE
            // Both
            IF DBSEEK(EVAL(bFind,uData))
                bIndex := GT_IndexBlock(0)
                DBEVAL({ | | NIL },NIL, ;
                    { | | (EVAL(bIndex)=EVAL(bFind,uData)) .AND. ;
                        .NOT. EVAL(bFor) })

                lFound := EVAL(bFor) .AND. ;
                    (EVAL(bIndex) = EVAL(bFind,uData))
            ELSE
                lFound := .F.
            ENDIF

    ENDCASE

    IF lFound
        IF bFind != NIL
            uData := EVAL(bFind,uData)
            IF EMPTY(uData)
                bFind := NIL
            ELSE
                bFind := &('{ | | "' + uData + '" }')
            ENDIF
        ENDIF
        SETCOLOR(p_HighColors)
        IF GT_Browse(aCargoData[LISTDATA], ;
            aCargoData[LISTTITLE],'Options:',nTop,nLeft, ;
            nBottom,nRight,bFind,bFor,1)

            nSelect := Recno()
        ENDIF
        SETCOLOR(cColour)
    ELSE
        GT_AskUser('No Items found. If you have asked for a selection, please blank the field and try again.',NIL,'Error:000',p_ErrColors)
    ENDIF

    // Restore screen
    RESTSCREEN(00,00,MAXROW(),MAXCOL(),cScreen)
    // Use ?
    IF nSelect > 0

        EVAL(oGet:Block,EVAL(bReturn,nSelect))

    ENDIF

ENDIF
/*
    End of GT_GetList()
*/
RETURN(nSelect>0) // 

