/*
-------------------------------------------------------------------------------
GEOM.C  (Geometric Image Operations)

CREATOR: Craig Muller P.Eng. 1991,1992,1993
			cmuller@ccu.umanitoba.ca
			Computer Vision Laboratory
			University of Manitoba
			Winnipeg, Manitoba. R3T 2N2
			CANADA.

DESCRIPTION:

These procedures provide a general purpose interface to the image
data and only have compatibility with the Windows environment.

PROCEDURES:
ReduceImage()    - Creates a new image which is a reduction of the original.
ScaleImage()     - Creates a new image which scales the original into it.
MirrorImage()    - Creates a new image which is a mirror of the original
FlipImage()      - Creates a new image which is a flip of the original
RotateImage()    - Creates a new image which is a rotation of the original
-------------------------------------------------------------------------------
*/
#include <windows.h>

#pragma hdrstop

#include <stdio.h>
#include <dos.h>
#include <math.h>
#include <mem.h>
#include <alloc.h>
#include <string.h>

#include "kernel.h"

//---------------------------------------------------------------------------
// Procedure prototypes.
//---------------------------------------------------------------------------
IMAGE *MirrorImage(IMAGE *s_image);
IMAGE *FlipImage  (IMAGE *s_image);
IMAGE *ScaleImage (IMAGE *s_image,int w,int h);
IMAGE *RotateImage(IMAGE *s_image,int dir);
IMAGE *ReduceImage(IMAGE *s_image,int reduction);

void ScaleImageRow(IMAGE *s_image,IMAGE *d_image,int y);


/*
---------------------------------------------------------------------------
MirrorImage()
~~~~~~~~~~~~

Creates a new image with the same dimensions and then mirrors the original
image into that space. The orginal image is not destroyed.
---------------------------------------------------------------------------
*/
IMAGE *MirrorImage(IMAGE *s_image)
	{
	int col,w,h;
	BYTE  *buf;
	IMAGE *image;

	// Set the dimensions of the processing rectangle
	w = s_image->hres;
	h = s_image->vres;

	// Create a new image to hold the output data.
	image = DuplicateImage(s_image);

	// Mirror the image data using a temporary storage buffer.
	buf = (BYTE *)malloc(h);            // Allocate scan buffer.
	for (col=0; col<w; ++col)           // For each row in the image
		{                                          // Write the row to the col
		GetImageCol(s_image,col   ,0,buf,h);
		PutImageCol(image,w-1-col,0,buf,h);
		}
	free(buf);                                    // Free the scan buffer.

	// Return the new image.
	return(image);
	}

/*
--------------------------------------------------------------------------
FlipImage()
~~~~~~~~~~~~

Creates a new image with the same dimensions and then flips the original
image into that space. The orginal image is not destroyed.
--------------------------------------------------------------------------
*/
IMAGE *FlipImage(IMAGE *s_image)
	{
	int row,w,h;
	BYTE  *buf;
	IMAGE *d_image;

	// Set the dimensions of the processing rectangle
	w = s_image->hres;
	h = s_image->vres;

	// Create a new image to hold the output data.
	d_image = DuplicateImage(s_image);

	// Flip the image data using a temporary storage buffer.
	buf = (BYTE *)malloc(d_image->hres);            // Allocate scan buffer.
	for (row=0; row<h; ++row)                     // For each row in the image
		{                                          // Write the row to the col
		GetImageRow(s_image,0  ,row,buf,w);
		PutImageRow(d_image,0,h-1-row,buf,w);
		}
	free(buf);                                    // Free the scan buffer.

	// Return the new image.
	return(d_image);
	}


/*
--------------------------------------------------------------------------
ScaleImage()
~~~~~~~~~~~~

Creates a new image with the dimensions given and then scales the original
image into that space. The orginal image is not destroyed.
--------------------------------------------------------------------------
*/
IMAGE *ScaleImage(IMAGE *s_image,int w,int h)
	{
	int  x,y,x0,y0;
	BYTE  *irow;
	BYTE  *orow;
	IMAGE *d_image;

	// Create new image buffer
	d_image = CreateImage(w,h);

	// Copy image parameters
	d_image->color = s_image->color;
	strcpy(d_image->fspec,s_image->fspec);

	irow = (BYTE *)malloc(s_image->hres);          // Allocate input buffer.
	orow = (BYTE *)malloc( d_image->hres);          // Allocate output buffer.

	// Transform the data to the new image
	for (y=0; y<h; ++y)
		{
		y0 = (long)(y) * (long)s_image->vres/(long)h;
		GetImageRow(s_image,0,y0,irow,s_image->hres);// Read pixel row from image
		for (x=0; x<w; ++x)
			{
			x0 = (long)(x) * (long)s_image->hres/(long)w;
			orow[x] = irow[x0];
			}
		PutImageRow(d_image,0,y,orow,d_image->hres);   // Write output row
		}

	free(irow);                                   // Free input buffer.
	free(orow);                                   // Free output buffer.

	// Copy palette data to the new image
	CopyImagePal(s_image,d_image);

	return(d_image);
	}



/*
--------------------------------------------------------------------------
RotateImage()
~~~~~~~~~~~~

Creates a new image with the dimensions reversed and then rotates the original
image left or right to fit into that space. The orginal image is not destroyed.
--------------------------------------------------------------------------
*/
IMAGE *RotateImage(IMAGE *s_image,int dir)
	{
	int  row,x,y,w,h,cb;
	BYTE  *buf;
	IMAGE *d_image;

	w = s_image->vres;
	h = s_image->hres;

	// Create new image buffer
	d_image = CreateImage(w,h);

	// Copy image parameters
	d_image->color = s_image->color;
	strcpy(d_image->fspec,s_image->fspec);

	// Transform the data to the new image
	buf = (BYTE *)malloc(s_image->hres);            // Allocate scan buffer.
	for (row=0; row<s_image->vres; ++row)           // For each row in the image
		{                                          // Write the row to the col
		x  = 0;
		y  = row;
		cb = s_image->hres;
		GetImageRow(s_image,x,y,buf,cb);               // Read pixel row from image

		if (dir == -1)
			{ x  = row; y  = s_image->hres-1; cb = -s_image->hres; }
		if (dir == 1)
			{ x  = s_image->vres-1-row; y = 0; cb = s_image->hres; }

		PutImageCol(d_image,x,y,buf,cb);               // Transpose to column
		}
	free(buf);                                    // Free the scan buffer.

	// Copy palette data to the new image
	CopyImagePal(s_image,d_image);

	// Return the new image.
	return(d_image);
	}

/*
--------------------------------------------------------------------------
int  ReduceImage(IMAGE *image,BYTE huge *hpBuf,int reduction)

DESCRIPTION:
Copy a complete image from source buffer to destination buffer. At
the same time an image reduction is performed to shrink the image.
Only integer reductions are performed.
--------------------------------------------------------------------------
*/
IMAGE *ReduceImage(IMAGE *s_image,int reduction)
	{
	int  d,w,h,x,y;
	BYTE huge *hpDataIn;
	BYTE huge *hpDataOut;
	IMAGE  *d_image;

	if (!s_image->hData)                               /* Check Frame Buffer   */
      return(NULL);
   if (reduction<0 || reduction>4)               /* Check bounds         */
		return(NULL);

	d = pow(2,reduction);
	w = s_image->hres / d;
	h = s_image->vres / d;

	// Create new image buffer
	d_image = CreateImage(w,h);

	// Copy image parameters
	d_image->color = s_image->color;
	strcpy(d_image->fspec,s_image->fspec);

	hpDataOut = (BYTE huge *)GlobalLock(d_image->hData);         /* Lock the IMAGE memory   */
	hpDataIn = (BYTE huge *)GlobalLock(s_image->hData);   /* Lock the IMAGE memory   */

	for (y=0; y<d_image->vres; ++y)
      for (x=0; x<d_image->hres; ++x)
			*(hpDataOut+ (DWORD)(d_image->vres-1-y  )* d_image->scansize+x  ) =
			*(hpDataIn + (DWORD)(s_image->vres-1-y*d)*s_image->scansize+x*d);

	GlobalUnlock(s_image->hData);                       // Unlock the IMAGE memory
	GlobalUnlock(d_image->hData);                        // Unlock the IMAGE memory

	// Copy palette data to the new image
	CopyImagePal(s_image,d_image);

	return(d_image);
   }


/*
--------------------------------------------------------------------------
ScaleImageRow()
~~~~~~~~~~~~~~~

This procedure scales an individual scan line of the destination image from
source image. It is implemented to allow for user feedback between each line.
--------------------------------------------------------------------------
*/
void ScaleImageRow(IMAGE *s_image,IMAGE *d_image,int y)
	{
	int  x,x0,y0;
	BYTE  *irow;
	BYTE  *orow;

	irow = (BYTE *)malloc(s_image->hres);          // Allocate input buffer.
	orow = (BYTE *)malloc(d_image->hres);          // Allocate output buffer.

	// Transform the data to the new image
   y0 = (long)(y) * (long)s_image->vres/(long)d_image->vres;
   GetImageRow(s_image,0,y0,irow,s_image->hres);// Read pixel row from image
   for (x=0; x<d_image->hres; ++x)
      {
      x0 = (long)(x) * (long)s_image->hres/(long)d_image->hres;
      orow[x] = irow[x0];
      }
   PutImageRow(d_image,0,y,orow,d_image->hres);   // Write output row

	free(irow);                                   // Free input buffer.
	free(orow);                                   // Free output buffer.
	}




