/*
*  file: sample1.c - match for assembly source files excluding comments
*
*  Copyright Invariant Software 1994
*  All Rights Reserved
*/

#include "externs.h"


void matchfiles_asm0(char huge *f0, char huge *f1, int cse)

/* Pre:  global CmmtStr[0]='X' => ignore all comments ^
*        global CharStr[] == ignore/include space characters ^
*        f0, f1 assembly source files ^
*        cse == ignore case sensitivity
*  Post: (A i,j: f0<=i<end ^ f1<=j<end ^ position(i,j) ^ 
*                                 (!cse ^ *i!=*j ||
*                                  cse ^ tolower(*i)!=tolower(*j)):
*                                                        call mismatch())
*        where postion(p,q) == crit(c,p) ^ crit(c,q) ^
*                              match0(pr(p),p,pr(p),q)
*              where crit(c,m) ==
*                        c=0 ^ !comments(m) ^ !spcchars(m) v
*                        c=1 ^ !comments(m) ^              v
*                    match(p0, p1, q0, q1) ==
*                        (A i,j: p0<=i<p ^ q0<=j<q1 ^
*                                crit(c,i) ^ crit(c,j) ^
*                                numchrs(p0,i)=numchrs(q0,j) ^
*                                numchrs(p0,i)<31: *i=*j)
*                    where numchrs(p, q) = (N i: p<=i<q: crit(c,i))
*
*  Notes: Compare files f0, f1 and display/write mismatches while ignoring
*         comments, including/ignoring space chars, and including/ignoring
*         case differences.
*/

{
  char huge *add_asm0(int c, char huge *m, int *qm0, int *qm1, int *lm);
  void errormsg_(char *s);
  int fixpos_asm0(int c, int cse, char huge *m, char huge *n,
                  int *ms, int *ns, int *lm, int *ln,
                  int *qm0, int *qm1, int *qn0, int *qn1);
  void qts_asm(char huge *m, int *q0, int *q1);
  void mismatch(char huge *f0, char huge *f1, char huge *m, char huge *n,
                int lm, int ln);
  char huge *skipcmtspc_asm0(int c, char huge *m, int *ln);
  void wrtedot(int r, int c, int clr, char ch);

  char huge *m, huge *n;
  int lm, ln, ms, ns, ps, c, qm0, qm1, qn0, qn1;

  if      (CmmtStr[0] == 'X' && CharStr[0] == 'X') c = 0;
  else if (CmmtStr[0] == 'X' && CharStr[0] == ' ') c = 1;
  
  /* loop invariants:
  *
  *    P0 == crit(c,m) ^ crit(c,n)
  *    P1 == ps = position(m,n)
  *    P2 == lm = (# of LINE_FEED encountered thus far in f0..m-1) + 1 ^
  *          ln = (# of LINE_FEED encountered thus far in f0..n-1) + 1
  *    P3 == qm0 == m within '..' quotes ^ qm1 == m within ".." quotes ^
  *          qn0 == n within '..' quotes ^ qn1 == n within ".." quotes
  */

  /* Note:  must keep track of quotes in case ';' (begin comment) is
  *         within a quote and mistaken for the beginning of a comment.
  *         Hence, the variables qm0, qm1, qn0, qn1.
  */

  DotColumn = 0;
  m = f0;
  n = f1;
  ps = 1;
  lm = ln = 1;
  qm0 = qm1 = qn0 = qn1 = 0;           // loop variables initialized
  if (!qm0 && !qm1 && 
      (c == 0 && (*m <= ' ' && *m != '\0' || *m == ';') ||
       c == 1 && (*m == ';'))) {
    m = skipcmtspc_asm0(c, m, &lm);
  }
  if (!qn0 && !qn1 &&
      (c == 0 && (*n <= ' ' && *n != '\0' || *n == ';') ||
       c == 1 && (*n == ';'))) {
    n = skipcmtspc_asm0(c, n, &ln);
  }                                    // P0 - P3 true
  while (ps && *m != '\0' && *n != '\0') {
    if (cse && tolower(*m) == tolower(*n) || !cse && *m == *n) {
      if (*m == LINE_FEED) {
        lm++;
        ln++;
        wrtedot(2, 13, 0, '.');        // write dot to screen
      }                                // maintain lm, ln (invariant P2)

      m = add_asm0(c, m, &qm0, &qm1, &lm);
      n = add_asm0(c, n, &qn0, &qn1, &ln);
      // comments and spaces skipped if necessary
    }
    else {
      mismatch(f0, f1, m, n, lm, ln);  // display/write mismatches
      ps = fixpos_asm0(c, cse, m, n, &ms, &ns, &lm, &ln,
                       &qm0, &qm1, &qn0, &qn1);
      // ps == found postion fix after mismatch
      if (ps) {
        m = m + ms;
        n = n + ns;
      }                                // position fixed after mismatch
    }                                  // P0 - P3 true
  }
  if (!ps) {
    errormsg_("Unable to find reasonable position match after last mismatch.");
  }
}




char huge *add_asm0(int c, char huge *m, int *qm0, int *qm1, int *lm)

/* Pre:  crit(c,m) ^ 
*        *qm0 == m within '..' quotes ^
*        *qm1 == m within ".." quotes ^
*        *lm == # of LINE_FEEDs thus far
*  Post: return m=m+1 such that
*        crit(c,m) ^ 
*        *qm0 == m within '..' quotes ^
*        *qm1 == m within ".." quotes ^
*        *lm == # of LINE_FEEDs thus far
*
*  Notes: Add 1 to m, skip comments, and maintain
*         values of *qm0, *qm1, *lm.
*/

{
  void qts_asm(char huge *m, int *q0, int *q1);
  char huge *skipcmtspc_asm0(int c, char huge *m, int *ln);

  m++;
  if (*m == '\'' || *m == '"') qts_asm(m, qm0, qm1);
  if (!*qm0 && !*qm1 &&
      (c == 0 && (*m <= ' ' && *m != '\0' || *m == ';') ||
       c == 1 && (*m == ';'))) {
    m = skipcmtspc_asm0(c, m, lm);
    if (*m == '\'' || *m == '"') qts_asm(m, qm0, qm1);
  }
  return(m);
}




int fixpos_asm0(int c, int cse, char huge *m, char huge *n,
                int *ms, int *ns, int *lm, int *ln,
                int *qm0, int *qm1, int *qn0, int *qn1)

/* Pre:  crit(c,m) ^ crit(c,n) ^ *m!=*n ^
*        cse == ignore case sensitivity ^
*        lm = (# of LINE_FEEDs thus far in f0..m) + 1 ^
*        ln = (# of LINE_FEEDs thus far in f0..n) + 1 ^
*        *qm0 == m within '..' quotes ^
*        *qm1 == m within ".." quotes ^
*        *qn0 == n within '..' quotes ^
*        *qn1 == n within ".." quotes
*  Post: return fix such that
*        fix == position(m,n) ^
*        *ms = m - m' ^ *ns = n - n' ^
*        *qm0 == m within '..' quotes ^
*        *qm1 == m within ".." quotes ^
*        *qn0 == n within '..' quotes ^
*        *qn1 == n within ".." quotes
*
*  Notes: Fix the position of m and n such that position(m,n) is true and
*         maintain values of *qm0, *qm1, *qn0, *qn1, *lm, and *ln.
*         (see post condition of match_asm0() for "postion(m,n)")
*/

{
  char huge *add_asm0(int c, char huge *m, int *qm0, int *qm1, int *lm);
  int match_asm0(int c, int cse, char huge *p, char huge *q,
                 int qm0, int qm1, int qn0, int qn1);

  char huge *tm, huge *tn;
  int fix, t, x, tqn0, tqn1, nbm, nbn;

  /* loop invariants:
  *
  *    P0 == crit(c,m) ^ crit(c,n)
  *    P1 == nbm = # of chars in tm..m-1 such that crit(c, ) true ^
  *          nbn = # of chars in tn..n-1 such that crit(c, ) true
  *    P2 == *qm0 == m within '..' quotes ^
  *          *qm1 == m within ".." quotes ^
  *          *qn0 == n within '..' quotes ^
  *          *qn1 == n within ".." quotes
  */

  tqn0 = *qn0;
  tqn1 = *qn1;
  t = 0;                               // dummy variable for add_asm0()
  tm = m;
  tn = n;
  nbm = nbn = 0;                       // P0 ^ P1 ^ P2 true
  while (*m != '\0' && nbm < SrchNum &&
         !match_asm0(c, cse, m, n, *qm0, *qm1, *qn0, *qn1)) {
    n = add_asm0(c, n, qn0, qn1, &t);
    nbn++;
    if (*n == '\0' || nbn >= SrchNum) {
      m = add_asm0(c, m, qm0, qm1, &t);
      nbm++;
      n = tn;
      nbn = 0;
      *qn0 = tqn0; *qn1 = tqn1;
    }                                  // P0 - P2 true
  }
  fix = *m != '\0' && nbm < SrchNum;   
  /* fix == found match position
  */

  if (fix) {
    /* m,n represent position fix, now must advance both 'MtchNum'
    *  characters to get to position(m,n)
    */

    x = 0;      
    while (*m !=     '\0' && x != MtchNum) {
      x++;
      m = add_asm0(c, m, qm0, qm1, &t);
    }
  
    x = 0;  
    while (*n != '\0' && x != MtchNum) {
      x++;
      n = add_asm0(c, n, qn0, qn1, &t);
    }
    /* position(m,n) true and *qm0, *qm1, *qn0, *qn1 have correct values
    */
  
    *ms = (int)(m - tm);
    *ns = (int)(n - tn);               // *ms, *ns = as define above
    /* must use *ms, *ns since we cannot pass new address for m,n back
    *  to calling function.
    */
  
    while (tm != m) {
      if (*tm == LINE_FEED) *lm = *lm + 1;
      tm++;
    }
    while (tn != n) {
      if (*tn == LINE_FEED) *ln = *ln + 1;
      tn++;
    }                                  // *lm, *ln = as defined above
  }
  return(fix);
}  




int match_asm0(int c, int cse, char huge *m, char huge *n,
           int qm0, int qm1, int qn0, int qn1)

/* Pre:  crit(c,m) ^ crit(c,n) ^
*        cse == ignore case sensitivity ^
*        qm0 == m within '..' quotes ^
*        qm1 == m within ".." quotes ^
*        qn0 == n within '..' quotes ^
*        qn1 == n within ".." quotes ^
*  Post: return (k==MtchNum) s.t.
*        k = (N i,j: m<=i ^ n<=i ^ crit(c,i) ^ crit(c,j) ^
*                                  numchrs(m,i)=numchrs(n,j) ^
*                                  numchrs(m,i)<31: *i=*j)
*
*  Notes: determine if next 'MtchNum' characters match according to
*         criteria (comments, spaces, case...).  If so, we have a
*         position fix. 
*/

{
  char huge *add_asm0(int c, char huge *m, int *qm0, int *qm1, int *lm);

  int k, t;

  /* loop invariants:
  *
  *    P0 == k = # of character matches within m'..m-1 and n'..n-1
  *              according to criteria (comments, space, case..)
  *    P1 == crit(c,m) ^ crit(c,n)
  *    P2 == qm0 == m within '..' quotes ^ qm1 == m within ".." quotes ^
  *          qn0 == n within '..' quotes ^ qn1 == n within ".." quotes
  */

  t = 0;
  k = 0;                               // P0 - P2 true
  while ((!cse && *m == *n || cse && tolower(*m) == tolower(*n)) &&
          *m != '\0' && *n != '\0' && k != MtchNum) {
    k++;
    m = add_asm0(c, m, &qm0, &qm1, &t);
    n = add_asm0(c, n, &qn0, &qn1, &t);// P0 - P2 true
  }
  // *m='\0' or *n='\0' or k=MtchNum or k!=MtchNum and *n!=*m
  return(k == MtchNum);
}




void qts_asm(char huge *m, int *q0, int *q1)

/* Pre;  only one true at one time: q0, q1
*        *q0 == m-1 within '..' quotes ^
*        *q1 == m-1 within ".." quotes
*  Post:	*q0 == m within '..' quotes ^
*        *q1 == m within ".." quotes
*
*  Notes: Update *q0, *q1 for m
*/

{
  *q0 = !*q1 && (!*q0 && *m == '\'' ||
                 *q0 && *m != '\'');
  *q1 = !*q0 && (!*q1 && *m == '"' ||
                 *q1 && *m != '"');
}




char huge *skipcmtspc_asm0(int c, char huge *m, int *ln)

/* Pre:  m not within quotes ^
*        *m<=' ' v *m=';' ^
*        *ln = # of LINE_FEEDs thus far
*  Post: return m s.t.
*        m = (Min i: m<=i ^ crit(c,i): i)
*        *ln = *ln + (# of LINE_FEEDs in m'..m-1)
*
*  Notes: find first postion of m such that m'<m and m is not
*         in comment and not a space character.
*/

{
  char huge *skipcmtspc_asm0(int c, char huge *m, int *ln);
  void wrtedot(int r, int c, int clr, char ch);

  if (*m <= ' ') {                     // skip space chars
    while (*m <= ' ' && *m != '\0') {
      if (*m == LINE_FEED) {
        wrtedot(2, 13, 0, '.');
        *ln = *ln + 1;
      }
      m++;
    }

    // must check if beginning of comments, skip if true
    if (*m == ';') {
      m = skipcmtspc_asm0(c, m, ln);
    }
  }
  else {                               // skip comments
    while (*(m - 1) != CRGE_RETURN || *m != LINE_FEED) m++;
    m++;
    wrtedot(2, 13, 0, '.');
    *ln = *ln + 1;

    // must check if at space characters or beg. of comments, skip if true
    if (c == 0 && (*m <= ' ' && *m != '\0' || *m == ';') ||
        c == 1 && (*m == ';')) {
      m = skipcmtspc_asm0(c, m, ln);
    }
  }
  return(m);
}
