#ifndef ALERTDRV_H
#define ALERTDRV_H



/*------------------------------------------------------------------------------
   Copyright           : (c)1994 by Logical Operators
                         All Rights Reserved.
   Filename            : AlertDrv.H
   Header File         : AlertDrv.H
   Purpose             : Header file for the AlertDriver classes. An
                         AlertDriver object is used as a device driver by other
                         objects. The controlling object can then alert the
                         user of errors or program conditions by communicating
                         through a descendant of an AlertDriver object,
                         resulting in the device-independence of the
                         controlling object.
   Compiler Directives : None

   Modification History:
   Version   Date    Programmer and Description of Changes
   ------- --------  --------------------------------------------------------
    1.00   01/18/94  Original version by Warren J. Hairston.
   ---------------------------------------------------------------------------*/



   //included files
   //--------------
   #include <ENVIRON.H>   //contains platform-specific definitions

   #include <fstream.h>   //C++ file stream declarations
   #include <string.h>     //strcpy() function prototype



   /* constants - Used to set the processing flags for an AlertDriver. The adf
                  prefix denotes an AlertDriver Flag code. */
   //--------------------------------------------------------------------------
   const unsigned short adfERROR      = 0x01,   //process errors
                        adfINFO       = 0x02,   //process info text
                        adfMESSAGE    = 0x04,   //process messages
                        adfWARNING    = 0x08,   //process warnings
                        adfALLALERTS  = 0x0F,   //process all alerts
                        adfTIMESTAMP  = 0x10,   //time-stamp text
                        adfDISCLOSURE = 0x20;   /*return cause, not necessarily
                                                  return value*/



   /* constants - These return codes represent user selections in some
                  AlertDriver methods. The ads prefix denotes an AlertDriver
                  Selection code. */
   //--------------------------------------------------------------------------
   const unsigned short adsCANCEL   = 0,   //CANCEL choice
                        adsDISABLED = 1,   //processing flag disabled
                        adsNO       = 2,   //NO choice
                        adsNODRIVER = 3,   //no AlertDriver exists
                        adsYES      = 4;   //YES choice



   /* constants - Options to indicate how AlertDriver::ChangeProcFlags() affects
                  the flags for each AlertDriver in the chain. The cfo prefix
                  denotes a Change Flag Option code. */
   //--------------------------------------------------------------------------
   const unsigned short cfoCHAIN     = 0x00,   /*specified flags should alter
                                                 the processing flags of all
                                                 AlertDrivers in the chain,
                                                 starting with the AlertDriver
                                                 to which the message was sent*/
                        cfoDRIVER    = 0x01,   /*specified flags should alter
                                                 only the processing flags of
                                                 the AlertDriver to which the
                                                 message was originally sent*/
                        cfoOVERWRITE = 0x00,   /*copy specified flags to
                                                AlertDriver, possibly altering
                                                other proessing flags*/
                        cfoENABLE    = 0x02,   /*enable only specified flags
                                                 without altering other
                                                 processing flags*/
                        cfoDISABLE   = 0x04;   /*disable only specified flags
                                                 without altering other
                                                 processing flags*/



   /* constants - Options to indicate how RecordingAlertDriver objects should
                  control their output device resources. The rad prefix indicates
                  a RecordingAlertDriver code. */
   //--------------------------------------------------------------------------
   const short radHOLDRESOURCE = 0x00,   /*open resource during construction and
                                           hold open until destruction*/
               radFREERESOURCE = 0x01,   /*open resource just before output and
                                           close resource after each output*/
               radASISRESOURCE = 0x02;   /*use the resource as-is:  do not open
                                           or close the resource*/



   //class declarations
   //------------------
   class AlertDriver;   /*defined below, declared here for reference in class
                          AlertDriverLink*/



   class AlertDriverLink
   #ifdef ROOTCLASS
      : public ROOTCLASS
   #endif   //ROOTCLASS
   {
      public:
         //constructor(s)
         //--------------
         AlertDriverLink(void)
         #ifdef ROOTCLASS
            : ROOTCLASS()
         #endif   //ROOTCLASS
            {linkedAlertDriver = NULL;}

         //destructor
         //----------
         virtual ~AlertDriverLink() {FreeLinkedAlertDriver();}

         //member functions
         //----------------
         void         ChangeLinkedAlertDriver(AlertDriver *anAlertDriver);
         AlertDriver* GetLinkedAlertDriver(void) const
                         {return linkedAlertDriver;}

      protected:
         //data members
         //------------
         AlertDriver *linkedAlertDriver;   /*Points to an AlertDriver which is
                                             linked to this object. AlertDrivers
                                             may be daisy-chained together to
                                             allow multiple types of alerting.*/

         //member functions
         //----------------
         virtual void FreeLinkedAlertDriver(void);
   };   //class AlertDriverLink



   class AlertDriver : public AlertDriverLink
   {
      public:
         //constructor(s)
         //--------------
         AlertDriver(void);

         //destructor
         //----------
         virtual ~AlertDriver() {referenceCount = 0;}

         //member functions
         //----------------
                 unsigned long  Attach(void) {return ++referenceCount;}
         virtual unsigned short ChangeProcFlags(const unsigned short newFlags,
                                                const unsigned short options =
                                                   cfoCHAIN | cfoOVERWRITE);
                 unsigned long  Detach(void) {return --referenceCount;}
                 unsigned short GetProcFlags(void) const
                                   {return processingFlags;}
         virtual void           GetTimestamp(char *timestamp) const;
                 void           HandleError(const char *text);
                 void           HandleInfo(const char *text);
                 void           HandleMessage(const char *text);
                 unsigned short HandleWarning(const char *text,
                                              const unsigned short
                                                 defaultChoice);
                 int            IsAttached(void) const
                                   {return (referenceCount > 0);}
                 unsigned long  NumAttached(void) const {return referenceCount;}

         //platform-specific member functions
         //----------------------------------
         #ifdef BORLAND_CONTAIN
            virtual hashValueType hashValue() const {return 0;}
            virtual classType     isA() const {return 0;}
            virtual int           isEqual (const Object& testObject) const
                                     {return 0;}
            virtual char*         nameOf() const {return "AlertDriver";}
            virtual void          printOn(ostream& outputStream) const {}
         #endif   //BORLAND_CONTAIN

      protected:
         //member functions
         //----------------
         virtual void           ReportError(const char *text) = 0;
         virtual void           ReportInfo(const char *text) = 0;
         virtual void           ReportMessage(const char *text) = 0;
         virtual unsigned short ReportWarning(const char *text,
                                              const unsigned short
                                                 defaultChoice) = 0;

      private:
         //data members
         //------------
         unsigned short processingFlags;   /*flags which control how this
                                             AlertDriver responds during the
                                             processing of member functions*/
         unsigned long  referenceCount;    /*number of objects which are
                                             currently referencing this
                                             AlertDriver*/
   };   //class AlertDriver



   class RecordingAlertDriver : public AlertDriver
   {
      public:
         //constructor(s)
         //--------------
         RecordingAlertDriver(const short resourceControlMode) :
            AlertDriver() {resourceControl = resourceControlMode;}

         //member functions
         //----------------
         short GetResourceControl(void) const {return resourceControl;}

         //platform-specific member functions
         //----------------------------------
         #ifdef BORLAND_CONTAIN
            virtual char* nameOf() const {return "RecordingAlertDriver";}
         #endif   //BORLAND_CONTAIN

      private:
         //data members
         //------------
         short resourceControl;   /*determines if this object should hold the
                                    resource or release it for use by other
                                    objects or programs*/
   };   //class RecordingAlertDriver



   class StdAlertDriver : public AlertDriver
   {
      public:
         //platform-specific member functions
         //----------------------------------
         #ifdef BORLAND_CONTAIN
            virtual char* nameOf() const {return "StdAlertDriver";}
         #endif   //BORLAND_CONTAIN

      protected:
         //member functions
         //----------------
         virtual void           ReportError(const char *text);
         virtual void           ReportInfo(const char *text);
         virtual void           ReportMessage(const char *text);
         virtual unsigned short ReportWarning(const char *text,
                                              const unsigned short
                                                 defaultChoice);
   };   //class StdAlertDriver



   class StreamAlertDriver : public RecordingAlertDriver
   {
      public:
         //constructor(s)
         //--------------
         StreamAlertDriver(ostream &outputStream,
                           const short resourceControlMode = radASISRESOURCE) :
            RecordingAlertDriver(resourceControlMode), outStream(outputStream){}

         //member functions
         //----------------
         ostream& GetStream(void) const {return outStream;}

         //platform-specific member functions
         //----------------------------------
         #ifdef BORLAND_CONTAIN
            virtual char* nameOf() const {return "StreamAlertDriver";}
         #endif   //BORLAND_CONTAIN

      protected:
         //data members
         //------------
         ostream &outStream;   //references the stream to be used for output

         //member functions
         //----------------
         virtual void           ReportError(const char *text);
         virtual void           ReportInfo(const char *text);
         virtual void           ReportMessage(const char *text);
         virtual unsigned short ReportWarning(const char *text,
                                              const unsigned short
                                                 defaultChoice);
   };   //class StreamAlertDriver



   class TextFileAlertDriver : public StreamAlertDriver
   {
      public:
         //constructor(s)
         //--------------
         TextFileAlertDriver(const char *filename,
                             const short resourceControlMode);

         //destructor
         //----------
         virtual ~TextFileAlertDriver();

         //member functions
         //----------------
         void GetFilename(char *filename) const {strcpy(filename, adFilename);}

         //platform-specific member functions
         //----------------------------------
         #ifdef BORLAND_CONTAIN
            virtual char* nameOf() const {return "TextFileAlertDriver";}
         #endif   //BORLAND_CONTAIN

      protected:
         //member functions
         //----------------
         virtual void           ReportError(const char *text);
         virtual void           ReportInfo(const char *text);
         virtual void           ReportMessage(const char *text);
         virtual unsigned short ReportWarning(const char *text,
                                              const unsigned short
                                                 defaultChoice);

      private:
         //data members
         //------------
         char *adFilename;   //name of file to which alerts will be reported
         ofstream adFile;   //stream used to record alerts 

         //member functions
         //----------------
         void SetFilename(const char *filename);
   };   //class TextFileAlertDriver



#if defined BORLAND_TV
   class TurboVisionAlertDriver : public AlertDriver
   {
      public:
         //constructor(s)
         //--------------
         TurboVisionAlertDriver(void);

         //destructor
         //----------
         virtual ~TurboVisionAlertDriver() {FreeMsgDialog();}

      protected:
         //data members
         //------------
         TDialog     *adMsgDialog;   /*points to the dialog box used for
                                       messages*/
         TStaticText *adMsgText;     //points to text used for messages

         //member functions
         //----------------
         virtual void           FreeMsgDialog(void);
         virtual void           FreeMsgText(void);
         virtual TDialog*       InitMsgDialog(void);
         virtual TStaticText*   InitMsgText(const TRect& bounds,
                                            const char *text)
                                   {return new TStaticText(bounds, text);}
         virtual void           ReportError(const char *text);
         virtual void           ReportInfo(const char *text);
         virtual void           ReportMessage(const char *text);
         virtual unsigned short ReportWarning(const char *text,
                                              const unsigned short
                                                 defaultChoice);
   };   //class TurboVisionAlertDriver
#endif   //BORLAND_TV



#ifdef TARGET_WINDOWS
   class WindowsAlertDriver : public AlertDriver
   {
      public:
         //constructor(s)
         //--------------
         WindowsAlertDriver(const HWND hParentWnd = NULL);

         //destructor
         //----------
         ~WindowsAlertDriver() {FreeMsgWindow();}

         //member functions
         //----------------
         HWND GetParent(void) {return hWndParent;}
         void SetParent(HWND hParentWnd) {hWndParent = hParentWnd;}
         
         //platform-specific member functions
         //----------------------------------
         #ifdef BORLAND_CONTAIN
            virtual char* nameOf() const {return "WindowsAlertDriver";}
         #endif   //BORLAND_CONTAIN

      protected:
         //data members
         //------------
         HWND hWndMsgText,     //handle to the message text
              hWndMsgWindow,   //handle to the message window
              hWndParent;      /*handle to the parent window of windows created
                                 by this object*/

         //member functions
         //----------------
         virtual void           FreeMsgWindow(void);
         virtual HWND           InitMsgWindow(void)
                                   {return WindowsInitMessage(hWndParent);}
         virtual void           ReportError(const char *text);
         virtual void           ReportInfo(const char *text);
         virtual void           ReportMessage(const char *text);
         virtual unsigned short ReportWarning(const char *text,
                                              const unsigned short
                                                 defaultChoice);
   };   //class WindowsAlertDriver



   #ifdef BORLAND_OWL
      class BWCCAlertDriver : public WindowsAlertDriver
      {
         public:
            //platform-specific member functions
            //----------------------------------
            #ifdef BORLAND_CONTAIN
               virtual char* nameOf() const {return "BWCCAlertDriver";}
            #endif   //BORLAND_CONTAIN

         protected:
            //member functions
            //----------------
            virtual void           ReportError(const char *text);
            virtual void           ReportInfo(const char *text);
            virtual unsigned short ReportWarning(const char *text,
                                                 const unsigned short
                                                    defaultChoice);
      };   //class BWCCAlertDriver
   #endif   //BORLAND_OWL



#endif   //TARGET_WINDOWS



   //variable(s)
   //-----------
   extern AlertDriver *defaultAlertDriver;   //the default AlertDriver



#endif   //!ALERTDRV_H
