//****************************************************************************
// File:        MOVELST.C        
//          
//
// Purpose:     The main module.  Brings up a private dialog, that contains
//              a list box with thirty items in it.  The user selectes an item, 
//              and then clicks and drags the item.  As the mouse moves up and
//              down the list, a line is PatBlt'd on the list to indicate its
//              new position.  When the mouse button is released, the item is
//              is moved.  If the mouse button is moved above or below the list,
//              the listbox will be scrolled (if appropriate).
//
// Functions:
//            WinMain()       -  initializes app and processes message loop 
//            ActualDlgProc   -  the actual Dialog Procedure
//            AboutDlgProc    -  Dialog procedure for the About box
//            ClassDlgProc    -  the Window Proc for the Private Dialog
//            NewListProc     -  the subclassed listbox window procedure 
//
// Development Team:                   
//
//                  JDL, July 1993
//
//
// Written by Microsoft Product Support Services, Windows Developer Support
// Copyright (c) 1992 Microsoft Corporation. All rights reserved.
//****************************************************************************

#include "windows.h"
#include "resource.h"
#include "stdlib.h"
#include "memory.h"

#define ID_TIMER 100
#define LINE_WIDTH 3


// style flags for the DrawIndicator() function
#define DI_TOPERASED        0x0001  // erasing a line drawn on the top of the list
#define DI_BOTTOMERASED     0x0002  // erasing a line drawn on the bottom of the list
#define DI_ERASEICON        0x0004  // erasing the icon

UINT idTimer;            // the id for the timer used in scrolling the list
HFONT hFont;             // a new font for the list box
HCURSOR hCurDrag;        // a cursor to indicate dragging
HINSTANCE hInst;         // the instance handle       
int nHtItem;             // the height of an individual item in the list box
BOOL bNoIntegralHeight;  // does the list box have the LBS_NOINTEGRALHEIGHT style flag 

HWND ghDlg;              // handle to the main window
HWND ghList;             // handle to the list box     
HBRUSH ghBrush;          // handle to the brush with the color of the windows background

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpCmdLine, int nCmdShow);
BOOL FAR PASCAL __export ActualDlgProc (HWND hDlg, UINT message,  WPARAM wParam, LPARAM lParam);
BOOL FAR PASCAL __export AboutDlgProc (HWND hDlg, UINT message,  WPARAM wParam, LPARAM lParam);
long FAR PASCAL __export ClassDlgProc(HWND hDlg, UINT message, WPARAM wParam , LPARAM lParam);
long FAR PASCAL __export NewListProc(HWND hwndList, UINT message, WPARAM wParam , LPARAM lParam); 
void DrawIndicator(HDC hDC, int nYpos, int nWidth, WORD wFlags);


FARPROC lpfnOldListProc, LstProc;      

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

****************************************************************************/
int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, 
                   LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;
    WNDCLASS wc;
    HWND hwnd;
    DLGPROC dlgProc;
    ATOM aRegReturn;  
    char szErr[128];
    hInst = hInstance;
             
    if (!hPrevInstance)
    {
        memset(&wc, NULL, sizeof(WNDCLASS));
        wc.style = 0;
        wc.lpfnWndProc = ClassDlgProc;
        wc.cbWndExtra = DLGWINDOWEXTRA;
        wc.hInstance = hInstance;
        wc.hIcon = LoadIcon(hInstance,MAKEINTRESOURCE(IDI_MAIN));
        wc.hCursor = LoadCursor(NULL, IDC_ARROW);
        wc.hbrBackground = COLOR_WINDOW + 1;
        wc.lpszMenuName = NULL;
        wc.lpszClassName = "PrivDlgClass";    
        aRegReturn = RegisterClass(&wc);        
        if (aRegReturn == NULL)
        {
            LoadString(hInst, IDS_ERRREGISTER, szErr, 128);
            MessageBox(NULL, szErr, "Error", MB_OK|MB_ICONSTOP);
            return 0;        
        
        }
    }               
    dlgProc = (DLGPROC)MakeProcInstance(ActualDlgProc, hInstance); 
    ghDlg = hwnd = CreateDialog(hInstance, MAKEINTRESOURCE(IDD_DIALOG), NULL, dlgProc);   
    if (!hwnd)
    {
        LoadString(hInst, IDS_ERRCREATE, szErr, 128);
        MessageBox(NULL, szErr, "Error", MB_OK|MB_ICONSTOP);
        return 0;    
    
    }                         
    
    ShowWindow(hwnd,nCmdShow);    
    while (GetMessage(&msg,NULL,0,0))
    {
        if (!IsDialogMessage(hwnd, &msg))
        {
            TranslateMessage(&msg);
            DispatchMessage(&msg);    
        }
    
    }              
                                                           
    FreeProcInstance((FARPROC)dlgProc); 
    
    return msg.wParam;                            
    
    
} 

/****************************************************************************

    ClassDlgProc(HWND hDlg, UINT message, WPARAM wParam , LPARAM lParam)

    PURPOSE: 
     this function gets placed between the dialog and the DefDlgProc because
      
        1. its a private dialog 
     and
        2. we specified a DLGPROC for the  third parameter 
           of the CreateDialog() call.
           
    we could handle all of the messages here (except for the WM_INITDIALOG 
    message which is not sent to non-dialogs) or we can pass the messages 
    off to DefDlgProc(), which will then call our dialog procedure 
    ActualDlgProc(), given below    

****************************************************************************/
long FAR PASCAL __export ClassDlgProc(HWND hDlg, UINT message, WPARAM wParam , LPARAM lParam)
{
  
    return DefDlgProc(hDlg, message, wParam, lParam);
    
}      


/****************************************************************************

    ActualDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)

    PURPOSE: 
    The procedure for the application that does most of the work.
    
    This is the function that we passed in as the last parameter to the
    CreateDialog() call.  We do this so that we can get the WM_INITDIALOG
    message, which is not passed to the WndProc of a Private Dialog.
    
    The listbox is subclassed so that we can trap the mouse messages

****************************************************************************/
       
BOOL FAR PASCAL __export ActualDlgProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{

    switch (message)
    {    
        case WM_INITDIALOG:
        {   
            int i;   
            char szBuf[128]; 
            int nRet;  
            LOGFONT lf;
            HMENU hSysMenu;  // handle to the system menu
            HDC hdc;         // a dc to find out the number of pixels per logcal inch            
            LOGBRUSH lb;
                
            lb.lbStyle = BS_SOLID;
            lb.lbColor = GetSysColor(COLOR_WINDOW);
            lb.lbHatch = NULL;
                
            ghBrush = CreateBrushIndirect(&lb);
            
            hSysMenu = GetSystemMenu(hDlg, FALSE);                            
            // disable the "maximize" option in the system menu
            EnableMenuItem(hSysMenu, 4, MF_GRAYED|MF_DISABLED|MF_BYPOSITION); 
            // disable the "size" option of the system menu                   
            EnableMenuItem(hSysMenu, 2, MF_GRAYED|MF_DISABLED|MF_BYPOSITION); 
    
            for (i=0; i<30; i++)
            {
                wsprintf((LPSTR)szBuf,"Line %d", i+1);                    
                nRet = (int)SendDlgItemMessage(hDlg,
                                               ID_SOURCE, 
                                               LB_ADDSTRING,
                                               0,
                                               (LPARAM)(LPSTR)szBuf);            
                
             }
                
            ghList = GetDlgItem(hDlg, ID_SOURCE);  
            LstProc = MakeProcInstance((FARPROC)NewListProc,hInst);
            lpfnOldListProc = (FARPROC)SetWindowLong(ghList,
                                                     GWL_WNDPROC, 
                                                     (LONG)LstProc);
                                                     
            // check to see if it has integral height
            bNoIntegralHeight = (BOOL)(GetWindowLong(ghList, GWL_STYLE) && LBS_NOINTEGRALHEIGHT);
            hdc = GetDC(hDlg);        
            memset(&lf, NULL, sizeof(lf));        
            lf.lfHeight = -MulDiv(14, GetDeviceCaps(hdc, LOGPIXELSY), 72);
            lstrcpy(lf.lfFaceName, "MS Sans Serif");
            hFont = CreateFontIndirect(&lf);
            ReleaseDC(hDlg, hdc);            
            SendMessage(ghList, WM_SETFONT, (WPARAM)hFont, (LPARAM)FALSE);
            // the drag cursor
            hCurDrag = LoadCursor(hInst, MAKEINTRESOURCE(IDC_DRAG));       
            
            return FALSE;   // didn't set the focus
        }                                                
        break;
        case WM_COMMAND:
            switch (wParam)
            {
                case IDCANCEL:            
                  DestroyWindow(hDlg);            
                  break; 
                case ID_ABOUT:
                {
                    DLGPROC dlgprc;
                    dlgprc = (DLGPROC) MakeProcInstance(AboutDlgProc, hInst);
                    DialogBox(hInst, MAKEINTRESOURCE(IDD_ABOUT), hDlg, dlgprc);
                    FreeProcInstance((FARPROC) dlgprc);                 
                    break;
                }                                
                break;                               
                default:
                    return FALSE;
                    break; 
            }                   
            break;        
        case WM_DESTROY: // clean up
        {
            DeleteObject(ghBrush);
            DeleteObject(hFont);
            DestroyCursor(hCurDrag);                         
            PostQuitMessage(0);            
        }
        break;                           
     default:
        return FALSE; // we didn't do anyting        
    } // end switch message
    return TRUE; // we did the processing

}                           
/***********************************************************************************

    FUNCTION: 
             NewListProc(HWND hwndList, UINT message, WPARAM wParam , LPARAM lParam)

    PURPOSE: 
             The subclassed listbox procedure.  The following messages are
             handled
             
             
             Message            Purpose
             
             WM_LBUTTONDOWN     capture the mouse, go into a PeekMessage() 
                                loop until a WM_LBUTTONUP message is received
             
             WM_MOUSEMOVE       check to see what item in the list box is under 
                                the mouse, and draw a line to indicate where
                                the item would be moved if the user released 
                                the button
                            
             WM_TIMER           check to see if the mouse is above or below the
                                list box, and if appropriate, force the list box
                                to scroll up or down.  redraw the lines as needed.
                            
             WM_LBUTTONUP       if the user moved an item, set a flag to indicate
                                success, and store the new position of the item                                                        
                            
             WM_CANCELMODE      if we were dragging the item cancel the drag
                                (see comment below)
                  
             
        
***********************************************************************************/                                              

long FAR PASCAL __export NewListProc(HWND hwndList, 
                                     UINT message, 
                                     WPARAM wParam, 
                                     LPARAM lParam)
{
    
    static BOOL bTracking = FALSE;
    static BOOL bDrag = FALSE;  
    static HCURSOR hCursorOld = NULL;
    

    
    switch (message)
    {    
      case WM_CANCELMODE:
       // WM_CANCELMODE is sent to the window that has captured the mouse before
       // a message box or modal dialog is displayed. If we were dragging the item
       // cancel the drag.
       bTracking = FALSE;
       ReleaseCapture();
       if (bDrag)
          SetCursor(hCursorOld);    
       break; 
      case WM_LBUTTONDOWN:
      {
        
        // Was the list box item dragged into the destination?        
        BOOL bDragSuccess = FALSE;  
        MSG msg;
        POINT pt;      
        POINT point;
        
        RECT rectIsDrag;            // Rectangle to determine if dragging has started.  
        int nOldPos;
        
        int nOldY = -1;                            // the last place that we drew on
        HDC hdc;   // dc to draw on  
        div_t divt;                            // get remainder a quotient with "div"   
        int nCount;
        div_t divVis;          
// space for scroll bar -  starts off at 1 so we don't overwrite the border
        int dxScroll = 1;      
        RECT rect;   
        int nVisible;                   // the number of items visible
        int idTimer;                    // id for the timer
        int nNewPos;                    // the new position
        int nTopIndex;                  // the top index        
        
        
        
         
         GetWindowRect(hwndList, &rect);        
           
         // Pass the WM_LBUTTONDOWN to the list box window procedure. Then
         // fake a WM_LBUTTONUP so that we can track the drag.
         CallWindowProc(lpfnOldListProc, hwndList, message, wParam, lParam);
         
         // the number of items in the list box
         nCount = (int)SendMessage(hwndList, LB_GETCOUNT,0,0L);         
         if (nCount == 0 ) // don't do anything to and empty list box
            return NULL;         
        // fake the WM_LBUTTONUP            
         CallWindowProc(lpfnOldListProc, hwndList, WM_LBUTTONUP, wParam, lParam);        
         // get a dc to draw on
         hdc = GetDC(hwndList);                               
         
         // the height of each item   
         nHtItem = (int)SendMessage(hwndList, LB_GETITEMHEIGHT,0,0L);          
         // the current item
         nOldPos = (int)SendMessage(hwndList, LB_GETCURSEL,0,0L);    
         
         divVis = div((rect.bottom - rect.top), nHtItem);
// the number of visible items                  
         nVisible = divVis.quot;
// some items are invisible - there must be scroll bars - we don't want
// to draw on them         
         if (nVisible < nCount)                                         
            dxScroll = GetSystemMetrics(SM_CXVSCROLL) + 1; 
            
         idTimer = NULL;
         idTimer = SetTimer(hwndList, ID_TIMER,100,NULL);  
        
              
     // Create a tiny rectangle to determine if item was dragged or merely clicked on.
     // If the user moves outside this rectangle we assume that the dragging has
     // started.
         point = MAKEPOINT(lParam);        
         SetRect(&rectIsDrag, point.x, point.y - nHtItem / 2,
                               point.x, point.y + nHtItem / 2); 
                               
                                          
         bTracking = TRUE;         
         SetCapture(hwndList);
         
         
         // Drag loop                      
         while (bTracking)
         {  
        // Retrieve mouse, keyboard, and timer messages. We retrieve keyboard
        // messages so that the system queue is not filled by keyboard messages
        // during the drag (This can happen if the user madly types while dragging!)
        // If none of these messages are available we wait. Both PeekMessage() 
        // and Waitmessage() will yield to other apps.   
                                      
            while (!PeekMessage(&msg, NULL, WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE)
                   && !PeekMessage(&msg, NULL, WM_KEYFIRST, WM_KEYLAST, PM_REMOVE)
                   && !PeekMessage(&msg, NULL, WM_TIMER, WM_TIMER, PM_REMOVE)) 
               WaitMessage();
           switch(msg.message)
            {
               case WM_MOUSEMOVE:
               {
                  pt = MAKEPOINT(msg.lParam);
                  if (!bDrag)
                  {
                     // Check if the user has moved out of the Drag rect. 
                     // in the vertical direction.  This indicates that 
                     // the drag has started.
                     if ( (pt.y > rectIsDrag.bottom) || 
                          (pt.y < rectIsDrag.top)) // !PtInRect(&rectIsDrag,pt))
                     {
                        hCursorOld = SetCursor(hCurDrag);      
                        bDrag = TRUE;     // Drag has started                           
                        
                     }
                  }
                          
                          
                  if (bDrag)
                  {  
  
                    SetCursor(hCurDrag);  
   // if we are above or below the list box, then we are scrolling it, and
   // we shouldn't be drawing here              
                    ClientToScreen(hwndList, &pt);
                    if ((pt.y >= rect.top) && (pt.y <= rect.bottom))
                    {
                        // convert the point back to client coordinates
                        ScreenToClient(hwndList, &pt);
                        divt = div(pt.y,nHtItem);                        
                                
                        // if we are half way to the item
                        // AND it is a new item
                        // AND we are not past the end of the list..                        
                        if ( divt.rem < nHtItem / 2 && 
                             (nOldY != nHtItem * divt.quot) && 
                             (divt.quot < nCount + 1)) 
                        {
                              
                           if (nOldY != -1)
                            {
                                // erase the old one                                
                                DrawIndicator(hdc, nOldY,(rect.right - rect.left) - dxScroll, DI_ERASEICON);
                            }  
                                    
                            nOldY = nHtItem * divt.quot;                            
                            DrawIndicator(hdc, nOldY,(rect.right - rect.left) - dxScroll, 0);                                                        
                                    
                        }
                     } // end if in the list box window                        
                            
                  } // end if bDrag
                            
               }
              break;                   
              case WM_TIMER:
               {
                  POINT pt;                  
                  GetCursorPos(&pt); 
                  nTopIndex = (int)SendMessage(hwndList, LB_GETTOPINDEX,0,0L);;                                      
                  if (pt.y < rect.top) // scroll up
                  {
                           
                       if (nTopIndex > 0)
                       {
                                
                            nTopIndex--;
                            SendMessage(hwndList, LB_SETTOPINDEX, nTopIndex,0L);
                         // when you scroll up, the line always stays on the top index                            
                         // erase the one we've moved down
                            DrawIndicator(hdc, nHtItem,(rect.right - rect.left) - dxScroll, DI_TOPERASED|DI_ERASEICON);
                         // draw the new one          
                            DrawIndicator(hdc, 0,(rect.right - rect.left) - dxScroll, 0);                                                             
                         // the new one was drawn at y = 0 
                           nOldY = 0;                           
                           
                       }                  
                      
                  }
                  else if (pt.y > rect.bottom) // scroll down
                  {                       
                       // if the number of visible items (ie seen in the list box)
                       // plus the number above the list is less than the total number
                       // of items, then we need to scroll down
                        if (nVisible + nTopIndex < nCount)
                        {                                
                            
                            if (nOldY - nTopIndex != nVisible)
                            {
                        // if them move below the list REALLY REALLY FAST, then
                        // the last line will not be on the bottom - so we want to reset the last
                        // line to be the bottom                            
                                
                                // erase the old line
                                DrawIndicator(hdc, nOldY,(rect.right - rect.left) - dxScroll, DI_ERASEICON);                                       
                                // reset the index
                                divt.quot = nVisible;
                                nOldY = divt.quot * nHtItem;                            
                                // draw the new line
                                DrawIndicator(hdc, nOldY,(rect.right - rect.left) - dxScroll, 0);                                       
                                
                                
                            }
                        // scroll up
                            nTopIndex++;
                            SendMessage(hwndList, LB_SETTOPINDEX, nTopIndex,0L);
                        
                       // erase the line that has moved up.. 
                            DrawIndicator(hdc, nOldY - nHtItem,(rect.right - rect.left) - dxScroll, DI_BOTTOMERASED|DI_ERASEICON);
                        // draw the new one
                            DrawIndicator(hdc, nOldY,(rect.right - rect.left) - dxScroll, 0);
                           
                        }
                      
                  }               
               }
               break;
               case WM_LBUTTONUP: 
                  // End of Drag                             
                        
                  nTopIndex = (int)SendMessage(hwndList, LB_GETTOPINDEX, 0, 0L);                  
                  if (bDrag) 
                  {                        
                    // get rid of any line we've drawn - the position of the line 
                    // divided by the height of the itme is where our new index
                    // is going to be                    
                    DrawIndicator(hdc, nOldY,(rect.right - rect.left) - dxScroll, DI_ERASEICON);
                    
                    nNewPos = (nOldY / nHtItem) + nTopIndex;                     
                    // the old position can't equal the new one                                        
                    if (nNewPos != nOldPos)
                        bDragSuccess = TRUE;
                  }
                  bTracking = FALSE;                  
                  break;                     
               default:
                  // Process the keyboard messages
                 TranslateMessage(&msg);
                 DispatchMessage(&msg);
                break;      
          }          
       }// end while bTracking
        
         ReleaseCapture();
         if (bDrag)
         {
                SetCursor(hCursorOld);
                // move the item
                if (bDragSuccess) 
                {
                    int nIndex;       
                    char s[256];  
                    
                    
                // we need to store the top index, because deleting and adding a new
                // string will change it, and we want to be able to see the item that 
                // we have moved
                    nTopIndex = (int)SendMessage(hwndList, LB_GETTOPINDEX,0,0L);                    
                    // stop most of the blinking..
                    SendMessage(hwndList, WM_SETREDRAW, FALSE,0L);
                    // get the text of the item - limited to 256 chars!
                    SendMessage(hwndList, LB_GETTEXT, nOldPos, (LPARAM)(LPSTR)s); 
                    
/*------------------------------------------------------------------------
 | strategy:  given ABCD and moving to BCAD do the following:
 |
 |           1. delete A -- giving BCD
 |           2. insert A -- giving BCAD
 |           3. hilite A
 |           4. set the top index so A is visible
 -------------------------------------------------------------------------*/                                    
                    // delete the original string
                    SendMessage(hwndList, LB_DELETESTRING, nOldPos, 0L);
                    
// if we've moved DOWN the list subtract one from the new index 
// (because we've deleted a string but if we are moving UP the list, 
// we don't subtract anything (the deleted item is below the new item, 
// so our new index hasn't changed
                     
                    if (nNewPos > nOldPos)
                        nNewPos--;                                   
                    // put it in the new pos       
                     nIndex = (int)SendMessage(hwndList,
                                               LB_INSERTSTRING, 
                                               nNewPos,
                                               (LPARAM)(LPSTR)s);  
                    
                    SendMessage(hwndList, LB_SETCURSEL, nIndex, 0L);                            
                    SendMessage(hwndList, LB_SETTOPINDEX, nTopIndex,0L);                    
                    SendMessage(hwndList, WM_SETREDRAW, TRUE,0L);                 
                            
                } // end if bDragSuccess                  
          } // end if bDrag
      bDrag = FALSE;    
      ReleaseDC(hwndList, hdc);
      KillTimer(hwndList, idTimer);    
    } 
    break;
    default:
      return  CallWindowProc(lpfnOldListProc, hwndList, message, wParam, lParam);
   }
   return NULL;
}     


/****************************************************************************

    FUNCTION: AboutDlgProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "About" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

****************************************************************************/
 
BOOL FAR PASCAL __export AboutDlgProc (HWND hDlg, UINT message,  
                                       WPARAM wParam, LPARAM lParam)
{

    switch (message)
    {
        case WM_INITDIALOG:
            return (TRUE);
        case WM_COMMAND:
            if (wParam == IDOK)
            {
                EndDialog(hDlg, TRUE);
                return (TRUE);
            }
            break;
    }
    return (FALSE);

}
/****************************************************************************

    FUNCTION: DrawIndicator(HDC hDC, int nYpos, int nWidth, WORD wFlags)

    PURPOSE:  gives the user a visual indication where the item will be 
              dropped.  Two methods are shown.  They are:
 
 
                     ______
                    |      |
               Icon | Line |
                --> ||----||
                    |      |
                    |______|
                    
               where the "-->"  is an arrow icon indicating the position, 
               and the "|---|" is a line drawn betwenn the items in the 
               list indicating the new position.  To correctly draw the 
               line in the border situations, the DC is clipped to the 
               client area of the list                    
              

    PARAMETERS: 
    
                hDC         a device context to draw on
                nYpos       the vertical postion
                wFlags      style bits used to indicate if something is 
                            being erased, drawn on a border, or any other
                            style that may be applicable.

    

****************************************************************************/
void DrawIndicator(HDC hDC, int nYpos, int nWidth, WORD wFlags)
{      

// draw a horizontal line     
    int nTop, nHeight;   
    HICON hIcon;   
    HRGN hClipRgn;                 // the clipping region    
    RECT rect;    

// we don't want the clip anything when we are drawing
// the icon outside the list box
    SelectClipRgn(hDC, NULL);    
   if (wFlags & DI_ERASEICON)
   {


      
      rect.left = -33;
      rect.right = -1;
      rect.top = nYpos -16;
      rect.bottom = nYpos + 16;   
      // ghBrush is created in WM_INITDIALOG   
      FillRect(hDC, &rect, ghBrush);
        
   }
   else
   {
        
       hIcon = LoadIcon(hInst, MAKEINTRESOURCE(IDI_ARROW)); 
       if (hIcon)
       {
           DrawIcon(hDC,-33,nYpos - 16,hIcon);
           DestroyIcon(hIcon);
        }
   
   
   }
   
    
// create a clipping region for drawing the lines in the list box
     GetWindowRect(ghList, &rect);         
     hClipRgn = CreateRectRgn(0,0, rect.right - rect.left, rect.bottom - rect.top);
     SelectClipRgn(hDC, hClipRgn);
     // we can delete it emmdiately because SelectClipRgn makes a COPY of the region
     DeleteObject(hClipRgn); 
    
    
/****************************************************

  erasing something drawn on top
  the top is drawn like 
  
   ______              |_____|
  |      |  instead of |     |
  
  so we want to NOT draw the two vertical lines
  above the horzontal

*****************************************************/    
  // if (nYpos = 0) wFlags |= DI_TOPERASED;
    if (wFlags & DI_TOPERASED) 
    {
        nTop = nYpos;
        nHeight = nHtItem / 4;
    }     
/****************************************************

  erasing something originally drawn on the bottom
  
  if the list box is NOT LBS_NOINTEGRALHEIGHT, then
  the botton line will be on the border of the list
  box, so we don't want to draw the horizontal line at  
  all, ie we draw
  
  |    |           |_____|
        instead of |     |  
   

*****************************************************/     
    else if (wFlags & DI_BOTTOMERASED && !bNoIntegralHeight)
    {
    
        nTop = nYpos - nHtItem / 4;                              
        nHeight = nHtItem / 4;    
    
    } 
    
    else
    {
        nTop = nYpos - nHtItem / 4;                          
        nHeight =  nHtItem / 2;    
    
    }

    
   if (!(wFlags & DI_BOTTOMERASED && !bNoIntegralHeight)) // see above comment  
   
   {
        
        PatBlt(hDC,
               LINE_WIDTH,
               nYpos,
               nWidth - 2 * LINE_WIDTH,
               LINE_WIDTH,
               PATINVERT);   
    }           
    PatBlt(hDC,
           0,
           nTop,
           LINE_WIDTH,
           nHeight , 
           PATINVERT);                  
            
    PatBlt(hDC,
           nWidth - LINE_WIDTH,
           nTop, 
           LINE_WIDTH,
           nHeight,
           PATINVERT);
    
    
    

}