{***************************************}
{                                       }
{     TVGraphic Library Demo1           }
{                                       }
{            5/22/93                    }
{                                       }
{        COPYRIGHT (C) 1993             }
{        RICHARD P. ANDRESEN            }
{                                       }
{***************************************}

{This demo program provides source code that illustrates the
 use of the TVGraphic library. It is part of the documentation
 of TVGraphic.
   Both very simple and complex commented Draw methods are here.
   Study TDemoApp.GetEvent to see how to modify/respond to
     events no matter what view is modal.
   The process of initializing and shutting down a TVGraphic
     application is completely shown.
   DOS shell and critical error handling in graphic mode are illustrated.
   Note: The code used here to reload the DeskTop from disk is not
     safe in Borland's Turbo Vision. View by View LowMemory checking
     in TVGraphic's TGroup.Load makes it safe here.
   Examples of setting the mouse cursor grid.
   The Help window is quick and dirty - not a real help.
   Note that all Dialog boxes here have their Frame draw a background
     for the entire box - this is TVGraphic's default for Dialogs.
   Examples of running Dialogs using both DeskTop^.ExecView() from TV1.0
     and ExecuteDialog() as used in TV2.0 are shown.
   Example of how to set up and use TVGraphic's TPanWindow with TSubWindows.

Users of TVGraphic may incorporate sections of this source code
into their own programs.
}


program TVGDemo1;

{$F+,X+}
uses CRT, DOS, Memory, MyGraph3, GObjects, GDrivers, GMENU6,
     MCursor2, GColors, GWindow,
     GViews1, GAppx2, GDialogs, GStdDlg ,GMsgBox;

const
  ProgName = 'TVGDemo1';
  Ver      = '1.4';

const
  dpTV1Dialog = 3;
  WinNum : integer = 0;
  hcMouseGrid   = 1000;
  cmSetColors        = 1100;
  cmDosCriticalError = 1101;
  cmCircleWindow     = 1102;
  cmScrollerWindow   = 1103;
  cmShowMessageBar   = 1104;
  cmAbout            = 1105;
  cmOptionsSave      = 1106;
  cmOptionsLoad      = 1107;

  cmDeskTopStyle     = 1114;
  cmDeskTopOptions   = 1115;
  cmVersion          = 1116;
  cmMouseGrids       = 1117;
  AString = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';

var
  OldExitProc : Pointer;  { Saves exit procedure address }
  Graphic  : boolean;     { true if screen is in graphic mode }


procedure GExitProc; far;     {must be FAR}
begin
  ExitProc := OldExitProc; { Restore exit procedure address }
  CloseGraph;              { Shut down the graphics system }
end;

  {GSystemError handles DOS Critical Errors while in grpahics mode.
      (Not an example of drawing Views in TVGraphics - see
          .Draw methods instead for that.)
   Note the saving and restoring of the Viewport (vital). Also of
   TextSettings which may not be necessary in every program.}
   {NOTE - BOMBS unless you use FarSelectKey to get user input.}
function GSystemError(ErrorCode: Integer; Drive: Byte): Integer; far;
                                                        {must be Far}
const
  SRetryOrCancel:  string[30] = '~Enter~: Retry  ~Esc~: Cancel';
var
  P: Pointer;
  S: string[63];
  X,YOff : integer;
  SS : string;
  VPort : ViewPortType;
  SaveText : TextSettingsType;
begin
  P := Pointer(Drive + Ord('A'));
  FormatStr(S, GetCritErrorStr(ErrorCode), P);
  SS := S + '      ' + SRetryOrCancel;
  X := (GetMaxX - (Length(SS))*Charlen) div 2;

  GetViewSettings(VPort);                  {save current viewport}
  SetViewPort(0,0,GetMaxX,GetMaxY,ClipOn); {set to full screen}

  GetTextSettings(SaveText);                 {save current font, style}
  SetTextStyle(font8x8,HorizDir,1);
  YOff := VertTextOffset(MenuBar^.Size.y);

  SetColor(lightcyan);
  SetFillStyle(solidfill, red);
      {draw over menu bar so can erase by calling MenuBar^.Draw}
  Bar3d(0, 0, GetMaxX, MenuBar^.Size.y, 0, false);
  Graph.MoveTo(X, YOff);
  WriteCStr(SS, white, yellow);
  SetColor(white);
  OutTextXY(Charlen,YOff, Chr($10));
  OutTextXY(GetMaxX-2*Charlen,YOff, Chr($11));

  GSystemError := FarSelectKey;        {get retry/cancel user input}
  MenuBar^.Draw;                        {erase error message}

  with SaveText do
    SetTextStyle(Font, Direction, CharSize);
  with VPort do
    SetViewPort(X1, Y1, X2, Y2, Clip);
end;


{------ Heap View object ----------}
{displays available heap space, updates using timer tick}

type
  PHeapView = ^THeapView;
  THeapView = object(TView)
    OldMem : LongInt;
    constructor Init(var Bounds: TRect);
    procedure Draw; virtual;
    procedure HandleEvent(var Event : TEvent); virtual;
  end;

constructor THeapView.Init(var Bounds: TRect);
begin
  TView.Init(Bounds);
  OldMem := 0;
  EventMask := evTimerTick;
  VFont := font8x8;
end;

procedure THeapView.Draw;
  {Because the HeapView is outside of the default viewport in this
   program, the viewport is changed and restored in this Draw routine.}
var
  S: string;
  C: word;
  VPort : ViewPortType;
  YOff : integer;
  Glob : TRect;
begin
  MCur.Hide;                                       {hide mouse cursor}
  GetViewSettings(VPort);                          {save current viewport}

  GetScreenCoords(Glob);      {set viewport to outline of this view}
  SetViewPort(Glob.A.x, Glob.A.y, Glob.B.x, Glob.B.y,ClipOn);

  GetVPRelCoords(Glob);       {get view outline in viewport relative coords}

  OldMem := MemAvail;
  Str(OldMem, S);
  C := GetColor(2);           {get normal menu text color pair from palette}

  SetColor(ForeColor(C));                      {set text color}
  SetFillStyle(solidfill,BackColor(C));        {set background color}
  Bar(Glob.A.x, Glob.A.y, Glob.B.x, Glob.B.y); {draw background}

  S := 'HEAP: ' + S;
  SetTextStyle(VFont,HorizDir,1);                  {set text font}
    {must set font Before calling VertTextOffset}
  YOff := VertTextOffset(Size.y); {center text vertically in view}

  OutTextXY(Glob.A.x+BXOffset,Glob.A.y+YOff,S);    {write text}

  with VPort do                                    {restore viewport}
    SetViewPort(X1, Y1, X2, Y2, Clip);
  MCur.Show;                                       {show mouse cursor}
end;

procedure THeapView.HandleEvent(var Event : TEvent);
begin
  if (Event.What = evTimerTick) and (OldMem <> MemAvail) then DrawView;
end;

{-----------------------------------}
const
  SArraySize = 33;
  SArray : array[1..SArraySize] of Str80 = (
    '         TVGraphic is a compiled library',
    'written in Borland''s Turbo Vision modified to run',
    'in DOS graphic mode using the EGA/VGA driver.',
    'It requires Turbo Vision and the Graph unit.',
    '',
    'Currently based on TV 1.0, it includes some fixes',
    'and upgrades from TV 2.0 plus other enhancements',
    'aimed at pure graphics applications.',
    '',
    'A new partial screen redraw mechanism provides',
    'automatic sizing of the viewport and the Clip variable.',
    '',
    'TView methods are included that calculate the',
    'global coordinates needed for graphic drawing calls.',
    '',
    'Two fast, clippable bit mapped fonts are included.',
    'Optional user settable grid for mouse cursor.',
    'Hooks are present for user flexibility.',
    '',
    'A Window (or any TGroup descendent) may have an',
    'interior larger than the screen which contains',
    'other windows(or TGroups/TViews).',
    '',
    'Units are available for Pascal versions 6 and 7.',
    'Protected mode, hooks to graphic printing and C++ ',
    'versions may be available in the future.',
    '',
    'For information, comments, wish items, bugs, etc.',
    '       or software consulting/development',
    '',
    '   Richard P Andresen     CompuServe# 71222,1200',
    '   RR2 Box 900',
    '   Hinesburg,Vermont 05461');


const
  TestStr : string =
    'A GOOD LONG PIECE OF LENGTHY, MONOTONOUS, BORING, REPETITIVE TEXT.';

type
  {A simple scrolling view with text and graphics}

  PMyScroller = ^TMyScroller;
  TMyScroller = object (TScroller)
    constructor Init(var Bounds: TRect; AHScrollBar,
                AVScrollBar: PScrollBar);
    procedure Draw; virtual;
  end;

constructor TMyScroller.Init(var Bounds: TRect; AHScrollBar,
                AVScrollBar: PScrollBar);
begin
  TScroller.Init(Bounds, AHScrollBar, AVScrollBar);
  GrowMode := gfGrowHiX + gfGrowHiY;
  SetLimit((3*Size.x div 2) div Charlen, 2*Size.y div Boxheight);
end;

procedure TMyScroller.Draw;
const
 Triangle: array[1..4] of TPoint = ((X: 200; Y: 100), (X: 300; Y:50),
   (X: 250; Y: 250), (X:  200; Y: 100));
var
  VPort : ViewPortType;
  R,C : TRect;
  Color,I,Err : integer;
  Glob : TRect;
begin
  MCur.Hide;                       {hide mouse cursor}
  GetViewSettings(VPort);          {save current viewport settings}
  Move(VPort,C, Sizeof(C));        {copy viewport outline to C}
  GetScreenCoords(R);              {get outline of this view in screen coords}
  R.Intersect(C);        {find outline of view contained within the viewport}
                         {reset viewport to clip at this outline}
     {Note that the viewport's size is set automatically by TVgraphic
      whenever it redraws only a portion of the screen (a common occurance).
      Thus the viewport may be larger than, smaller than
      or cover only a part of this view when this Draw is called.
      However, because Scrollers have a drawable interior larger than
      their Size, we must prevent drawing outside the View by resetting
      the viewport for the duration of this Draw method.
      Re-size the viewport to the rectangle of this View that falls
      within the viewport.
      Because there may be other views to redraw, ALWAYS restore a re-sized
      viewport to the values saved in VPort at the end of a .Draw!}

      {Debugging note: When calling a view's Draw via it's DrawView method,
       the view's Exposed function will prevent Draw from being called if
       no part of the view lies in the Clip variable. At the start of a
       partial redraw, the viewport is set to match (cover) the Clip area.}

  SetViewPort(R.A.x,R.A.y,R.B.x,R.B.y,ClipOn);

  GetVPRelCoords(Glob);   {get view's outline in Viewport Relative coords}
                          {must call after setting viewport!}

  Color := GetColor(1);       {call palette for normal text color}
                              {note that GetColor returns both foreground
                              and background colors in single integer}
              {color for text-use ForeColor for foreground color}
              {color for background - use BackColor}

  SetFillStyle(solidfill,BackColor(Color));
  Bar(Glob.A.x,Glob.A.y,Glob.B.x,Glob.B.y);   {draw background}

  SetColor(ForeColor(Color));
  SetTextStyle(font8x8,HorizDir,1);

   {draw scrolling text using scroller offset "Delta"}
   {Note we are assuming text is Charlen wide by Boxheight tall. These are
    the scroll step sizes this scroller was set to with SetLimit in .Init}

    {kludge for demo program - text varies with window title}
if PWindow(Owner)^.Title^[1] <> 'A' then begin
  SetTextStyle(font8x14,HorizDir,1);
  OutTextXY(Glob.A.x+Charlen-(Delta.x*Charlen),
            Glob.A.y+(10-Delta.y)*Boxheight, TestStr);

   {Draw some text that doesn't move as view scrolls}
   {Will need to have path to BGI font directory to use Gothicfont
    or else TVGraphic will substitute its Font8x8 font.}
{  SetTextStyle(Gothicfont,HorizDir,1);}
  SetTextStyle(font8x8,HorizDir,1);
  OutTextXY(Glob.A.x,Glob.A.y+200,'This line doesn''t scroll.');


  {Now for something Graphic
   Note that since we have set TScroller up as a text scroller, we
   have to multiply Delta.x by Charlen and Delta.y by Boxheight to
   get graphic coords. By changing the constants used with SetLimits
   in Init and using these same constants in this Draw method,
   you could get any scroll step size you wanted.}

  SetColor(green);
  for I := 1 to 3 do
    Line(Triangle[I].x-(Delta.x*Charlen),Triangle[I].y-(Delta.y*Boxheight),
      Triangle[I+1].x-(Delta.x*Charlen),Triangle[I+1].y-(Delta.y*Boxheight));
end
else         {normal scroller code}
    {for fastest speed, call OutTextXY only when it is in current viewport!}
    {remember that GetVPRelCoords is viewport relative.}
  for I := 1 to SArraySize do
    if ((Glob.A.y + (I+1-Delta.y)*Boxheight) > 0)
     and ((Glob.A.y + (I-2-Delta.y)*Boxheight) < Glob.B.y) then
       OutTextXY(Glob.A.x+Charlen-(Delta.x*Charlen),
           Glob.A.y+(I-Delta.y)*Boxheight +BYOffset, SArray[I]);


  with VPort do       {restore viewport}
    SetViewPort(X1,Y1,X2,Y2,Clip);

  MCur.Show;          {show mouse cursor}
end;

type
  {demonstrates very simple Draw method and using TimerTick events}

  PCircles = ^TCircles;
  TCircles = object(TWinBackground)
    Count : integer;
    Speed : integer;
    constructor Init(var Bounds: TRect);
    procedure Draw; virtual;
    procedure DrawCircle;
    procedure HandleEvent(var Event : TEvent); virtual;
  end;

  constructor TCircles.Init(var Bounds: TRect);
  begin
    TWinBackground.Init(Bounds);
    GrowMode := gfGrowHiX + gfGrowHiY;
    EventMask := evTimerTick;
    VColor := black;               {store drawing color}
  end;

  procedure TCircles.Draw;
  var   Glob : TRect;
  begin
    MCur.Hide;              {hide cursor}
    GetVPRelCoords(Glob);   {get view's outline in viewport relative coords}

    SetFillStyle(solidfill,VColor);            {set background color}
    Bar(Glob.A.x,Glob.A.y,Glob.B.x,Glob.B.y);  {draw background}

    DrawCircle;

    MCur.Show;
  end;

  procedure TCircles.DrawCircle;
  var
    Radius : word;
    Glob : TRect;
    Color : integer;
  begin
    MCur.Hide;              {hide cursor}
    GetVPRelCoords(Glob);   {get view's outline in viewport relative coords}

    if (Count = 0) or (Count =8) then Color := 14
      else Color := Count;
    SetColor(Color);               {set circle Color based on Count}

                                   {compute radius based on view's size}
    if Size.x < Size.y then Radius := Size.x
    else Radius := Size.y;
    Radius := Radius div 3;
                                   {draw circle}
    Circle(Glob.A.x+Size.x div 2, Glob.A.y+Size.y div 2, Radius);
    MCur.Show;               {show the mouse cursor}
  end;

  procedure TCircles.HandleEvent(var Event : TEvent);
    {Note that this code will continue to draw the circle even if
     you open a menu or dialog on top of it (Menus and most
     dialogs are modal views). This is part of the demonstration
     of TimerTick events. To prevent this,
     extend the object by adding a field P to store pointer to the
     current modal view. Set this field in Init (P := TopView).
     Test here to see if modal view (TopView) is different from
     field's value. If it is, don't draw.}
  begin
    if Event.What = evTimerTick then begin
      Inc(Speed);
      if Speed > 1023 then Speed := 0;
      if (Speed mod 8 = 0) then begin
        Inc(Count);
        if Count > 15 then Count := 0;   {limit to highest color}
        if GetState(sfActive) then DrawCircle;
      end;
    end;
  end;

{RegisterTypes}
const
  RMyScroller: TStreamRec = (
     ObjType: 3000;
     VmtLink: Ofs(TypeOf(TMyScroller)^);
     Load:    @TMyScroller.Load;
     Store:   @TMyScroller.Store
  );
  RCircles: TStreamRec = (
     ObjType: 3001;
     VmtLink: Ofs(TypeOf(TCircles)^);
     Load:    @TCircles.Load;
     Store:   @TCircles.Store
  );

  procedure RegisterLocals;
  begin
    RegisterType(RMyScroller);
    RegisterType(RCircles);
  end;
{--------------------------------}
type
  TDemoApp = object(TProgram)
    DeskTopStyle : word;            {style currently in use}
    ThePanWindow : PPanWindow;      {pointer to panning window if it exists}
    constructor Init;
    procedure GetEvent(var Event : TEvent); virtual;
    procedure DoAboutBox;
    procedure DosShell;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure InsertCircleWin;
    procedure InsertScrollerWin(ATitle : string);
    procedure InitHeapViewer;
    procedure InitMenuBar; virtual;
    procedure InitShiftView;
    procedure InitMessageBar;       {message that covers over the MenuBar}
    procedure InitStatusLine; virtual;
    procedure IntroScreen;
    procedure NewWindow;
    procedure SaveDeskTop;
    procedure SelectDeskTopStyle;
    procedure LoadDeskTop;
    procedure ShowHelp;
    procedure ShowMouseBox;
    destructor Done; virtual;
    destructor HaltDone;
  end;

destructor TDemoApp.Done;
  {called for normal program termination}
begin
  TProgram.Done;
  MCur.Done;            {releases mouse cursor memory}
  CloseGraph;
  Graphic := false;

  {DoneHistory;}
  DoneSysError;
  DoneEvents;
  DoneVideo;
  DoneMemory;
end;

destructor TDemoApp.HaltDone;
  {used if program halts while trying to initilize graphic mode}
begin
  {DoneHistory;}
  DoneSysError;
  DoneEvents;
  DoneVideo;
  DoneMemory;
end;

constructor TDemoApp.Init;
  procedure DoStreamRegistration;
    {register objects and views for stream I/O}
    {vary contents to match your program}
  begin
    RegisterObjects;
    RegisterViews;
    RegisterDialogs;
    RegisterMenus;
    RegisterApp;
    RegisterStdDlg;
    RegisterWindows;
    RegisterLocals;
  end;
var
  GraphDriver,GraphMode,ErrorCode : integer;
begin
  Graphic := false;

  InitMemory;
  InitVideo;
  InitEvents;
  InitSysError;
  {InitHistory;}

     {register driver from TVGraph}
  if RegisterBGIdriver(@EGAVGADriverProc) < 0 then begin
    HaltDone;
    Writeln('Internal EGA/VGA driver not linked.');
    Halt(1);
  end;
     {enter graphics mode}
  DetectGraph(GraphDriver, GraphMode);
  if not ((GraphDriver = VGA) or (GraphDriver = EGA)) then begin
    HaltDone;
    Writeln('Error - system does not support EGA or VGA graphics.');
    Halt(1);
  end;
  SetVideoMode(smCO80);                {in case in mono mode}
  if GraphDriver = VGA then GraphMode := VGAHi
    else GraphMode := EGAHi;
  InitGraph(GraphDriver,GraphMode,'');
  ErrorCode := GraphResult;
  if ErrorCode <> grOK then begin
    HaltDone;
    Writeln('Graphics Error: ',GraphErrorMsg(ErrorCode));
    Halt(1);
  end
  else begin   {install exit proc to Close graphics}
    OldExitProc := ExitProc;                { save previous exit proc }
    ExitProc := @GExitProc;                { insert our exit proc in chain }
    Graphic := true;
  end;

  MCur.Init;         {mouse cursor object}
  InitShiftView;     {must do before TProgram.Init if calling
                      ShiftView.HandleEvent from TDemoApp.HandleEvent.
                      ShiftView is needed for Panning windows only.}
  TProgram.Init;

  SysErrorFunc := GSystemError;

      {following items may be different for your program}
  DoStreamRegistration;

  InitMessageBar;

  DoubleDelay := 6;     {time between mouse button presses for double press}
                        {TV uses 8 - very slow}

  {set default viewport to just cover the DeskTop. The MainMenu,MessageBar
  and StatusLine temporarily reset viewport when they draw themselves.}
  with DeskTop^ do
    SetViewPort(Origin.x, Origin.y,
           Origin.x + Size.x, Origin.y + Size.y, ClipOn);

  {set mouse grids to off}
  MGridSize.X := 1;
  MGridSize.Y := 1;
  MouseSnapToMenuGrid := false;
  MouseSnapToDialogGrid := false;

  InitHeapViewer;   {optional}

  IntroScreen;      {optional}
end;

procedure TDemoApp.DosShell;
  {Must override method TApplication.DosShell for graphics.}
begin
   {USE TurboVision 2.0 MEMORY Unit if compiling with TP 7.0,
      use MEMORY ver 1.0 with TP.6.0}
  RestoreCrtMode;          {back to text mode}

  DoneSysError;
  DoneEvents;
  DoneVideo;
  {$IFDEF VER60}
    SetMemTop(HeapPtr);      {reduce reserved memory size}
  {$ELSE}
    DoneDosMem;
  {$ENDIF}
  Writeln('Type EXIT to return to '+ ProgName + '...');
  SwapVectors;
  Exec(GetEnv('COMSPEC'), '');
  SwapVectors;
  {$IFDEF VER60}
    SetMemTop(HeapEnd);      {reserve all of memory}
  {$ELSE}
    InitDosMem;
  {$ENDIF}

  InitVideo;
  InitEvents;
  InitSysError;

  SetGraphMode(GetGraphMode);
  Redraw;                          {Use Redraw here, not Draw.}
  if DosError <> 0 then DOSErrorMessageBox(DosError, 'Running DOS shell');
end;


procedure TDemoApp.GetEvent(var Event : TEvent);
  {override the mouse coordinates returned with button events
  when in graphics mode!!}
const
  HelpInUse : boolean = false;
  LastPressDouble : boolean = false;
begin
  TProgram.GetEvent(Event);    {usual call}

   {OPTIONAL : Insert timer tick events.
       These events will occur about every 18 msec IF no other events
       are being processed. If busy with processing, the event will
       be skipped. The tick count for the event is in Event.InfoLong.
       If you need to, you can get the current timer tick count
       by calling function GetBiosTicks. This returns a longint.
       DOS increments the tick count every 18.2 msec and resets it
       to 0 at midnight.}
       {to detect Timer tick events,
        1. Must set view's EventMask:
                 EventMask := EventMask or evTimerTick;
        2. in view's HandleEvent, check if Event.What = evTimerTick}
  if (Event.What = evNothing) then begin
    GetBiosTickEvent(Event);
       {if some other view is modal, broadcast event using App.HandleEvent}
    if not (Event.What = evNothing) and not (TopView = @Self)
      then HandleEvent(Event);
  end;

     {update mouse coords}
  if (Event.What and evMouse <> 0) and (Graphic = true) then begin
    MCur.Move(Event.Where);    {move cursor to mouse location}
    if (Event.What = evMouseDown) then begin
          {OPTIONAL - remap middle button of 3 button mouse}
      if (Event.Buttons > mbRightButton) then Event.Buttons := mbLeftButton;
         {eliminate sequential double press events}
      if (Event.Double) then
        if not LastPressDouble then LastPressDouble := true  {remember this double press}
        else begin
          Event.Double := false;         {reset the double flag}
          LastPressDouble := false;
        end
      else LastPressDouble := false;     {clear flag if non-double press}
    end;
  end;

  if (Event.What = evCommand) and (Event.Command = cmHelp)
   and not HelpInUse then begin
    HelpInUse := true;
    ShowHelp;             {in GetEvent to cover Modal situations!}
    ClearEvent(Event);
    HelpInUse := false;
  end;
end;

procedure TDemoApp.ShowHelp;
  var
    HWin : PGDialog;
    S : string;
    Control : integer;
    HCtx : word;
    PS : PGStaticText;
    B : PGButton;
    R : TRect;
  begin
    HCtx := GetHelpCtx;
    Str(HCtx,S);
    if HCtx = hcMouseGrid then
      S := 'TVGraphic allows the mouse cursor to be snapped to any user specified grid for the screen in general.'+
         ' Grid Off (uses every pixel) and two other choices are provided here.'
    else
      S := ^C'THIS IS NO HELP AT ALL'^M^M^M+
         ^C+ 'Help Context = ' + S;
    R.A.x := 0;  R.B.x := R.A.x + 49*Charlen;
    R.A.Y := 0;  R.B.y := R.A.y + (11+3)*Boxheight;

    HWin := New(PGDialog,Init(R,'HELP',DefaultOpts));
    HWin^.Options := Options or OfCentered;  {autocenter}

    Inc(R.A.x, 4*Charlen);
    Dec(R.B.x, 4*Charlen);
    Inc(R.A.y, 4*Boxheight);
    R.B.y := R.A.y + 4*Boxheight;
    PS := New(PGStaticText, Init(R,S,DefaultOpts));
    HWin^.Insert(PS);

    R.A.x := HWin^.Size.x - 11*Charlen;
    R.A.y := HWin^.Size.y - 2*Boxheight;
    B := New(PGCancelButton, Init(R.A));
    HWin^.Insert(B);

    Control := DeskTop^.ExecView(HWin);
    Dispose(HWin,Done);
  end;


procedure TDemoApp.DoAboutBox;
begin
    InsertScrollerWin('ABOUT TVGRAPHIC');
end;

procedure TDemoApp.HandleEvent(var Event: TEvent);

  procedure Colors;
  var
    D: PColorDialog;
  begin
    D := New(PColorDialog, Init('',
      ColorGroup('Desktop',       DesktopColorItems(nil),
      ColorGroup('Menus',         MenuColorItems(nil),
      ColorGroup('Dialogs',  DialogColorItems(dpTV1Dialog, nil),
      ColorGroup('Windows',  WindowColorItems(wpBlueWindow, nil),
      ColorGroup('Help',     WindowColorItems(wpCyanWindow, nil),
        nil)))))));

    if ExecuteDialog(D, Application^.GetPalette) <> cmCancel then
    begin
      ReDraw;        { Redraw application with new palette }
    end;
  end;

  procedure DosErr;
  var
    F: Text;
    Cmd : integer;
  begin
    Cmd := MessageBox(^C'Testing DOS Critical Error'+
          ^M^M^C'Remove any disk in drive A:',nil,mfWarning+mfOKCancel);
    if Cmd <> cmOK then exit;

    Assign(F, 'a:\8anyfile.7Z3');
  {$I-}
    Reset(F);
    Close(F);
    Cmd := IOResult;   {added 5/17/93}
  {$I+}
  end;

  procedure DeskTopOptionsInfo;
  var Cmd : integer;
  begin
    Cmd := MessageBox(
^C'The heights and fonts of the Menubar and the StatusLine are adjustable in the code.',
      nil, mfInformation+mfOKButton);
  end;

  procedure ShowVersion;
  var Cmd : integer;
  begin
    Cmd := MessageBox(^C'TVGraphic Demo1 ver '+Ver,
      nil, mfInformation+mfOKButton);
  end;

  procedure MakeSound;
  begin
    Sound(1000);
    Delay(10);
    NoSound;
  end;
var
  R: TRect;
  PDir,FInputBox : PView;
  Cmd : integer;
begin
  if (ShiftViewPtr <> Nil) then ShiftViewPtr^.HandleEvent(Event);
      {ShiftViewPtr will be nil unless InitShiftView has been
      called. A ShiftView is needed if using TPanWindow type. Call to
      ShiftViewPtr^.HandleEvent must come before call to
      TProgram.HandleEvent.}

  TProgram.HandleEvent(Event);    {usual call to ancestor method}

  if Event.What = evCommand then
  begin
    case Event.Command of
      cmNew: NewWindow;
      cmShowMessageBar:
        begin
          MessageBar^.ShowText('~T~HIS IS THE MESSAGE BAR.');
          Delay(1000);
          MessageBar^.Hide;
        end;
      cmAbout: DoAboutBox;
      cmOpen:
        begin
          FInputBox := New(PFileDialog, Init('*.*', 'OPEN A FILE', '~N~ame', fdOpenButton,0));
          Cmd := DeskTop^.ExecView(FInputBox);
          Dispose(FInputBox, Done);
        end;
      cmChangeDir:
        begin
          PDir := New(PChDirDialog, Init(cdNormal {+ cdHelpButton},0));
          Cmd := DeskTop^.ExecView(PDir);
          Dispose(PDir, Done);
        end;
      cmSetColors: Colors;
      cmDOSshell : DOSShell;
      cmDosCriticalError : DosErr;
      cmOptionsSave : SaveDeskTop;
      cmOptionsLoad : LoadDeskTop;
      cmCircleWindow : InsertCircleWin;
      cmScrollerWindow : InsertScrollerWin('WINDOW WITH SCROLLER');
      cmMouseGrids   : ShowMouseBox;
      cmDeskTopStyle : SelectDeskTopStyle;
      cmDeskTopOptions : DeskTopOptionsInfo;
      cmVersion       : ShowVersion;
    end;
  end;
end;

procedure TDemoApp.InsertCircleWin;
var
  P : PView;
  W : PWindow;
  R : TRect;
begin
  R.Assign((WinNum+20)*Grid, (WinNum+20)*Grid,
           (WinNum+40)*Grid, (WinNum+40)*Grid);

   {use a TSubWindow here rather than TWindow since window may be
   inserted into another window instead of the DeskTop}
  W := New(PSubWindow, Init(R,'CIRCLES',wnNoNumber));
  W^.GetInteriorSize(R);
  P := New(PCircles, Init(R));
  W^.Insert(P);
  if ThePanWindow <> nil then ThePanWindow^.Insert(W)
   else DeskTop^.Insert(W);
end;

procedure TDemoApp.InsertScrollerWin(Atitle : string);
var
  WinTitle : string;
  TheWindow : PSubWindow;
  PScrollH,PScrollV : PScrollBar;
  PS : PView;
  R : TRect;
begin
  Inc(WinNum);
  R.Assign((WinNum+4)*Charlen, (WinNum+4)*Boxheight,
           (WinNum+64)*Charlen,(WinNum+24)*Boxheight);
  WinTitle := ATitle;
  TheWindow := New(PSubWindow, Init(R, WinTitle, WinNum{wnNoNumber}));

  PScrollH := TheWindow^.StandardScrollBar(sbHorizontal + sbHandleKeyboard);
  PScrollV := TheWindow^.StandardScrollBar(sbVertical + sbHandleKeyboard);

  TheWindow^.GetInteriorSize(R);
     {GetInteriorSize returns the rectangle that needs to be filled
     with views - here fill it with the scroller}
  PS := New(PMyScroller, Init(R,PScrollH,PScrollV));
  TheWindow^.Insert(PS);

  if ThePanWindow <> nil then ThePanWindow^.Insert(TheWindow)
  else DeskTop^.Insert(TheWindow);
end;

procedure TDemoApp.InitHeapViewer;
var
  P : PView;
  R : TRect;
begin
  R.Assign(Size.x - 14*Charlen, StatusLine^.Origin.y, Size.x, Size.y);
  P := New(PHeapView, Init(R));
  Insert(P);
end;

procedure TDemoApp.InitMessageBar;  {message that cover over the MenuBar}
begin
  MessageBar := New(PGMessageBar,Init);
  Insert(MessageBar);
end;

procedure TDemoApp.InitShiftView;
 {Used with full desktop panning window(s).
  Zero or One ShiftView per application.
  The shape of ShiftView is the top row of pixels on the screen.}
var
  R : TRect;
begin
  R.A.x := 0;  R.B.x := GetMaxX;
  R.A.y := 0;  R.B.y := 1;           {shape = slit above menubar}
  ShiftViewPtr := New(PShiftView,Init(R));
end;

procedure TDemoApp.InitMenuBar;
var
  R: TRect;
begin
  MenuBarHeight := {15}20;   {user choice}
  GetExtent(R);
  MenuBar := New(PGMenuBar, Init(R, NewMenu(
    NewSubMenu('~F~ile', hcNoContext, NewMenu(
      NewItem('~N~ew', '', kbNoKey, cmNew, hcNew,
      NewItem('~O~pen...', 'F3', kbF3, cmOpen, hcOpen,
      NewLine(
      NewItem('~C~hange dir...', '', kbNoKey, cmChangeDir, hcChangeDir,
      NewItem('~D~OS shell', '', kbNoKey, cmDosShell, hcDosShell,
      NewItem('E~x~it', 'Alt+X', kbAltX, cmQuit, hcExit,
      nil))))))),
    NewSubMenu('~D~eskTop', hcNoContext, NewMenu(
      NewItem('~S~et Style...', '', kbNoKey, cmDeskTopStyle, hcNoContext,
      NewItem('~O~ptions...', '', kbNoKey, cmDeskTopOptions, hcNoContext,
      NewLine(
      NewItem('S~a~ve desktop', '', kbNoKey, cmOptionsSave, hcNoContext,
      NewItem('~L~oad desktop', '', kbNoKey, cmOptionsLoad, hcNoContext,
      nil)))))),
    NewSubMenu('~W~indows', hcNoContext, NewMenu(
      NewItem('~C~ircleWindow', '', kbNoKey, cmCircleWindow , hcNoContext,
      NewItem('~S~crollerWindow1', '', kbNoKey, cmScrollerWindow , hcNoContext,
      NewItem('~S~crollerWindow2', '', kbNoKey, cmAbout, hcNoContext,
      NewLine(
      NewItem('~S~ize/Move','Ctrl+F5', kbCtrlF5, cmResize, hcResize,
      NewItem('~Z~oom', 'F5', kbF5, cmZoom, hcZoom,
      NewItem('~N~ext', 'F6', kbF6, cmNext, hcNext,
      NewItem('~P~revious', 'Shift+F6', kbShiftF6, cmPrev, hcPrev,
      NewItem('~C~lose', 'Alt+F3', kbAltF3, cmClose, hcClose,
      nil)))))))))),
    NewSubMenu('~M~ouse', hcNoContext, NewMenu(
      NewItem('~S~et Cursor grids...', '', kbNoKey, cmMouseGrids, hcNoContext,
      nil)),
    NewSubMenu('O~p~tions', hcNoContext, NewMenu(
      NewItem('~S~how MessageBar', '', kbNoKey, cmShowMessageBar, hcNoContext,
      NewItem('~D~os Crit Error', '', kbNoKey, cmDosCriticalError, hcNoContext,
      NewItem('Set ~C~olors...', '', kbNoKey, cmSetColors, hcNoContext,
      nil)))),
    NewSubMenu('~I~nfo', hcNoContext, NewMenu(
      NewItem('~A~bout...', '', kbNoKey, cmAbout, hcNoContext,
      NewItem('~V~ersion #', '', kbNoKey, cmVersion, hcNoContext,
      nil))),
    nil))))))
  )));
end;

procedure TDemoApp.InitStatusLine;
  function  HiddenStatusKeys(Next : PStatusItem) : PStatusItem;
  begin
    HiddenStatusKeys :=
        NewStatusKey('', kbF10, cmMenu,
      {  NewStatusKey('', kbAltF3, cmClose, will not work - see documentation}
        NewStatusKey('', kbF5, cmZoom,
        NewStatusKey('', kbCtrlF5, cmResize,
        NewStatusKey('', kbF6, cmNext,
        Next))));
  end;
var
  R: TRect;
begin
  GetExtent(R);
  R.B.x := R.B.x - 14*Charlen;           {leave space for heap viewer}
  R.A.Y := R.B.Y - 9 {Boxheight};    {this gives a 10 pixel tall StatusLine}
StatusLine :=   New(PGStatusLine, Init(R,
    NewStatusDef(0, $FFFF,
      NewStatusKey('~F1~ Help', kbF1, cmHelp,
      NewStatusKey('~F6~ Next', kbF6, cmNext,
      NewStatusKey('~Shift+F6~ Prev', kbShiftF6, cmPrev,
      NewStatusKey('~Alt-X~ Exit', kbAltX, cmQuit,
      HiddenStatusKeys(nil))))),
       nil)));

StatusLine^.VFont := font8x8;
   {use for 10 pixel tall StatusLine - default font is Font8x14}
end;

procedure TDemoApp.IntroScreen;
var
  R : TRect;
  I : integer;
  Msg : string;
begin
  DeskTop^.GetExtent(R);
  SetTextStyle(defaultfont,HorizDir,2);
  SetColor(white);
  for I := 1 to 22 do begin
    if I > 10 then SetTextStyle(font8x14,HorizDir,1);
    if I > 18 then SetTextStyle(font8x8,HorizDir,1);
    OutTextXY(I*20,I*20{R.A.y}, 'TVGraphic');
  end;
  Delay(500);
  R.B.x := 52{46}*Charlen;
  R.B.y := 14{12}*Boxheight;
  Msg := ^C'WELCOME TO TVGraphic Demo1'^M^M^C+
          'TVGraphic is written in Borland Turbo Vision.'+
          ^M^M^C'Make your TV application look like this one with TVGraphic(tm).'
          +^M^M^C+
          'CopyRight 1993 Richard P Andresen';

  MessageBoxRect(R, Msg, nil, mfInformation+mfOKButton);
  DeskTop^.Draw;
end;

procedure TDemoApp.SaveDeskTop;
const
  FName = 'TVGDEMO.DSK';
var
  SaveFile : TBufStream;
  FStatus,Cmd : integer;
  Pal : PString;
begin
    SaveFile.Init(FName, stCreate, 1048);          {create a save file}
    Pal := PString(GetPalette);   {get pointer to palette}
    SaveFile.WriteStr(Pal);          {save palette}
    SaveFile.Put(DeskTop);           {save DeskTop}
    SaveFile.Flush;
    FStatus := SaveFile.Status;
    SaveFile.Done;          {flushes buffer}
    if FStatus <> stOK then
      if FStatus = stPutError then
        Cmd := MessageBox('Put of unregistered object.',nil, mfError + mfOkButton)
      else if SaveFile.ErrorInfo <> 0 then
           DOSErrorMessageBox(SaveFile.ErrorInfo, FName)
           else
             Cmd := MessageBox('Error saving file.',nil, mfError + mfOkButton);
end;

procedure TDemoApp.LoadDeskTop;
  procedure CloseView(P: PView); far;
  begin
    Message(P, evCommand, cmClose, nil);
  end;
  procedure ReadFile(var S : TBufStream);
  var
    Pal : PString;
  begin
    if Desktop^.Valid(cmClose) then
    begin
      Pal := S.ReadStr;
      if Pal <> nil then
      begin
        GetPalette^ := Pal^;
        DisposeStr(Pal);
      end;
      Delete(DeskTop);
      Dispose(DeskTop,Done);
      DeskTop := PDeskTop(ValidView(PDeskTop(S.Get)));
                   {May overflow memory in TV, safe in TVGraphic}
                   {note pointer type conversion to PDeskTop}

      Insert(DeskTop);
    end;
  end;
const
  FName = 'TVGDEMO.DSK';
var
  SaveFile : TBufStream;
  FStatus,Cmd : integer;
begin
  SaveFile.Init(FName, stOpenRead, 1048);
  if (SaveFile.Status = stOK) then begin      {found file}
    ReadFile(SaveFile);
    FStatus := SaveFile.Status;
    SaveFile.Done;          {flushes buffer}
    if FStatus <> stOK then
      if FStatus = stGetError then
        Cmd := MessageBox('Get of unregistered object.',nil, mfError + mfOkButton)
      else if SaveFile.ErrorInfo <> 0 then
           DOSErrorMessageBox(SaveFile.ErrorInfo, FName)
           else
             Cmd := MessageBox('Error reading file.',nil, mfError + mfOkButton);
  end;
end;

procedure TDemoApp.NewWindow;
var
  Cmd : integer;
begin
  Cmd := MessageBox(^C'Use the Windows Menu to open Windows',nil,
                       mfInformation+mfOKButton);
end;

procedure TDemoApp.ShowMouseBox;
type
  Temptype = record
    RW : word;
    CW : word;
  end;
var
  Win : PGDialog;
  WinTitle : PGStaticText;
  OKButton,CancelButton : PGButton;
  StyleStr  : string;
  Control,SaveStyle : integer;
  R     : TRect;
  Org   : TPoint;
  Lab   : PGLabel;
  Radio : PGRadioButtons;
  Check : PGCheckBoxes;
  Temp : Temptype;
begin
        R.A.x := 0;  R.B.x := R.A.x + 42 * Charlen;
        R.A.y := 0;  R.B.y := R.A.y + 19 * Boxheight;
        Win := New(PGDialog,Init(R,'MOUSE GRIDS', DefaultOPts));
        Win^.Options := Win^.Options or ofCentered;
        {Win^.HelpCtx := hcMouseGrid;}

        {add note}
        StyleStr:= ^C'The mouse Cursor can be continuous or snapped to an invisible grid.'+
        ^M^M^C'BoxMenus and Dialogs use the desktop grid if their own grid is not enabled.';
        R.Assign(Charlen{0},2*Boxheight, Win^.Size.x-Charlen, 7*Boxheight);
        WinTitle := New(PGStaticText,Init(R,StyleStr,DefaultOpts));
        Win^.Insert(WinTitle);

        {create buttons}
        Org.x := 2*Charlen; Org.y := Win^.Size.y - 2*Boxheight;
        OkButton := New(PGOKButton,Init(Org, true));
        Win^.Insert(OKButton);

        Org.x := Win^.Size.x - 9 * Charlen;
        Org.y := Win^.Size.y - 2*Boxheight;
        CancelButton := New(PGCancelButton,Init(Org));
        Win^.Insert(CancelButton);

        {create RadioButtons}
        R.A.x := 4*Charlen;
        R.B.x := Win^.Size.x - 4*Charlen;
        R.A.y := R.B.y + 2*Boxheight;
        R.B.y := R.A.y + 3*Boxheight;
        Radio := New(PGRadioButtons, Init(R,
          NewSItem('Desktop Grid off',
          NewSItem('10x10 grid for desktop',
          NewSItem('8x14 Text grid for desktop',
          nil)))));
        Radio^.HelpCtx := hcMouseGrid;
        Win^.Insert(Radio);

        Dec(R.A.y, Boxheight);
          {note use of txAdjustSize to avoid specifying exact size}
        Lab := New(PGLabel, Init(R,'~D~esktop',Radio,txAdjustSize));
        Win^.Insert(Lab);
        Inc(R.A.y, Boxheight);

        {create CheckBoxes}
        R.A.y := R.B.y + 2*Boxheight;
        R.B.y := R.A.y + 2*Boxheight;
        Check := New(PGCheckBoxes, Init(R,
          NewSItem('8x14 Grid for Box Menus',
          NewSItem('8x14 Grid for Dialog Boxes',
          nil))));
        Win^.Insert(Check);

        Dec(R.A.y, Boxheight);
          {note use of txAdjustSize to avoid specifying exact size}
        Lab := New(PGLabel, Init(R,'Use ~S~pecialty grids',Check,txAdjustSize));
        Win^.Insert(Lab);
        Inc(R.A.y, Boxheight);

        {set Temp variable}
          {Desktop mouse grid: 0 = 1x1, 1=10x10, 2=8x14}
        case MGridSize.X of
          1  : Temp.RW := 0;
          10 : Temp.RW := 1;
          8  : Temp.RW := 2;
        end;

        Temp.CW := 0;
        if MouseSnapToMenuGrid then Temp.CW := Temp.CW or $01;
        if MouseSnapToDialogGrid then Temp.CW := Temp.CW or $02;

        Radio^.Select;
        Win^.SetData(Temp);
        Control := DeskTop^.ExecView(Win);   {MODAL, owner is DeskTop}
        Win^.GetData(Temp);

        if (Control <> cmCancel) then begin
         {Setting the mouse grid with MCur.SetGrid
          The third and fourth parameters are an Xoffset and
          YOffset of the grid from the screen's upper left corner.
          Note that MCur.SetGrid(1,1,0,0) causes the mouse coords to
          be used as they come from the mouse driver.
          Unit MCursor also provides functions to limit the area
          of the screen the mouse cursor can move in.}

          case byte(Temp.RW) of
            0 : MCur.SetGrid(1,1,0,0);
            1 : MCur.SetGrid(10,10,0,0);
            2 : MCur.SetGrid(Charlen,Boxheight,0,0);
          end;

          if (Temp.CW and $01 <> 0) then MouseSnapToMenuGrid := true
            else MouseSnapToMenuGrid := false;
          if (Temp.CW and $02 <> 0) then MouseSnapToDialogGrid := true
            else MouseSnapToDialogGrid := false;
        end;

        Dispose(Win,Done);
end;

procedure TDemoApp.SelectDeskTopStyle;
  {While you probably won't switch Desktop styles in a real application,
   it does show two different ways you can set up a program.
   You could also change the heights and fonts of the MenuBar
   and the StatusLine. Or eliminate the StatusLine if you wish.}

   {Example of changing entire Appplication palette and also
    color pairs within the palette.}

  procedure SetDeskTopStyle;
  var
    R : TRect;
    PanStep,IntSize : TPoint;
    TheWindow: PWindow;
    PScrollH,PSCrollV : PScrollbar;
    PS : PScroller;
    P,PBak : PView;
    WinTitle,TestStr : string;
    Pal : PPalette;
  begin
    Delete(DeskTop);
    Dispose(DeskTop, Done);      {dispose old desktop and everything in it}
    InitDeskTop;
    Insert(DeskTop);             {insert the new one}
    ThePanWindow := nil;          {tested for nil elsewhere in program}
    Dispose(ShiftViewPtr, Done);    {dispose to reset shiftview}
    InitShiftView;

    if DeskTopStyle = 1 then begin       {Panning window}
      Inc(WinNum);
      R.Assign(0, 0, 60*Charlen, 20*Boxheight);
      WinNum := 1;
(*      R.Assign(0, 0, GetMaxX,
                 GetMaxY-MenuBar^.Size.y-StatusLine^.Size.y -2);*)
      DeskTop^.GetExtent(R);
      IntSize.x := GetMaxX+200;
      IntSize.y := GetMaxY+100;
      PanStep.x := ScrnShiftX;
      PanStep.y := ScrnShiftY;
        {The pan window should be a even multiple of the mouse grid size}
      ThePanWindow := New(PPanWindow, Init(R,
        'Larger Than Screen Panning Window', wnNoNumber, IntSize, PanStep));

      with ThePanWindow^ do begin
        Flags := 0;        {prevent from closing}
        VOffset.y := 50;
        R.Assign(Charlen,200,InteriorSize.x-Charlen,200+4*Boxheight);
        P := New(PGStaticText, Init(R,AString+AString+AString,txAuto+font8x14));
        P^.VOptions := P^.VOptions or txDrawBackground;
        Insert(P);
      end;
      DeskTop^.Insert(ThePanWindow);
      Pal := GetPalette;
      Pal^[2] := Chr($30);    {change menu background color to cyan}
    end
    else begin
      Pal := GetPalette;
      Pal^[2] := Chr($70);    {change menu background color to light gray}
    end;
{DeskTop^.ExecView(ThePanWindow);}  {Test}
  end;

type
  Temptype = record
    W : word;
  end;
var
  Win : PGDialog;
  WinTitle : PGStaticText;
  OKButton,CancelButton : PGButton;
  StyleStr  : Str80;
  Control,SaveStyle : integer;
  R     : TRect;
  Org   : TPoint;
  Radio : PGRadioButtons;
  Temp : Temptype;
begin
        R.A.x := 0;  R.B.x := R.A.x + 42 * Charlen;
        R.A.y := 0;  R.B.y := R.A.y + 10 * Boxheight;
        Win := New(PGDialog,Init(R,'SELECT STYLE', DefaultOPts));
        Win^.Options := Win^.Options or ofCentered;
        {SizeWin^.HelpCtx := hcSizeWin;}

                      {add note}
        StyleStr:= ^C'Changing Style clears the DeskTop.';
        R.Assign(0,2*Boxheight, Win^.Size.x, 3*Boxheight);
        WinTitle := New(PGStaticText,Init(R,StyleStr,DefaultOpts));
        Win^.Insert(WinTitle);

        {create buttons}
        Org.x := 2*Charlen; Org.y := Win^.Size.y - 2*Boxheight;
        OkButton := New(PGOKButton,Init(Org, true));
        Win^.Insert(OKButton);

        Org.x := Win^.Size.x - 9 * Charlen;
        Org.y := Win^.Size.y - 2*Boxheight;
        CancelButton := New(PGCancelButton,Init(Org));
        Win^.Insert(CancelButton);

        {create RadioButtons}
        R.A.x := 4*Charlen;
        R.B.x := Win^.Size.x - 4*Charlen;
        Inc(R.A.y, 2*Boxheight);
        R.B.y := R.A.y + 2*Boxheight;
        Radio := New(PGRadioButtons, Init(R,
          NewSItem('Multiple Non-Panning Windows',
          NewSItem('Full Screen Panning Window',
          nil))));
        Win^.Insert(Radio);

          {DeskTopStyle: 0 = non-panning, 1=panning}
        Temp.W := DeskTopStyle;
        SaveStyle := Temp.W;

        Win^.SetData(Temp.W);
        Control := DeskTop^.ExecView(Win);   {MODAL, owner is DeskTop}
        Win^.GetData(Temp.W);

        if (Control <> cmCancel) and (Temp.W <> SaveStyle) then begin
          DeskTopStyle := Temp.W;
          SetDeskTopStyle;
        end;

        Dispose(Win,Done);
end;


var
  DemoApp: TDemoApp;
  R : TRect;
begin
  DemoApp.Init;
  R.Assign(0,0,1,1);
  DemoApp.Run;
  DemoApp.Done;
end.
