

/*  example.c                                                            */
/* --------------------------------------------------------------------- */
/*  example problem.                                                     */
/*     - it is the same program that f-lsode has in the statements       */
/*                                                                       */
/*  the following is a simple example problem, with the coding           */
/*  needed for its solution by lsode.  the problem is from chemical      */
/*  kinetics, and consists of the following three rate equations..       */
/*      dy1/dt = -.04*y1 + 1.e4*y2*y3                                    */
/*      dy2/dt = .04*y1 - 1.e4*y2*y3 - 3.e7*y2**2                        */
/*      dy3/dt = 3.e7*y2**2                                              */
/*  on the interval from t = 0.0 to t = 4.e10, with initial conditions   */
/*  y1 = 1.0, y2 = y3 = 0.  the problem is stiff.                        */
/*                                                                       */
/*  the following coding solves this problem with lsode, using mf = 21   */
/*  and printing results at t = .4, 4., ..., 4.e10.  it uses             */
/*  itol = 2 and atol much smaller for y2 than y1 or y3 because          */
/*  y2 has much smaller values.                                          */
/*  at the end of the run, statistical quantities of interest are        */
/*  printed (see optional outputs in the full description below).        */
/* --------------------------------------------------------------------- */

#include <stdlib.h>
#include "lsode.h"



void fcn(int[], double, double[], double[]);
void jac(int[], double, double[], int, int, double[], int);


int main()
{
    double atole[3+1],rtol[2],*rwork,t,tout,*y ;
    int itol,itask,istate,iopt,lrw,liw,mf,*iwork,neq[2];
    int i;

      itol = 2;
      lrw = 58;
      liw = 23;
      neq[1] = 3;

      iwork = ivector(liw);
      rwork = dvector(lrw);
      y = dvector(neq[1]);

      y[1] = 1.0e0;
      y[2] = 0.0e0;
      y[3] = 0.0e0;
      t = 0.0e0;
      tout = 0.40e0;

      rtol[1] = 1.e-4;
      atole[1] = 1.e-6;
      atole[2] = 1.e-10;
      atole[3] = 1.e-6;

      itask = 1;
      istate = 1;
      iopt = 0;
      mf = 21;

      for(i=1;i<=12;i++) {
            lsode(fcn,neq,y,&t,tout,itol,rtol,atole,itask,&istate,
                      iopt,rwork,lrw,iwork,liw,jac,mf,stdout);
            printf("\n at t =%11.4e  y =%14.6e  %14.6e  %14.6e",
                       t, y[1], y[2], y[3]);
            if (istate < 0) {
                 printf("\n\n error halt.. istate =%3d",istate);
                 break;
            }
            tout *= 10.0e0;
      }

      printf("\n\n no. steps = %4d  no. f-s = %4d  no. j-s =%4d\n\n",
                       iwork[11], iwork[12], iwork[13]);
      free(iwork);
      free(rwork);
      free(y);
      return 1;
}


void fcn(int neq[], double t, double y[], double ydot[])
{
      ydot[1] = -.04e0*y[1] + 1.e4*y[2]*y[3];
      ydot[3] = 3.e7*y[2]*y[2];
      ydot[2] = -ydot[1] - ydot[3];
      return;
}




#define   pd(a,b)  pd[nr*(b-1)+a]
void jac(int neq[],double t,double y[],int ml,int mu,double pd[],int nr)
{
      pd(1,1) = -0.04e0;                      /*  pd[1][1]  */
      pd(1,2) = 1.e4*y[3];                    /*  pd[1][2]  */
      pd(1,3) = 1.e4*y[2];                    /*  pd[1][3]  */
      pd(2,1) = 0.04e0;                       /*  pd[2][1]  */
      pd(2,3) = -pd(1,3);                     /*  pd[2][3]  */
      pd(3,2) = 6.e7*y[2];                    /*  pd[3][2]  */
      pd(2,2) = -pd(1,2) - pd(3,2);           /*  pd[2][2]  */
      return;
}
/* --------------------------------------------------------------------- */
/*  the output(*) of this program is as follows..                        */
/*                                                                       */
/*  at t = 4.0000e-01  y =  9.851726e-01   3.386406e-05   1.479357e-02   */
/*  at t = 4.0000e+00  y =  9.055142e-01   2.240418e-05   9.446344e-02   */
/*  at t = 4.0000e+01  y =  7.158050e-01   9.184616e-06   2.841858e-01   */
/*  at t = 4.0000e+02  y =  4.504846e-01   3.222434e-06   5.495122e-01   */
/*  at t = 4.0000e+03  y =  1.831701e-01   8.940379e-07   8.168290e-01   */
/*  at t = 4.0000e+04  y =  3.897016e-02   1.621193e-07   9.610297e-01   */
/*  at t = 4.0000e+05  y =  4.935213e-03   1.983756e-08   9.950648e-01   */
/*  at t = 4.0000e+06  y =  5.159269e-04   2.064759e-09   9.994841e-01   */
/*  at t = 4.0000e+07  y =  5.306413e-05   2.122677e-10   9.999469e-01   */
/*  at t = 4.0000e+08  y =  5.494530e-06   2.197824e-11   9.999945e-01   */
/*  at t = 4.0000e+09  y =  5.129458e-07   2.051784e-12   9.999995e-01   */
/*  at t = 4.0000e+10  y = -7.170560e-08  -2.868224e-13   1.000000e+00   */
/*                                                                       */
/*  no. steps =  330  no. f-s =  405  no. j-s =  69                      */
/*                                                                       */
/*  *. the output is the same as that of f-lsode (see teste.doc).        */
/*     however, it may vary slightly from compiler to compiler.          */
/* --------------------------------------------------------------------- */

