
// +--------------------------------------------------------------------------+
// |                                                                          |
// |          Secure File Deletion Example  11/4/93 by Mark Gamber            |
// |                                                                          |
// +--------------------------------------------------------------------------+
// |                                                                          |
// |   This utility accepts files from a drag-drop source such as File        |
// |   Manager, zeroes out the contents and deletes the file, removing any    |
// |   trace of it's data from the disk.                                      |
// |                                                                          |
// +--------------------------------------------------------------------------|
// |                                                                          |
// |   This program is PUBLIC DOMAIN and may be freely distributed. In using  |
// |   the program or any part thereof, the user assumes responsibility for   |
// |   it's proper use.                                                       |
// |                                                                          |
// +--------------------------------------------------------------------------+

#include "windows.h"
#include "memory.h"
#include "ntrash11.h"


HANDLE hInst;                                                  //  Some globals
HWND MainWin;
BOOL StayOnTop = TRUE;
char *IniSection = "NTRASH";
char *IniFile = "NTRASH.INI";                      //  Various string constants
char *CLASSNAME = "SECDEL";
char *XPOS = "XPOS";
char *YPOS = "YPOS";
char *SOT = "SOT";
char *PD = "%d";

// +--------------------------------------------------------------------------+
// |                                                                          |
// |                                                                          |
// +--------------------------------------------------------------------------+

int PASCAL WinMain( HANDLE hInstance, HANDLE hPrev, LPSTR lpCmd, int nCmdShow )
{
   WNDCLASS wc;
   HWND hWnd;
   MSG msg;
   UINT Xpos, Ypos;

   if( hPrev )
      return( FALSE );

   wc.style = CS_DBLCLKS;                              //  Accept double-clicks
   wc.lpfnWndProc = MainWndProc;
   wc.cbClsExtra = 0;
   wc.cbWndExtra = 0;
   wc.hInstance = hInstance;
   wc.hIcon = LoadIcon( hInstance, MAKEINTRESOURCE( 102 ) );
   wc.hCursor = LoadCursor( NULL, IDC_ARROW );
   wc.hbrBackground = GetStockObject( LTGRAY_BRUSH );
   wc.lpszMenuName = NULL;
   wc.lpszClassName = CLASSNAME;

   if( ! RegisterClass( &wc ) )              // If class doesn't register, exit
      return( FALSE );

   hInst = hInstance;
                                       //  Get position when it was closed last
   Xpos = GetPrivateProfileInt( IniSection, XPOS, 0, IniFile );
   Ypos = GetPrivateProfileInt( IniSection, YPOS, 0, IniFile );
                                      //  Does program "stay on top" of others?
   StayOnTop = GetPrivateProfileInt( IniSection, SOT, 1, IniFile );

                                  //  Create the window accepting dropped files
   hWnd = CreateWindowEx( WS_EX_ACCEPTFILES | WS_EX_TOPMOST, CLASSNAME,
                          (LPSTR)"Test", 
								  WS_VISIBLE | WS_CLIPSIBLINGS | WS_POPUP,
                          Xpos, Ypos, 48, 48, NULL, NULL, hInstance, NULL );
   if( ! hWnd )
      return( FALSE );                        // If create failed, exit program

   MainWin = hWnd;
   ShowWindow( hWnd, SW_SHOWNORMAL );                         // Display window
   UpdateWindow( hWnd );

   if( StayOnTop )              //  Do Z-order business if set to "stay on top"
      SetWindowPos( hWnd, HWND_TOPMOST, 0, 0, 0, 0, 
						  SWP_NOMOVE | SWP_NOSIZE );
   else
      SetWindowPos( hWnd, HWND_NOTOPMOST, 0, 0, 0, 0, 
						  SWP_NOMOVE | SWP_NOSIZE );

   while( GetMessage( &msg, 0, 0, 0 ) )                      // Do message loop
   {
      TranslateMessage( &msg );
      DispatchMessage( &msg );
   }
   return( msg.wParam );
}


// +--------------------------------------------------------------------------+
// |                                                                          |
// |   MainWndProc(): This is the window that accepts drag-drop files.        |
// |                                                                          |
// +--------------------------------------------------------------------------+

long APIENTRY MainWndProc( HWND hWnd, UINT msg, UINT wParam, LONG lParam )
{
   switch( msg )
   {
      case WM_LBUTTONDOWN:	         // If left button pressed, move window
      {
         PostMessage( hWnd, WM_SYSCOMMAND, 0xF012, 0x00110017 );
         break;
      }

      case WM_RBUTTONDOWN:                        // If right button pressed...
      {                     // Note - no more of this MakeProcInstance nonsense
         DialogBox( hInst, MAKEINTRESOURCE( 10000 ), hWnd, MainDlgProc );
         break;
      }

      case WM_LBUTTONDBLCLK:		                       // Quit on double-click
      {
         RECT Rect;
         char str[ 8 ];

         GetWindowRect( hWnd, &Rect );             //  Get our current position
         wsprintf( str, PD, Rect.left );              //  and save it for later
         WritePrivateProfileString( IniSection, XPOS, str, IniFile );
         wsprintf( str, PD, Rect.top );
         WritePrivateProfileString( IniSection, YPOS, str, IniFile );
         wsprintf( str, PD, StayOnTop );
         WritePrivateProfileString( IniSection, SOT, str, IniFile );

         PostQuitMessage( 0 );
         break;
      }

      case WM_DROPFILES:             // When files are dropped on the window...
      {
         WORD Files, Count;
			DWORD dwSize;
         char Filename[ 256 ];
         HDC hDC;
         HDC mDC;
         HBITMAP hBmp;
			HANDLE hFile;
			HANDLE hMap;
			LPSTR lpFile;
                                                 // Get number of files dropped
         Files = DragQueryFile( (HDROP)wParam, -1, Filename, 128 );
         if( ! Files )                                    //  If no files, exit
         {
            DragFinish( (HDROP)wParam );
            return( TRUE );
         }

         hDC = GetDC( hWnd );
         mDC = CreateCompatibleDC( hDC );          //  Draw can with the lid up
         hBmp = LoadBitmap( hInst, MAKEINTRESOURCE( 100 ) );
         SelectObject( mDC, hBmp );
         BitBlt( hDC, 0, 0, 48, 48, mDC, 0, 0, SRCCOPY );
         DeleteDC( mDC );
         DeleteObject( hBmp );
         ReleaseDC( hWnd, hDC );

         for( Count = 0; Count < Files; Count++ )     // Get every file dropped
         {
            DragQueryFile( (HDROP)wParam, Count, Filename, 128 );
                                              //  Open each file for read/write
				hFile = CreateFile( Filename, GENERIC_READ | GENERIC_WRITE,
										  0, NULL, OPEN_EXISTING, 
										  FILE_ATTRIBUTE_NORMAL, NULL );

				if( hFile == INVALID_HANDLE_VALUE )  //  If unable to open, skip it
				{
					dwSize = GetLastError();       //  These calls are for debugging
					continue;
				}

				dwSize = GetFileSize( hFile, NULL );        //  Get the file's size
				if( ! dwSize )
				{
					CloseHandle( hFile );
					DeleteFile( Filename );		//  If nothing to erase, just kill it
					continue;
				}
                                               //  Create a file mapping object
				hMap = CreateFileMapping( hFile, NULL, PAGE_READWRITE, 
                                      0, 0, NULL );
				if( ! hMap )
				{
					dwSize = GetLastError();        //  If it didn't work, next file
					CloseHandle( hFile );
					continue;
				}
                                        //  Map the file into a memory space...
				lpFile = MapViewOfFile( hMap, FILE_MAP_WRITE, 0, 0, 0 );
				if( lpFile == NULL )
				{
					dwSize = GetLastError();              //  On error, back way out
					CloseHandle( hFile );
					CloseHandle( hMap );
					continue;
				}
				memset( lpFile, 0, dwSize );        //  Set the whole thing to zero
				FlushViewOfFile( lpFile, dwSize );             //  Flush it to disk
				UnmapViewOfFile( lpFile );                     //  Kill mapped area

				CloseHandle( hMap );                // Close all the handles and...
				CloseHandle( hFile );
            DeleteFile( Filename );            //  ...(finally) delete the file
         }
         DragFinish( (HDROP)wParam );                  // Tell SHELL we're done
         InvalidateRect( hWnd, NULL, FALSE );          // Repaint with lid down
         return( TRUE );
      }

      case WM_PAINT:
      {
         PAINTSTRUCT ps;
         HDC hDC;
         HDC mDC;
         HBITMAP hBmp;

         hDC = BeginPaint( hWnd, &ps );         // Blasts out can with lid down
         mDC = CreateCompatibleDC( hDC );
         hBmp = LoadBitmap( hInst, MAKEINTRESOURCE( 101 ) );
         SelectObject( mDC, hBmp );
         BitBlt( hDC, 0, 0, 48, 48, mDC, 0, 0, SRCCOPY );
         DeleteDC( mDC );
         DeleteObject( hBmp );
         EndPaint( hWnd, &ps );
         break;
      }

      case WM_DESTROY:
         PostQuitMessage( 0 );
         break;

      default:
         return( DefWindowProc( hWnd, msg, wParam, lParam ) );
   }
   return( FALSE );
}


// +--------------------------------------------------------------------------+
// |                                                                          |
// |  MainDlgProc(): Get topmost setting from user                            |
// |                                                                          |
// +--------------------------------------------------------------------------+

BOOL APIENTRY MainDlgProc( HWND hDlg, UINT msg, UINT wParam, LONG lParam )
{
   switch( msg )
   {
      case WM_INITDIALOG:                                       // Set checkbox
         CheckDlgButton( hDlg, IDM_STAYONTOP, StayOnTop );
         return( TRUE );

      case WM_COMMAND:
         if( wParam == IDCANCEL )                    // When Exit it pressed...
         {                                        // Save 'Stay On Top' setting
            StayOnTop = IsDlgButtonChecked( hDlg, IDM_STAYONTOP );

                                         // If checked, mark window accordingly
            if( StayOnTop )
               SetWindowPos( GetParent( hDlg ), HWND_TOPMOST, 0, 0, 0, 0,
                             SWP_NOMOVE | SWP_NOSIZE );
            else
               SetWindowPos( GetParent( hDlg ), HWND_NOTOPMOST, 0, 0, 0, 0,
                             SWP_NOMOVE | SWP_NOSIZE );

            EndDialog( hDlg, TRUE );
            return( TRUE );
         }
         break;
   }
   return( FALSE );
}
